// 
// Copyright (C) 2010 Trendseek, for FREE use on your own risk!
// Based on "New Trading Dimensions" by Erich Florek and EasyLanguage Coding 
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// The RSI (Relative Strength Index) is a price-following oscillator that ranges between 0 and 100.
	/// </summary>
	[Description("The RMI (Relative Momentum Index), develloped by Roger Altman. Similar to RSI but based on comparing Momentums instead of prices. Typical Settings are 3/8 or 5/13")]
	public class RMI : Indicator
	{
		#region Variables
			private int								period	= 14;
			private int								shift	= 3;	
		
			private DataSeries						avgUp;
			private DataSeries						avgDown;
		
		#endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add(new Plot(Color.Lime, "RMI"));
			
			Add(new Line(System.Drawing.Color.Green, 30, "Lower"));
			Add(new Line(System.Drawing.Color.Black, 50, "Center"));
			Add(new Line(System.Drawing.Color.Red, 70, "Upper"));
			
			avgUp				= new DataSeries(this);
			avgDown				= new DataSeries(this);
			
			
			PriceTypeSupported	= true;
		}

		/// <summary>
		/// Calculates the indicator value(s) at the current index.
		/// </summary>
		protected override void OnBarUpdate()
		{
			double amountUp 	= 0;
			double amountDown 	= 0;
			double rmi			= 0;
			
			if (CurrentBar ==0)
			{
				avgUp.Set(0);
				avgDown.Set(0);			
				
				return;
			
			}
			
			if (CurrentBar < (Period + Shift)) 
			{
				return;
			}
			else if (CurrentBar == (Period + Shift))
			{			
				double sumUp = 0;
				double sumDown = 0;
				
				for (int barsAgo = 0; barsAgo < Period; barsAgo++)
				{
					amountUp = Input[barsAgo] - Input[barsAgo + Shift];
					if (amountUp >= 0) 
					{
						amountDown 	= 0;
					}
					else
					{
						amountDown 	= -amountUp;
						amountUp	= 0;
					}
					sumUp	=	sumUp + amountUp;
					sumDown	=	sumDown + amountDown;
				}
				avgUp.Set( sumUp / Period);
				avgDown.Set( sumDown / Period);			
			}
			else
			{
				amountUp = Input[0] - Input[Shift];
				if (amountUp >= 0)
				{
					amountDown 	= 0;
				}
				else
				{
					amountDown 	= -amountUp;
					amountUp	= 0;
				}
				avgUp.Set((avgUp[1] * (Period - 1) + amountUp) / Period);
				avgDown.Set((avgDown[1] * (Period - 1) + amountDown) / Period);
			}
			
			if ((avgUp[0] + avgDown[0]) != 0)
			{
				rmi = 100 * avgUp[0] / (avgUp[0] + avgDown[0]);
			}
			else rmi = 0;

			
			Value.Set(rmi);
		}

		#region Properties
		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Default
		{
			get { return Values[0]; }
		}
		
		/// <summary>
		/// </summary>
		[Description("Numbers of bars used for calculations")]
		[Category("Parameters")]
		public int Period
		{
			get { return period; }
			set { period = Math.Max(2, value); }
		}

		/// <summary>
		/// </summary>
		[Description("Number of bars for smoothing")]
		[Category("Parameters")]
		public int Shift
		{
			get { return shift; }
			set { shift = Math.Max(1, value); }
		}
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private RMI[] cacheRMI = null;

        private static RMI checkRMI = new RMI();

        /// <summary>
        /// The RMI (Relative Momentum Index), develloped by Roger Altman. Similar to RSI but based on comparing Momentums instead of prices. Typical Settings are 3/8 or 5/13
        /// </summary>
        /// <returns></returns>
        public RMI RMI(int period, int shift)
        {
            return RMI(Input, period, shift);
        }

        /// <summary>
        /// The RMI (Relative Momentum Index), develloped by Roger Altman. Similar to RSI but based on comparing Momentums instead of prices. Typical Settings are 3/8 or 5/13
        /// </summary>
        /// <returns></returns>
        public RMI RMI(Data.IDataSeries input, int period, int shift)
        {
            checkRMI.Period = period;
            period = checkRMI.Period;
            checkRMI.Shift = shift;
            shift = checkRMI.Shift;

            if (cacheRMI != null)
                for (int idx = 0; idx < cacheRMI.Length; idx++)
                    if (cacheRMI[idx].Period == period && cacheRMI[idx].Shift == shift && cacheRMI[idx].EqualsInput(input))
                        return cacheRMI[idx];

            RMI indicator = new RMI();
            indicator.BarsRequired = BarsRequired;
            indicator.CalculateOnBarClose = CalculateOnBarClose;
            indicator.Input = input;
            indicator.Period = period;
            indicator.Shift = shift;
            indicator.SetUp();

            RMI[] tmp = new RMI[cacheRMI == null ? 1 : cacheRMI.Length + 1];
            if (cacheRMI != null)
                cacheRMI.CopyTo(tmp, 0);
            tmp[tmp.Length - 1] = indicator;
            cacheRMI = tmp;
            Indicators.Add(indicator);

            return indicator;
        }

    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// The RMI (Relative Momentum Index), develloped by Roger Altman. Similar to RSI but based on comparing Momentums instead of prices. Typical Settings are 3/8 or 5/13
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.RMI RMI(int period, int shift)
        {
            return _indicator.RMI(Input, period, shift);
        }

        /// <summary>
        /// The RMI (Relative Momentum Index), develloped by Roger Altman. Similar to RSI but based on comparing Momentums instead of prices. Typical Settings are 3/8 or 5/13
        /// </summary>
        /// <returns></returns>
        public Indicator.RMI RMI(Data.IDataSeries input, int period, int shift)
        {
            return _indicator.RMI(input, period, shift);
        }

    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// The RMI (Relative Momentum Index), develloped by Roger Altman. Similar to RSI but based on comparing Momentums instead of prices. Typical Settings are 3/8 or 5/13
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.RMI RMI(int period, int shift)
        {
            return _indicator.RMI(Input, period, shift);
        }

        /// <summary>
        /// The RMI (Relative Momentum Index), develloped by Roger Altman. Similar to RSI but based on comparing Momentums instead of prices. Typical Settings are 3/8 or 5/13
        /// </summary>
        /// <returns></returns>
        public Indicator.RMI RMI(Data.IDataSeries input, int period, int shift)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.RMI(input, period, shift);
        }

    }
}
#endregion
