#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Calculates Z-value of price change for a given period.
    /// </summary>
    [Description("Calculates Z-value of price change for a given period.")]
    public class ZTrend : Indicator
    {
        #region Variables
        // Wizard generated variables
            private int period = 30; // Default setting for Period
        // User defined variables (add any user defined variables below)
		private DataSeries					Prc_chg;
		private DataSeries					Calc_mean;
		private DataSeries					Calc_sd;
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
            Add(new Plot(Color.FromKnownColor(KnownColor.Red), PlotStyle.Line, "ZValue"));
            Add(new Line(Color.FromKnownColor(KnownColor.Black), 0, "Zero"));
			Prc_chg				= new DataSeries(this);
			Calc_mean			= new DataSeries(this);
			Calc_sd  			= new DataSeries(this);
            CalculateOnBarClose	= true;
            Overlay				= false;
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
            // Use this method for calculating your indicator values. Assign a value to each
            // plot below by replacing 'Close[0]' with your own formula.
			if (CurrentBar < Period)
				return;
				
			Prc_chg.Set((Close[0] - Close[1]) / Close[1]);
			Calc_mean.Set((SUM(Prc_chg,Period)[0])/Period);
			Calc_sd.Set(StdDev(Prc_chg,Period)[0]);
			ZValue.Set(Calc_mean[0] / Calc_sd[0]);
			
        }

        #region Properties
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries ZValue
        {
            get { return Values[0]; }
        }

        [Description("Period used for calculation")]
        [GridCategory("Parameters")]
        public int Period
        {
            get { return period; }
            set { period = Math.Max(2, value); }
        }
        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private ZTrend[] cacheZTrend = null;

        private static ZTrend checkZTrend = new ZTrend();

        /// <summary>
        /// Calculates Z-value of price change for a given period.
        /// </summary>
        /// <returns></returns>
        public ZTrend ZTrend(int period)
        {
            return ZTrend(Input, period);
        }

        /// <summary>
        /// Calculates Z-value of price change for a given period.
        /// </summary>
        /// <returns></returns>
        public ZTrend ZTrend(Data.IDataSeries input, int period)
        {
            if (cacheZTrend != null)
                for (int idx = 0; idx < cacheZTrend.Length; idx++)
                    if (cacheZTrend[idx].Period == period && cacheZTrend[idx].EqualsInput(input))
                        return cacheZTrend[idx];

            lock (checkZTrend)
            {
                checkZTrend.Period = period;
                period = checkZTrend.Period;

                if (cacheZTrend != null)
                    for (int idx = 0; idx < cacheZTrend.Length; idx++)
                        if (cacheZTrend[idx].Period == period && cacheZTrend[idx].EqualsInput(input))
                            return cacheZTrend[idx];

                ZTrend indicator = new ZTrend();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.Period = period;
                Indicators.Add(indicator);
                indicator.SetUp();

                ZTrend[] tmp = new ZTrend[cacheZTrend == null ? 1 : cacheZTrend.Length + 1];
                if (cacheZTrend != null)
                    cacheZTrend.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheZTrend = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Calculates Z-value of price change for a given period.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.ZTrend ZTrend(int period)
        {
            return _indicator.ZTrend(Input, period);
        }

        /// <summary>
        /// Calculates Z-value of price change for a given period.
        /// </summary>
        /// <returns></returns>
        public Indicator.ZTrend ZTrend(Data.IDataSeries input, int period)
        {
            return _indicator.ZTrend(input, period);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Calculates Z-value of price change for a given period.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.ZTrend ZTrend(int period)
        {
            return _indicator.ZTrend(Input, period);
        }

        /// <summary>
        /// Calculates Z-value of price change for a given period.
        /// </summary>
        /// <returns></returns>
        public Indicator.ZTrend ZTrend(Data.IDataSeries input, int period)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.ZTrend(input, period);
        }
    }
}
#endregion
