// 
// Copyright (C) 2008, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Double stochastics
    /// </summary>
    [Description("Double Stochastics")]
    [Gui.Design.DisplayName("anaDoubleStochastics")]
    public class anaDoubleStochastics : Indicator
    {
        #region Variables
        private int 				period 		= 34;
        private int 				periodEMA 	= 21;
		private Color				upColor		= Color.Blue;
		private Color				downColor	= Color.Red;
		private int					plot0Width	= 2;
		private PlotStyle			plot0Style	= PlotStyle.Line;
		private DashStyle			dash0Style	= DashStyle.Solid;
		private DataSeries p1;
		private DataSeries p2;
		private DataSeries p3;
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
            Add(new Plot(Color.Gray, PlotStyle.Line, "K"));
            Add(new Line(Color.LightSteelBlue, 90, "Upper"));
            Add(new Line(Color.LightSteelBlue, 50, "Middle"));
            Add(new Line(Color.LightSteelBlue, 10, "Lower"));
			Lines[0].Pen.DashStyle = DashStyle.Dash;
			Lines[1].Pen.DashStyle = DashStyle.Dash;

			p1 = new DataSeries(this);
			p2 = new DataSeries(this);
			p3 = new DataSeries(this);
			
  			PlotsConfigurable	= false;
       }

 		/// <summary>
		/// </summary>
		protected override void OnStartUp()
		{
			Plots[0].Pen.Width 		= plot0Width;
			Plots[0].PlotStyle		= plot0Style;
			Plots[0].Pen.DashStyle 	= dash0Style;
		}
		
       /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
			double r = MAX(High, Period)[0] - MIN(Low, Period)[0];
            r = r.Compare(0, 0.000000000001) == 0 ? 0 : r;

            if (r == 0)
                p1.Set(CurrentBar == 0 ? 50 : p1[1]);
            else
                p1.Set(Math.Min(100, Math.Max(0, 100 * (Close[0] - MIN(Low, Period)[0]) / r)));

			p2.Set(EMA(p1, periodEMA)[0]);
			
			double s = MAX(p2, Period)[0] - MIN(p2, Period)[0];
            s = s.Compare(0, 0.000000000001) == 0 ? 0 : s;

            if (s == 0)
                p3.Set(CurrentBar == 0 ? 50 : p3[1]);
            else
                p3.Set(Math.Min(100, Math.Max(0, 100 * (p2[0] - MIN(p2, Period)[0]) / s)));

			K.Set(EMA(p3, periodEMA)[0]);
 			
			if(CurrentBar < 1)
			{
				PlotColors[0][0] = upColor;
				return;
			}
			else if (Rising(K))
				PlotColors[0][0] = upColor;
			else 
				PlotColors[0][0] = downColor;
       }

        #region Properties
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries K
        {
            get { return Values[0]; }
        }

        [Description("Select Period for Double Stochastics")]
        [GridCategory("Options")]
		[Gui.Design.DisplayName("Double Stoch Period")]
        public int Period
        {
            get { return period; }
            set { period = Math.Max(1, value); }
        }

		[Description("Select Period for Smoothing Average")]
        [GridCategory("Options")]
		[Gui.Design.DisplayName("EMA Period")]
        public int PeriodEMA
        {
            get { return periodEMA; }
            set { periodEMA = Math.Max(1, value); }
        }
		
		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Select color for rising Double Stochastics")]
		[Category("Plot Colors")]
		[Gui.Design.DisplayName("Double Stoch Rising")]
		public Color UpColor
		{
			get { return upColor; }
			set { upColor = value; }
		}
		
		// Serialize Color object
		[Browsable(false)]
		public string UpColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(upColor); }
			set { upColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Select color for falling Double Stochastics")]
		[Category("Plot Colors")]
		[Gui.Design.DisplayName("Double Stoch Falling")]
		public Color DownColor
		{
			get { return downColor; }
			set { downColor = value; }
		}
		
		// Serialize Color object
		[Browsable(false)]
		public string DownColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(downColor); }
			set { downColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
 		/// <summary>
		/// </summary>
		[Description("Width for Double Stochastics.")]
		[Category("Plot Parameters")]
		[Gui.Design.DisplayNameAttribute("Line Width")]
		public int Plot0Width
		{
			get { return plot0Width; }
			set { plot0Width = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[Description("DashStyle for Double Stochastics")]
		[Category("Plot Parameters")]
		[Gui.Design.DisplayNameAttribute("Plot Style")]
		public PlotStyle Plot0Style
		{
			get { return plot0Style; }
			set { plot0Style = value; }
		}
		
		/// <summary>
		/// </summary>
		[Description("DashStyle for Double Stochastics")]
		[Category("Plot Parameters")]
		[Gui.Design.DisplayNameAttribute("Dash Style")]
		public DashStyle Dash0Style
		{
			get { return dash0Style; }
			set { dash0Style = value; }
		}
       #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private anaDoubleStochastics[] cacheanaDoubleStochastics = null;

        private static anaDoubleStochastics checkanaDoubleStochastics = new anaDoubleStochastics();

        /// <summary>
        /// Double Stochastics
        /// </summary>
        /// <returns></returns>
        public anaDoubleStochastics anaDoubleStochastics(int period, int periodEMA)
        {
            return anaDoubleStochastics(Input, period, periodEMA);
        }

        /// <summary>
        /// Double Stochastics
        /// </summary>
        /// <returns></returns>
        public anaDoubleStochastics anaDoubleStochastics(Data.IDataSeries input, int period, int periodEMA)
        {
            if (cacheanaDoubleStochastics != null)
                for (int idx = 0; idx < cacheanaDoubleStochastics.Length; idx++)
                    if (cacheanaDoubleStochastics[idx].Period == period && cacheanaDoubleStochastics[idx].PeriodEMA == periodEMA && cacheanaDoubleStochastics[idx].EqualsInput(input))
                        return cacheanaDoubleStochastics[idx];

            lock (checkanaDoubleStochastics)
            {
                checkanaDoubleStochastics.Period = period;
                period = checkanaDoubleStochastics.Period;
                checkanaDoubleStochastics.PeriodEMA = periodEMA;
                periodEMA = checkanaDoubleStochastics.PeriodEMA;

                if (cacheanaDoubleStochastics != null)
                    for (int idx = 0; idx < cacheanaDoubleStochastics.Length; idx++)
                        if (cacheanaDoubleStochastics[idx].Period == period && cacheanaDoubleStochastics[idx].PeriodEMA == periodEMA && cacheanaDoubleStochastics[idx].EqualsInput(input))
                            return cacheanaDoubleStochastics[idx];

                anaDoubleStochastics indicator = new anaDoubleStochastics();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.Period = period;
                indicator.PeriodEMA = periodEMA;
                Indicators.Add(indicator);
                indicator.SetUp();

                anaDoubleStochastics[] tmp = new anaDoubleStochastics[cacheanaDoubleStochastics == null ? 1 : cacheanaDoubleStochastics.Length + 1];
                if (cacheanaDoubleStochastics != null)
                    cacheanaDoubleStochastics.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheanaDoubleStochastics = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Double Stochastics
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.anaDoubleStochastics anaDoubleStochastics(int period, int periodEMA)
        {
            return _indicator.anaDoubleStochastics(Input, period, periodEMA);
        }

        /// <summary>
        /// Double Stochastics
        /// </summary>
        /// <returns></returns>
        public Indicator.anaDoubleStochastics anaDoubleStochastics(Data.IDataSeries input, int period, int periodEMA)
        {
            return _indicator.anaDoubleStochastics(input, period, periodEMA);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Double Stochastics
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.anaDoubleStochastics anaDoubleStochastics(int period, int periodEMA)
        {
            return _indicator.anaDoubleStochastics(Input, period, periodEMA);
        }

        /// <summary>
        /// Double Stochastics
        /// </summary>
        /// <returns></returns>
        public Indicator.anaDoubleStochastics anaDoubleStochastics(Data.IDataSeries input, int period, int periodEMA)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.anaDoubleStochastics(input, period, periodEMA);
        }
    }
}
#endregion
