// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Collections;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// The ZigZag indicator shows trend lines filtering out changes below a defined level. 
    /// </summary>
    [Description("The ZigZag indicator shows trend lines filtering out changes below a defined level. ")]
    public class ZigZag : Indicator
    {
        #region Variables
		private double			currentZigZagHigh	= 0;
		private double			currentZigZagLow	= 0;
		private DeviationType	deviationType		= DeviationType.Points;
		private double			deviationValue		= 0.5;
		private DataSeries		zigZagHighZigZags; 
		private DataSeries		zigZagLowZigZags; 
		private DataSeries		zigZagHighSeries; 
		private DataSeries		zigZagLowSeries; 
		private int				lastSwingIdx		= -1;
		private double			lastSwingPrice		= 0.0;
		private int				trendDir			= 0; // 1 = trend up, -1 = trend down, init = 0
		private bool			useHighLow			= false;

        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
            Add(new Plot(Color.Blue, PlotStyle.Line, "ZigZag"));

			zigZagHighSeries	= new DataSeries(this); 
			zigZagHighZigZags	= new DataSeries(this); 
			zigZagLowSeries		= new DataSeries(this); 
			zigZagLowZigZags	= new DataSeries(this); 

            CalculateOnBarClose	= true;
			DisplayInDataBox	= false;
            Overlay				= true;
			PaintPriceMarkers	= false;
            PriceTypeSupported	= true;
        }

		/// <summary>
		/// Returns the number of bars ago a zig zag low occurred. Returns a value of -1 if a zig zag low is not found within the look back period.
		/// </summary>
		/// <param name="barsAgo"></param>
		/// <param name="instance"></param>
		/// <param name="lookBackPeriod"></param>
		/// <returns></returns>
		public int LowBar(int barsAgo, int instance, int lookBackPeriod) 
		{
			if (instance < 1)
				throw new Exception(GetType().Name + ".LowBar: instance must be greater/equal 1 but was " + instance);
			else if (barsAgo < 0)
				throw new Exception(GetType().Name + ".LowBar: barsAgo must be greater/equal 0 but was " + barsAgo);
			else if (barsAgo >= Count)
				throw new Exception(GetType().Name + ".LowBar: barsAgo out of valid range 0 through " + (Count - 1) + ", was " + barsAgo + ".");

			for (int idx=CurrentBar - barsAgo - 1; idx >= CurrentBar - barsAgo - 1 - lookBackPeriod; idx--)
			{
				if (idx < 0)
					return -1;
				if (idx >= zigZagLowZigZags.Count)
					continue;				

				if (zigZagLowZigZags.Get(idx).Equals(0.0))			
					continue;

				if (instance == 1) // 1-based, < to be save
					return CurrentBar - idx;	

				instance--;
			}
	
			return -1;
		}

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
			if (CurrentBar < 2) // need 3 bars to calculate Low/High
				return;

			// Initialization
			if (lastSwingPrice == 0.0)
				lastSwingPrice = Close[0];

			IDataSeries highSeries	= High;
			IDataSeries lowSeries	= Low;

			if (!useHighLow)
			{
				highSeries	= Input;
				lowSeries	= Input;
			}

			// Calculation always for 1-bar ago !

			double tickSize = Bars.Instrument.MasterInstrument.TickSize;
			bool isSwingHigh	= highSeries[1] >= highSeries[0] - double.Epsilon 
								&& highSeries[1] >= highSeries[2] - double.Epsilon;
			bool isSwingLow		= lowSeries[1] <= lowSeries[0] + double.Epsilon 
								&& lowSeries[1] <= lowSeries[2] + double.Epsilon;  
			bool isOverHighDeviation	= (deviationType == DeviationType.Percent && IsPriceGreater(highSeries[1], (lastSwingPrice * (1.0 + deviationValue * 0.01))))
										|| (deviationType == DeviationType.Points && IsPriceGreater(highSeries[1], lastSwingPrice + deviationValue));
			bool isOverLowDeviation		= (deviationType == DeviationType.Percent && IsPriceGreater(lastSwingPrice * (1.0 - deviationValue * 0.01), lowSeries[1]))
										|| (deviationType == DeviationType.Points && IsPriceGreater(lastSwingPrice - deviationValue, lowSeries[1]));

			double	saveValue	= 0.0;
			bool	addHigh		= false; 
			bool	addLow		= false; 
			bool	updateHigh	= false; 
			bool	updateLow	= false; 

			if (!isSwingHigh && !isSwingLow)
			{
				zigZagHighSeries.Set(currentZigZagHigh);
				zigZagLowSeries.Set(currentZigZagLow);
				return;
			}
			
			if (trendDir <= 0 && isSwingHigh && isOverHighDeviation)
			{	
				saveValue	= highSeries[1];
				addHigh		= true;
				trendDir	= 1;
			}	
			else if (trendDir >= 0 && isSwingLow && isOverLowDeviation)
			{	
				saveValue	= lowSeries[1];
				addLow		= true;
				trendDir	= -1;
			}	
			else if (trendDir == 1 && isSwingHigh && IsPriceGreater(highSeries[1], lastSwingPrice)) 
			{
				saveValue	= highSeries[1];
				updateHigh	= true;
			}
			else if (trendDir == -1 && isSwingLow && IsPriceGreater(lastSwingPrice, lowSeries[1])) 
			{
				saveValue	= lowSeries[1];
				updateLow	= true;
			}

			if (addHigh || addLow || updateHigh || updateLow)
			{
				if (updateHigh && lastSwingIdx >= 0)
					zigZagHighZigZags.Set(CurrentBar - lastSwingIdx, 0);
				else if (updateLow && lastSwingIdx >= 0)
					zigZagLowZigZags.Set(CurrentBar - lastSwingIdx, 0);

				if (addHigh || updateHigh)
				{
					zigZagHighZigZags.Set(1, saveValue);
					zigZagHighZigZags.Set(0, 0);

					currentZigZagHigh = saveValue;
					zigZagHighSeries.Set(1, currentZigZagHigh);
				}
				else if (addLow || updateLow) 
				{
					zigZagLowZigZags.Set(1, saveValue);
					zigZagLowZigZags.Set(0, 0);

					currentZigZagLow = saveValue;
					zigZagLowSeries.Set(1, currentZigZagLow);
				}

				lastSwingIdx	= CurrentBar - 1;
				lastSwingPrice	= saveValue;
			}

			zigZagHighSeries.Set(currentZigZagHigh);
			zigZagLowSeries.Set(currentZigZagLow);
        }

        #region Properties
        [Description("Deviation in percent or points regarding on the deviation type")]
        [Category("Parameters")]
		[Gui.Design.DisplayName("Deviation value")]
        public double DeviationValue
        {
            get { return deviationValue; }
            set { deviationValue = Math.Max(0.0, value); }
        }

        [Description("Type of the deviation value")]
        [Category("Parameters")]
		[Gui.Design.DisplayName("Deviation type")]
        public DeviationType DeviationType
        {
            get { return deviationType; }
            set { deviationType = value; }
        }

        [Description("If true, high and low instead of selected price type is used to plot indicator.")]
        [Category("Parameters")]
		[Gui.Design.DisplayName("Use high and low")]
		[RefreshProperties(RefreshProperties.All)]
        public bool UseHighLow
        {
            get { return useHighLow; }
            set 
			{ 
				useHighLow			= value; 
				PriceTypeSupported	= !value;
			}
        }

		/// <summary>
		/// Gets the ZigZag high points.
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries ZigZagHigh
		{
			get 
			{ 
				Update();
				return zigZagHighSeries; 
			}
		}

		/// <summary>
		/// Gets the ZigZag low points.
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries ZigZagLow
		{
			get 
			{ 
				Update();
				return zigZagLowSeries; 
			}
		}
        #endregion

		#region Miscellaneous

		/// <summary>
		/// Returns the number of bars ago a zig zag high occurred. Returns a value of -1 if a zig zag high is not found within the look back period.
		/// </summary>
		/// <param name="barsAgo"></param>
		/// <param name="instance"></param>
		/// <param name="lookBackPeriod"></param>
		/// <returns></returns>
		public int HighBar(int barsAgo, int instance, int lookBackPeriod) 
		{
			if (instance < 1)
				throw new Exception(GetType().Name + ".HighBar: instance must be greater/equal 1 but was " + instance);
			else if (barsAgo < 0)
				throw new Exception(GetType().Name + ".HighBar: barsAgo must be greater/equal 0 but was " + barsAgo);
			else if (barsAgo >= Count)
				throw new Exception(GetType().Name + ".HighBar: barsAgo out of valid range 0 through " + (Count - 1) + ", was " + barsAgo + ".");

			for (int idx=CurrentBar - barsAgo - 1; idx >= CurrentBar - barsAgo - 1 - lookBackPeriod; idx--)
			{
				if (idx < 0)
					return -1;
				if (idx >= zigZagHighZigZags.Count)
					continue;				

				if (zigZagHighZigZags.Get(idx).Equals(0.0))			
					continue;

				if (instance <= 1) // 1-based, < to be save
					return CurrentBar - idx;	

				instance--;
			}

			return -1;
		}

		private bool IsPriceGreater(double a, double b)
		{
			if (a > b && a - b > TickSize / 2)
				return true; 
			else 
				return false;
		}

		public override void Plot(Graphics graphics, Rectangle bounds, double min, double max)
		{
			if (Bars == null || ChartControl == null)
				return;

			int preDiff = 1;
			for (int i = ChartControl.FirstBarPainted - 1; i >= BarsRequired; i--)
			{
				if (i < 0)
					break;

				bool isHigh	= zigZagHighZigZags.IsValidPlot(i) && zigZagHighZigZags.Get(i) > 0;
				bool isLow	= zigZagLowZigZags.IsValidPlot(i) && zigZagLowZigZags.Get(i) > 0;
				
				if (isHigh || isLow)
					break;

				preDiff++;
			}

			int postDiff = 0;
			for (int i=ChartControl.LastBarPainted; i <= zigZagHighZigZags.Count; i++)
			{
				bool isHigh	= zigZagHighZigZags.IsValidPlot(i) && zigZagHighZigZags.Get(i) > 0;
				bool isLow	= zigZagLowZigZags.IsValidPlot(i) && zigZagLowZigZags.Get(i) > 0;

				if (isHigh || isLow)
					break;

				postDiff++;
			}

			GraphicsPath	path		= new GraphicsPath();
			int				barWidth	= ChartControl.ChartStyle.GetBarPaintWidth(ChartControl.BarWidth);

			int		lastCount	= -1; 
			double	lastValue	= -1; 

			bool linePlotted = false;
			for (int count = 0-preDiff; count < ChartControl.BarsPainted + postDiff; count++) // - (CalculateOnBarClose ? 1 : 0)
			{
				int idx = ChartControl.LastBarPainted - ChartControl.BarsPainted + 1 + count;
				if (idx - Displacement < 0 || idx - Displacement >= Bars.Count || (!ChartControl.ShowBarsRequired && idx - Displacement < BarsRequired))
					continue;

				bool isHigh	= zigZagHighZigZags.IsValidPlot(idx) && zigZagHighZigZags.Get(idx) > 0;
				bool isLow	= zigZagLowZigZags.IsValidPlot(idx) && zigZagLowZigZags.Get(idx) > 0;

				if (!isHigh && !isLow)
					continue;
				
				double value = isHigh ? zigZagHighZigZags.Get(idx) : zigZagLowZigZags.Get(idx);
				if (lastValue >= 0)
				{	
					int x0 = (int) (ChartControl.CanvasRight - ChartControl.BarMarginRight - barWidth / 2
								- (ChartControl.BarsPainted - 1) * ChartControl.BarSpace + lastCount * ChartControl.BarSpace) + 1;
					int x1 = (int) (ChartControl.CanvasRight - ChartControl.BarMarginRight - barWidth / 2
								- (ChartControl.BarsPainted - 1) * ChartControl.BarSpace + count * ChartControl.BarSpace) + 1;
					int y0 = (int) ((bounds.Y + bounds.Height) - ((lastValue - min ) / Gui.Chart.ChartControl.MaxMinusMin(max, min)) * bounds.Height);
					int y1 = (int) ((bounds.Y + bounds.Height) - ((value - min ) / Gui.Chart.ChartControl.MaxMinusMin(max, min)) * bounds.Height);

					path.AddLine(x0, y0, x1, y1);
					linePlotted = true;
				}

				// save as previous point
				lastCount	= count; 
				lastValue	= value; 
			}

			SmoothingMode oldSmoothingMode = graphics.SmoothingMode;
			graphics.SmoothingMode = SmoothingMode.AntiAlias;
			graphics.DrawPath(Plots[0].Pen, path);
			graphics.SmoothingMode = oldSmoothingMode;

			if (!linePlotted)
				DrawTextFixed("ZigZagErrorMsg", "ZigZag can not plot any values since the deviation value is too large. Please reduce it.", TextPosition.BottomRight);
		}
		
		#endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private ZigZag[] cacheZigZag = null;

        private static ZigZag checkZigZag = new ZigZag();

        /// <summary>
        /// The ZigZag indicator shows trend lines filtering out changes below a defined level. 
        /// </summary>
        /// <returns></returns>
        public ZigZag ZigZag(DeviationType deviationType, double deviationValue, bool useHighLow)
        {
            return ZigZag(Input, deviationType, deviationValue, useHighLow);
        }

        /// <summary>
        /// The ZigZag indicator shows trend lines filtering out changes below a defined level. 
        /// </summary>
        /// <returns></returns>
        public ZigZag ZigZag(Data.IDataSeries input, DeviationType deviationType, double deviationValue, bool useHighLow)
        {
            checkZigZag.DeviationType = deviationType;
            deviationType = checkZigZag.DeviationType;
            checkZigZag.DeviationValue = deviationValue;
            deviationValue = checkZigZag.DeviationValue;
            checkZigZag.UseHighLow = useHighLow;
            useHighLow = checkZigZag.UseHighLow;

            if (cacheZigZag != null)
                for (int idx = 0; idx < cacheZigZag.Length; idx++)
                    if (cacheZigZag[idx].DeviationType == deviationType && Math.Abs(cacheZigZag[idx].DeviationValue - deviationValue) <= double.Epsilon && cacheZigZag[idx].UseHighLow == useHighLow && cacheZigZag[idx].EqualsInput(input))
                        return cacheZigZag[idx];

            ZigZag indicator = new ZigZag();
            indicator.BarsRequired = BarsRequired;
            indicator.CalculateOnBarClose = CalculateOnBarClose;
            indicator.Input = input;
            indicator.DeviationType = deviationType;
            indicator.DeviationValue = deviationValue;
            indicator.UseHighLow = useHighLow;
            indicator.SetUp();

            ZigZag[] tmp = new ZigZag[cacheZigZag == null ? 1 : cacheZigZag.Length + 1];
            if (cacheZigZag != null)
                cacheZigZag.CopyTo(tmp, 0);
            tmp[tmp.Length - 1] = indicator;
            cacheZigZag = tmp;
            Indicators.Add(indicator);

            return indicator;
        }

    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// The ZigZag indicator shows trend lines filtering out changes below a defined level. 
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.ZigZag ZigZag(DeviationType deviationType, double deviationValue, bool useHighLow)
        {
            return _indicator.ZigZag(Input, deviationType, deviationValue, useHighLow);
        }

        /// <summary>
        /// The ZigZag indicator shows trend lines filtering out changes below a defined level. 
        /// </summary>
        /// <returns></returns>
        public Indicator.ZigZag ZigZag(Data.IDataSeries input, DeviationType deviationType, double deviationValue, bool useHighLow)
        {
            return _indicator.ZigZag(input, deviationType, deviationValue, useHighLow);
        }

    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// The ZigZag indicator shows trend lines filtering out changes below a defined level. 
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.ZigZag ZigZag(DeviationType deviationType, double deviationValue, bool useHighLow)
        {
            return _indicator.ZigZag(Input, deviationType, deviationValue, useHighLow);
        }

        /// <summary>
        /// The ZigZag indicator shows trend lines filtering out changes below a defined level. 
        /// </summary>
        /// <returns></returns>
        public Indicator.ZigZag ZigZag(Data.IDataSeries input, DeviationType deviationType, double deviationValue, bool useHighLow)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.ZigZag(input, deviationType, deviationValue, useHighLow);
        }

    }
}
#endregion
