#region Using declarations
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

/*
changelog:

v2.0:
- added option to turn off histograms (saves a lot of memory)
- realigned position of smaller bars better with chart price
- there's a bug on using the current bar when ninja starts with auto opened charts, open a new chart to avoid it
*/

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Volume at price per # of bars.  Neutral prices are ignored.  edsfreedom.blogspot.com
    /// </summary>
    [Description("Real-time volume at price per # of bars which helps in visualizing time & sales.  Neutral prices are ignored.  There's a bug on using the current bar when ninja starts with auto opened charts, open a new chart to avoid it.  edsfreedom.blogspot.com")]
    public class EdsVolPriceBar : Indicator
    {
        public override string ToString()
		{
			return "Eds_Vol_Price_Bar_v2.0";
		}
		
        #region Variables
		private SortedList<int, SortedList<double, VolumeTradedRow>> volumeRowsArray = new SortedList<int, SortedList<double, VolumeTradedRow>>();
		private SortedList<double, VolumeTradedRow> volumeRows = new SortedList<double, VolumeTradedRow>();
		private VolumeTradedRow oneVolumeRow = null;
		private double mostRecentBearPrice = 0;
		private double mostRecentBullPrice = 0;
		private int mostRecentBearVolume = 0;
		private int mostRecentBullVolume = 0;
		
		private Font textFont;
		private StringFormat stringFormat;
		private Color textColor2 = Color.Gray;
		private Brush textBrush2;
		private Color textBearColor = Color.Red;
		private Brush textBearBrush;
		private Color textBullColor = Color.Blue;
		private Brush textBullBrush;
		private Brush histBearBrush;
		private Brush histBullBrush;
		private Brush whichBrush;
		private Rectangle rect;
		
		private bool showHistogram = true;
		private bool showVolumeTradedText = true;
		private DateTime lastRefresh = DateTime.MinValue;
		private int refreshDelay = 300;
		private int histBarOpacity = 25;
		private int xOffset = 52;
		
		private int x;
		private int y;
		private int totalY;
		private double maxMinusMin;
		private int barLength;
		private int barYOffset;
		private double barLengthScale = 1;
		private int barHeight;
		private int largestVolume = 0;
		private int i = 0;
		private int totalIndex = 0;
		private bool isCramped = false;
		
		private double lastPrice = 0;
		private double askPrice = 0;
		private double bidPrice = 0;
		
		int savedBar = 0;
		int numOfBarsBeforeReset = 5;
		int volumeX = 0;
		#endregion

		#region Properties
		[XmlIgnore()]
		[Description("Histogram bar opacity percentage.")]	
		[Category("Volume Traded")]
		[NinjaTrader.Gui.Design.DisplayName("Bar Opacity")]		
		public int HistogramOpacity
		{
			get{ return histBarOpacity; }
			set{ histBarOpacity = Math.Max(Math.Min(value, 100), 0); }
		}
		
		[XmlIgnore()]
		[Description("Right side margin for number text")]	
		[Category("Volume Traded")]
		[NinjaTrader.Gui.Design.DisplayName("Right Margin")]		
		public int XOffset
		{
			get{ return xOffset; }
			set{ xOffset = value; }
		}
		
		[XmlIgnore()]
		[Description("Refresh delay in milliseconds to avoid overloading CPU.")]	
		[Category("Volume Traded")]
		[NinjaTrader.Gui.Design.DisplayName("Refresh Delay")]		
		public int RefreshDelay
		{
			get{ return refreshDelay; }
			set{ refreshDelay = value; }
		}
		
		[XmlIgnore()]
		[Description("Shows histogram (hide so to avoid using a lot of memory)")]	
		[Category("Volume Traded")]
		[NinjaTrader.Gui.Design.DisplayName("Show Histograms")]		
		public bool ShowHistogram
		{
			get{ return showHistogram; }
			set{ showHistogram = value; }
		}
		
		[XmlIgnore()]
		[Description("Shows the volume traded numbers of the most recent group of bars.")]	
		[Category("Volume Traded")]
		[NinjaTrader.Gui.Design.DisplayName("Show Numbers")]		
		public bool ShowVolumeTradedText
		{
			get{ return showVolumeTradedText; }
			set{ showVolumeTradedText = value; }
		}
		
		[XmlIgnore()]
		[Description("The cumulative volume text and bar color where last price was less than or equal to bid price and less than ask price.")]	
		[Category("Volume Traded")]
		[NinjaTrader.Gui.Design.DisplayName("Bear Color")]		
		public Color BearTextColor
		{
			get{ return textBearColor; }
			set{ textBearColor = value; }
		}
		
		[XmlIgnore()]
		[Description("The cumulative volume text and bar color where last price was greater than or equal to ask price and greater than bid price.")]	
		[Category("Volume Traded")]
		[NinjaTrader.Gui.Design.DisplayName("Bull Color")]		
		public Color BullTextColor
		{
			get{ return textBullColor; }
			set{ textBullColor = value; }
		}
		
		[XmlIgnore()]
		[Description("Number of bars per group of histogram.")]	
		[Category("Volume Traded")]
		[NinjaTrader.Gui.Design.DisplayName("Number of bars")]		
		public int NumOfBarsBeforeReset
		{
			get{ return numOfBarsBeforeReset; }
			set{ numOfBarsBeforeReset = Math.Max(value, 1); }
		}
        #endregion
		
		#region Initialize
		protected override void Initialize()
		{
			CalculateOnBarClose	= true;
			Overlay				= true;
			DrawOnPricePanel	= true;
			PaintPriceMarkers 	= false;
			PriceTypeSupported	= false;
			PlotsConfigurable 	= false;
			
			textFont = new Font(FontFamily.GenericSansSerif, 9, FontStyle.Regular, GraphicsUnit.Pixel);
			stringFormat = new StringFormat();
			stringFormat.Alignment = StringAlignment.Far;
			textBrush2 = new SolidBrush(textColor2);
			textBearBrush = new SolidBrush(textBearColor);
			textBullBrush = new SolidBrush(textBullColor);
			histBearBrush = new SolidBrush(Color.FromArgb(histBarOpacity * 255 / 100, textBearColor));
			histBullBrush = new SolidBrush(Color.FromArgb(histBarOpacity * 255 / 100, textBullColor));
		}
		#endregion
	
		#region On Event Updates
		protected override void OnBarUpdate()
		{	
			if (CurrentBar >= savedBar + numOfBarsBeforeReset)
			{
				if (showHistogram)
					volumeRowsArray.Add(savedBar, new SortedList<double, VolumeTradedRow>(volumeRows));
				savedBar = CurrentBar;
				volumeRows.Clear();
			}
		}
		
		protected override void OnMarketData(MarketDataEventArgs e) 
		{ 
			// Print some data to the Output window 
			if (e.MarketDataType == MarketDataType.Last)
			{	
				lastPrice = e.Price;
				if ((lastPrice >= askPrice) && (lastPrice > bidPrice))
				{
					if (volumeRows.ContainsKey(e.Price) == false)
						volumeRows.Add(e.Price, new VolumeTradedRow(0, (int) e.Volume));
					else
						volumeRows[e.Price].Bull += (int) e.Volume;
					mostRecentBullPrice = e.Price;
					mostRecentBullVolume = volumeRows[e.Price].Bull;
				}
				else if ((lastPrice <= bidPrice) && (lastPrice < askPrice))
				{
					if (volumeRows.ContainsKey(e.Price) == false)
						volumeRows.Add(e.Price, new VolumeTradedRow((int) e.Volume, 0));
					else
						volumeRows[e.Price].Bear += (int) e.Volume;
					mostRecentBearPrice = e.Price;
					mostRecentBearVolume = volumeRows[e.Price].Bear;
				}
				else // neutrals are too insignificant to waste ram space and processing power, just return
					return;
					
				if (volumeRows[e.Price].Bear + volumeRows[e.Price].Bull > largestVolume)
					largestVolume = volumeRows[e.Price].Bear + volumeRows[e.Price].Bull;
					
				if (DateTime.Now > lastRefresh.AddMilliseconds(refreshDelay))
				{
					ChartControl.Refresh();
					lastRefresh = DateTime.Now;
				}
			}
			else if (e.MarketDataType == MarketDataType.Ask)
				askPrice = e.Price;
			
			else if (e.MarketDataType == MarketDataType.Bid)
				bidPrice = e.Price;
		}	
		#endregion
		
		#region Draw Plot
		public override void Plot(Graphics graphics, Rectangle bounds, double min, double max)
		{	
			x = ChartControl.CanvasRight;
			totalY = bounds.Y + bounds.Height;
			maxMinusMin = ChartControl.MaxMinusMin(max, min);
			
			// if cramped, draw one histogram bar per two depths
			isCramped = (((totalY - (int) (((min) / maxMinusMin) * bounds.Height)) -
						  (totalY - (int) (((min + TickSize) / maxMinusMin) * bounds.Height))) < 8);
			
			if (showHistogram)
			{
				barLengthScale = (ChartControl.BarSpace * numOfBarsBeforeReset - 1) * 1.0 / largestVolume;
				barHeight = isCramped ? 3 : 7;
				barYOffset = isCramped ? 5 : 3;
				
				foreach (KeyValuePair<int, SortedList<double, VolumeTradedRow>> keyArray in volumeRowsArray)
				{
					volumeX = ChartControl.GetXByBarIdx(BarsArray[0], keyArray.Key) + (ChartControl.BarSpace / 2) + 1;
						
					foreach (KeyValuePair<double, VolumeTradedRow> key in keyArray.Value)
					{
						oneVolumeRow = key.Value;
						y = totalY - ((int) (((key.Key - min) / maxMinusMin) * bounds.Height)) - 7;	
						barLength = (int) (oneVolumeRow.Bull * barLengthScale);
						rect = new Rectangle(volumeX, y + barYOffset, barLength, barHeight);
						graphics.FillRectangle(histBullBrush, rect);
						rect = new Rectangle(volumeX + barLength, y + barYOffset, (int) (oneVolumeRow.Bear * barLengthScale), barHeight);
						graphics.FillRectangle(histBearBrush, rect);
					}
				}
				
				volumeX = ChartControl.GetXByBarIdx(BarsArray[0], savedBar) + (ChartControl.BarSpace / 2) + 1;
			}
			
			foreach (KeyValuePair<double, VolumeTradedRow> key in volumeRows)
			{
				oneVolumeRow = key.Value;
				y = totalY - ((int) (((key.Key - min) / maxMinusMin) * bounds.Height)) - 7;
				if (showVolumeTradedText && (isCramped == false) && (ChartControl.LastBarPainted >= Bars.Count - 1))
				{
					whichBrush = (key.Key == mostRecentBearPrice) ? textBearBrush : textBrush2;
					if (oneVolumeRow.Bear > 0)
						graphics.DrawString(RowVolumeString(oneVolumeRow.Bear), textFont, whichBrush, x - xOffset, y, stringFormat);
					whichBrush = (key.Key == mostRecentBullPrice) ? textBullBrush : textBrush2;
					if (oneVolumeRow.Bull > 0)
						graphics.DrawString(RowVolumeString(oneVolumeRow.Bull), textFont, whichBrush, x - xOffset - 26, y, stringFormat);
				}
				
				if (showHistogram)
				{
					barLength = (int) (oneVolumeRow.Bull * barLengthScale);
					rect = new Rectangle(volumeX, y + barYOffset, barLength, barHeight);
					graphics.FillRectangle(histBullBrush, rect);
					rect = new Rectangle(volumeX + barLength, y + barYOffset, (int) (oneVolumeRow.Bear * barLengthScale), barHeight);
					graphics.FillRectangle(histBearBrush, rect);
				}
			}
			
			if (showVolumeTradedText && isCramped && (ChartControl.LastBarPainted >= Bars.Count - 1))
			{
				y = totalY - ((int) (((mostRecentBearPrice - min) / maxMinusMin) * bounds.Height)) - 7;
				graphics.DrawString(RowVolumeString(mostRecentBearVolume), textFont, textBearBrush, x - xOffset, y, stringFormat);
				y = totalY - ((int) (((mostRecentBullPrice - min) / maxMinusMin) * bounds.Height)) - 7;
				graphics.DrawString(RowVolumeString(mostRecentBullVolume), textFont, textBullBrush, x - xOffset - 26, y, stringFormat);
			}
		}
		#endregion
		
		#region Support Methods
		private string RowVolumeString(int volume)
		{
			if (volume >= 1000000)
				return (volume / 1000000).ToString() + "M";
			else if (volume >= 100000)
				return (volume / 1000).ToString() + "K";
			else
				return volume.ToString();
		}
		
		private class VolumeTradedRow
		{
			public int Bear;
			public int Bull;

			public VolumeTradedRow(int myBear, int myBull)
			{
				Bear = myBear;
				Bull = myBull;
			}
		}
		#endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private EdsVolPriceBar[] cacheEdsVolPriceBar = null;

        private static EdsVolPriceBar checkEdsVolPriceBar = new EdsVolPriceBar();

        /// <summary>
        /// Real-time volume at price per # of bars which helps in visualizing time & sales.  Neutral prices are ignored.  There's a bug on using the current bar when ninja starts with auto opened charts, open a new chart to avoid it.  edsfreedom.blogspot.com
        /// </summary>
        /// <returns></returns>
        public EdsVolPriceBar EdsVolPriceBar()
        {
            return EdsVolPriceBar(Input);
        }

        /// <summary>
        /// Real-time volume at price per # of bars which helps in visualizing time & sales.  Neutral prices are ignored.  There's a bug on using the current bar when ninja starts with auto opened charts, open a new chart to avoid it.  edsfreedom.blogspot.com
        /// </summary>
        /// <returns></returns>
        public EdsVolPriceBar EdsVolPriceBar(Data.IDataSeries input)
        {
            if (cacheEdsVolPriceBar != null)
                for (int idx = 0; idx < cacheEdsVolPriceBar.Length; idx++)
                    if (cacheEdsVolPriceBar[idx].EqualsInput(input))
                        return cacheEdsVolPriceBar[idx];

            lock (checkEdsVolPriceBar)
            {
                if (cacheEdsVolPriceBar != null)
                    for (int idx = 0; idx < cacheEdsVolPriceBar.Length; idx++)
                        if (cacheEdsVolPriceBar[idx].EqualsInput(input))
                            return cacheEdsVolPriceBar[idx];

                EdsVolPriceBar indicator = new EdsVolPriceBar();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                Indicators.Add(indicator);
                indicator.SetUp();

                EdsVolPriceBar[] tmp = new EdsVolPriceBar[cacheEdsVolPriceBar == null ? 1 : cacheEdsVolPriceBar.Length + 1];
                if (cacheEdsVolPriceBar != null)
                    cacheEdsVolPriceBar.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheEdsVolPriceBar = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Real-time volume at price per # of bars which helps in visualizing time & sales.  Neutral prices are ignored.  There's a bug on using the current bar when ninja starts with auto opened charts, open a new chart to avoid it.  edsfreedom.blogspot.com
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.EdsVolPriceBar EdsVolPriceBar()
        {
            return _indicator.EdsVolPriceBar(Input);
        }

        /// <summary>
        /// Real-time volume at price per # of bars which helps in visualizing time & sales.  Neutral prices are ignored.  There's a bug on using the current bar when ninja starts with auto opened charts, open a new chart to avoid it.  edsfreedom.blogspot.com
        /// </summary>
        /// <returns></returns>
        public Indicator.EdsVolPriceBar EdsVolPriceBar(Data.IDataSeries input)
        {
            return _indicator.EdsVolPriceBar(input);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Real-time volume at price per # of bars which helps in visualizing time & sales.  Neutral prices are ignored.  There's a bug on using the current bar when ninja starts with auto opened charts, open a new chart to avoid it.  edsfreedom.blogspot.com
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.EdsVolPriceBar EdsVolPriceBar()
        {
            return _indicator.EdsVolPriceBar(Input);
        }

        /// <summary>
        /// Real-time volume at price per # of bars which helps in visualizing time & sales.  Neutral prices are ignored.  There's a bug on using the current bar when ninja starts with auto opened charts, open a new chart to avoid it.  edsfreedom.blogspot.com
        /// </summary>
        /// <returns></returns>
        public Indicator.EdsVolPriceBar EdsVolPriceBar(Data.IDataSeries input)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.EdsVolPriceBar(input);
        }
    }
}
#endregion
