// 
// Copyright (C) 2007, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//
#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Automatically draws a line representing the current trend and generates an alert if the trend line is broken.
    /// </summary>
    [Description("")]
    [Gui.Design.DisplayName("AutoTrendLine")]
    public class AutoTrendLine : Indicator
    {
        #region Variables
			private bool alertOnBreak = true;
            private int strength = 5; // Default setting for Strength
            private int lineWidth = 1; // Default setting for LineWidth
        	private Color downTrendColor = Color.Red;
			private Color upTrendColor = Color.Green;
			private int triggerBarIndex = 0;
			private int signal = 0; // 0 = no signal, 1 = buy signal on down trend break, 2 = sell signal on up trend break
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
			DisplayInDataBox 	= false;
            CalculateOnBarClose	= true;
            Overlay				= true;
            PriceTypeSupported	= false;
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
			signal = 0;
			
			// Calculate up trend line
			int upTrendStartBarsAgo		= 0;
			int upTrendEndBarsAgo 		= 0;
			int upTrendOccurence 		= 1;
			
			while (Low[upTrendEndBarsAgo] <= Low[upTrendStartBarsAgo])
			{
				upTrendStartBarsAgo 	= Swing(Strength).SwingLowBar(0, upTrendOccurence + 1, CurrentBar);
				upTrendEndBarsAgo 		= Swing(Strength).SwingLowBar(0, upTrendOccurence, CurrentBar);
					
				if (upTrendStartBarsAgo < 0 || upTrendEndBarsAgo < 0)
					break;

				upTrendOccurence++;
			}
			
			
			// Calculate down trend line	
			int downTrendStartBarsAgo	= 0;
			int downTrendEndBarsAgo 	= 0;
			int downTrendOccurence 		= 1;
			
			while (High[downTrendEndBarsAgo] >= High[downTrendStartBarsAgo])
			{
				downTrendStartBarsAgo 		= Swing(Strength).SwingHighBar(0, downTrendOccurence + 1, CurrentBar);
				downTrendEndBarsAgo 		= Swing(Strength).SwingHighBar(0, downTrendOccurence, CurrentBar);
					
				if (downTrendStartBarsAgo < 0 || downTrendEndBarsAgo < 0)
					break;
					
				downTrendOccurence++;
			}
			
			
			// Always clear out arrows that mark trend line breaks
			RemoveDrawObject("DownTrendBreak");							
			RemoveDrawObject("UpTrendBreak");
			
			
			// We have found an uptrend and the uptrend is the current trend
			if (upTrendStartBarsAgo > 0 && upTrendEndBarsAgo > 0 && upTrendStartBarsAgo < downTrendStartBarsAgo)
			{
				RemoveDrawObject("DownTrendLine");
				
				// Reset the alert if required
				if (triggerBarIndex != CurrentBar - upTrendEndBarsAgo)
				{
					triggerBarIndex = 0;
					ResetAlert("Alert");
				}
				
				double startBarPrice 	= Low[upTrendStartBarsAgo];
				double endBarPrice 		= Low[upTrendEndBarsAgo];
				double changePerBar 	= (endBarPrice - startBarPrice) / (Math.Abs(upTrendEndBarsAgo - upTrendStartBarsAgo));
				
				// Draw the up trend line
				DrawRay("UpTrendLine", true, upTrendStartBarsAgo, startBarPrice, upTrendEndBarsAgo, endBarPrice, UpTrendColor, DashStyle.Solid, LineWidth);

				// Check for an uptrend line break
				for (int barsAgo = upTrendEndBarsAgo - 1; barsAgo >= 0; barsAgo--) 
				{
					if (Close[barsAgo] < endBarPrice + (Math.Abs(upTrendEndBarsAgo - barsAgo) * changePerBar))
					{
						DrawArrowDown("UpTrendBreak", barsAgo, High[barsAgo] + TickSize, Color.Blue);
					
						// Set the signal only if the break is on the right most bar
						if (barsAgo == 0)
							signal = 2;
						
						// Alert will only trigger in real-time
						if (AlertOnBreak && triggerBarIndex == 0)
						{
							triggerBarIndex = CurrentBar - upTrendEndBarsAgo;
							Alert("Alert", Priority.High, "Up trend line broken", "Alert2.wav", 100000, Color.Black, Color.Red);
						}
						
						break;
					}
				}
			}
			// We have found a downtrend and the downtrend is the current trend
			else if (downTrendStartBarsAgo > 0 && downTrendEndBarsAgo > 0  && upTrendStartBarsAgo > downTrendStartBarsAgo)
			{
				RemoveDrawObject("UpTrendLine");
				
				// Reset the alert if required
				if (triggerBarIndex != CurrentBar - downTrendEndBarsAgo)
				{
					triggerBarIndex = 0;
					ResetAlert("Alert");
				}
				
				double startBarPrice 	= High[downTrendStartBarsAgo];
				double endBarPrice 		= High[downTrendEndBarsAgo];
				double changePerBar 	= (endBarPrice - startBarPrice) / (Math.Abs(downTrendEndBarsAgo - downTrendStartBarsAgo));
				
				// Draw the down trend line
				DrawRay("DownTrendLine", true, downTrendStartBarsAgo, startBarPrice, downTrendEndBarsAgo, endBarPrice, DownTrendColor, DashStyle.Solid, LineWidth);

				// Check for a down trend line break
				for (int barsAgo = downTrendEndBarsAgo - 1; barsAgo >= 0; barsAgo--) 
				{
					if (Close[barsAgo] > endBarPrice + (Math.Abs(downTrendEndBarsAgo - barsAgo) * changePerBar))
					{
						DrawArrowUp("DownTrendBreak", barsAgo, Low[barsAgo] - TickSize, Color.Blue);
						
						// Set the signal only if the break is on the right most bar
						if (barsAgo == 0)
							signal = 1;
						
						// Alert will only trigger in real-time
						if (AlertOnBreak && triggerBarIndex == 0)
						{
							triggerBarIndex = CurrentBar - downTrendEndBarsAgo;
							Alert("Alert", Priority.High, "Down trend line broken", "Alert2.wav", 100000, Color.Black, Color.Green);
						}
						
						break;
					}
				}
			}		
        }

        #region Properties

		[Description("Generates a visual and audible alert on a trend line break")]
        [Category("Parameters")]
        public bool AlertOnBreak
        {
            get { return alertOnBreak; }
            set { alertOnBreak = value; }
        }
		
        [Description("Number of bars required on each side swing pivot points used to connect the trend lines")]
        [Category("Parameters")]
        public int Strength
        {
            get { return strength; }
            set { strength = Math.Max(1, value); }
        }

        [Description("Trend line width")]
        [Category("Parameters")]
        public int LineWidth
        {
            get { return lineWidth; }
            set { lineWidth = Math.Max(1, value); }
        }
		
		/// <summary>
		/// Gets the trade signal. 0 = no signal, 1 = Buy signal on break of down trend line, 2 = Sell signal on break of up trend line
		/// </summary>
		/// <returns></returns>
		public int Signal
		{
			get { Update(); return signal; }
		}

		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Color of the down trend line.")]
		[Category("Colors")]
		[Gui.Design.DisplayNameAttribute("Down trend")]
		public Color DownTrendColor
		{
			get { return downTrendColor; }
			set { downTrendColor = value; }
		}

		/// <summary>
		/// </summary>
		[Browsable(false)]
		public string DownTrendColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(downTrendColor); }
			set { downTrendColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Color of the up trend line.")]
		[Category("Colors")]
		[Gui.Design.DisplayNameAttribute("Up trend")]
		public Color UpTrendColor
		{
			get { return upTrendColor; }
			set { upTrendColor = value; }
		}

		/// <summary>
		/// </summary>
		[Browsable(false)]
		public string UpTrendColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(upTrendColor); }
			set { upTrendColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private AutoTrendLine[] cacheAutoTrendLine = null;

        private static AutoTrendLine checkAutoTrendLine = new AutoTrendLine();

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public AutoTrendLine AutoTrendLine(bool alertOnBreak, int lineWidth, int strength)
        {
            return AutoTrendLine(Input, alertOnBreak, lineWidth, strength);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public AutoTrendLine AutoTrendLine(Data.IDataSeries input, bool alertOnBreak, int lineWidth, int strength)
        {
            if (cacheAutoTrendLine != null)
                for (int idx = 0; idx < cacheAutoTrendLine.Length; idx++)
                    if (cacheAutoTrendLine[idx].AlertOnBreak == alertOnBreak && cacheAutoTrendLine[idx].LineWidth == lineWidth && cacheAutoTrendLine[idx].Strength == strength && cacheAutoTrendLine[idx].EqualsInput(input))
                        return cacheAutoTrendLine[idx];

            lock (checkAutoTrendLine)
            {
                checkAutoTrendLine.AlertOnBreak = alertOnBreak;
                alertOnBreak = checkAutoTrendLine.AlertOnBreak;
                checkAutoTrendLine.LineWidth = lineWidth;
                lineWidth = checkAutoTrendLine.LineWidth;
                checkAutoTrendLine.Strength = strength;
                strength = checkAutoTrendLine.Strength;

                if (cacheAutoTrendLine != null)
                    for (int idx = 0; idx < cacheAutoTrendLine.Length; idx++)
                        if (cacheAutoTrendLine[idx].AlertOnBreak == alertOnBreak && cacheAutoTrendLine[idx].LineWidth == lineWidth && cacheAutoTrendLine[idx].Strength == strength && cacheAutoTrendLine[idx].EqualsInput(input))
                            return cacheAutoTrendLine[idx];

                AutoTrendLine indicator = new AutoTrendLine();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.AlertOnBreak = alertOnBreak;
                indicator.LineWidth = lineWidth;
                indicator.Strength = strength;
                Indicators.Add(indicator);
                indicator.SetUp();

                AutoTrendLine[] tmp = new AutoTrendLine[cacheAutoTrendLine == null ? 1 : cacheAutoTrendLine.Length + 1];
                if (cacheAutoTrendLine != null)
                    cacheAutoTrendLine.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheAutoTrendLine = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.AutoTrendLine AutoTrendLine(bool alertOnBreak, int lineWidth, int strength)
        {
            return _indicator.AutoTrendLine(Input, alertOnBreak, lineWidth, strength);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public Indicator.AutoTrendLine AutoTrendLine(Data.IDataSeries input, bool alertOnBreak, int lineWidth, int strength)
        {
            return _indicator.AutoTrendLine(input, alertOnBreak, lineWidth, strength);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.AutoTrendLine AutoTrendLine(bool alertOnBreak, int lineWidth, int strength)
        {
            return _indicator.AutoTrendLine(Input, alertOnBreak, lineWidth, strength);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public Indicator.AutoTrendLine AutoTrendLine(Data.IDataSeries input, bool alertOnBreak, int lineWidth, int strength)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.AutoTrendLine(input, alertOnBreak, lineWidth, strength);
        }
    }
}
#endregion
