//
//	Correlation Coefficient of 2 Instruments - For NinjaTrader 7 & Higher
//	Kevin Doren version of 5/4/2011
//
//	9/8/2010: Added lookback parameter for Pct Change
//  5/4/2011: Fixes for NT7 compatibility
//
//

#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Correlation of Two Instruments (for NinjaTrader 7)
	/// 
	/// This indicator calculates the Correlation Coefficient of 2 Instruments over "Period" synchronized bars.
	/// It will skip over any unsynchronized bars while making its calculation.
	/// A synchronized bar is one where both input dataseries have a bar with the same time stamp.
	///
	/// For intraday use, use the "Session Template" to exclude bars outside of Regular Trading Hours.
	/// 
    /// </summary>
	/// 
	/// 
	
[Description("Correlation of Two Instruments")]
    public class Correlation : Indicator
    {
		private const int Instrument1=0, Instrument2=1;
		
        #region Variables
            private string symbol2 = "";
			private int period = 100;
			private int periodPctChange = 1;
			private CorrelationType corrType = CorrelationType.Pearson;
			private SyncedDataSeries Instrument2Close;
			private bool supportedBarsPeriodType = false;
        #endregion
	
		#region SynchronizationMethods
/*
  Synchronization of Multi-Instrument Indicators & Strategies - For NinjaTrader 7
  Kevin Doren - Version 4/9/2011

  7/29/2010:  Added "SynchronizedBarCountToday" field
  9/8/2010:   Added "SyncedGet" method
  4/1/2011:   Added "LastDataTime" field, streamlined & simplified some code, added check for time-based Period
  4/9/2011:   Fixed problem with syncing daily bars with different session end times while COBC=false w/ historical data
			
  Multi-instrument indicators (and strategies, for the same concepts apply) by their nature are looking
  at relationships between instruments.  Any calculations or comparisons we do need to be time-synchronized - we want
  to be using data that occurred at the same time.  For certain kinds of analysis, such as intraday
  analysis of heavily-traded instruments during regular trading hours on historical data, the techniques
  shown here probably aren't necessary.  But for real time analysis, especially with instruments that
  might have missing bars, synchronization techniques are neeeded to avoid the misleading results that
  come from comparing data that occurred at different times when the analysis only has meaning if they occurred at the same time.

  There are 3 scenarios: Historical, real-time with (CalculateOnBarClose == true), 
  and real-time with (CalculateOnBarClose == false).  Each has its issues.

  The simplest case is Historical.  In this case, NT7 will synchronize the bars, which means that when
  OnBarUpdate is called for a Bar Series with a particular timestamp, any other Bar Series which
  has a bar with the same timestamp will have that bar set to be the current bar.  This means that
  Closes[PrimaryInstrument][0] and Closes[SyncedInstrument][0] will both point to bars with the same timestamp IF Closes[1] has a
  bar with that timestamp.  If not, Close[SyncedInstrument][0] will point to the most recent bar preceding it.
  So even Close[PrimaryInstrument][0] and Close[SyncedInstrument][0] are not necessarily synchronized.  As you look back in time,
  it can get even more out of sync, because bars might be missing from either series.

  For an indicator like correlation, the calculation makes no sense unless the bars are all synchronized.
  The solution implemented here is to create 2 additional Data Series, each synchronized to the Primary 
  Instrument, because the indicator output is synced to the Primary Instrument.
  The first (base class) Data Series ("SyncedDataSeries") is a synchronized copy of the secondary instrument, Closes[SyncedInstrument].
  The second Data Series is a BoolSeries field (".Synced") which flags the bars that are synced.
		
  In other words, Closes[PrimaryInstrument][x] and SyncedDataSeries class SyncedDataSeries[x] will refer to the same time,
  and the field SyncedDataSeries.Synced[x] will be set to true if a tick from SyncedInstrument occurred during that bar.

  When operating with Historical data, if a synchronized bar exists, it will be Closes[SyncedInstrument][0].
  In real time, with (CalculateOnBarClose==false), it gets more complicated, because NT7 is event-driven,
  which mean that bars don't close because of time.  They close because a tick arrives with a timestamp of
  a later bar, at which time OnBarUpdate is called and the new bar starts forming.  This could take a long time.
  So when we are processing a bar for the primary instrument, we need to also check the unclosed bar of the secondary
  instrument, which we can do by referencing Times[1][-1]; this is safe to do because we know the bar's time
  has expired - we are running OnBarUpdate for the primary instrument for the same timestamp, which can
  only happen because a tick arrived for a later bar.

  Also, in real time, it's possible that the synced instrument has received bars after the time of the
  bar we are currently processing.  A synchronized bar could exist at Closes[SyncedInstrument][1], Closes[SyncedInstrument][2], or earlier.
  So we need to scan back through the seconday instrument's closes to see if the Synced bar is there.
  
  The (base class) Data Series "SyncedDataSeries" is filled from OnBarUpdate for the primary instrument, and it will hold the
  synchronized bars we found using the above techniques.  We set a flag in the "Synced" data series to show which
  locations are synced.  Routines which require lookbacks can be written to perform their calculations using only synced data.

  There is an additional issue when using daily bars.  As of this writing, daily bars carry a timestamp of the end of the
  NT7 session template.  This can change depending on which session template is applied to the instrument, either in the
  instrument manager or in the chart properties.  There is no issue if both instruments have the same session template,
  then their daily bars will have the same timestamp and thus be synchronized.  But if the secondary instrument has a timestamp
  later than the primary instrument, the days will be out of sync unless we strip away the times and just use the dates.
*/		
		public class SyncedDataSeries : DataSeries
		{
			public IndicatorBase IndBase;
			public int PrimaryInstrument;			// BarsArray Index of primary instrument (usually 0 in indicators)
													//   but could be higher than 0 for multi-instrument, multi-timeframe indicators
			public int SyncedInstrument;			// BarsArray Index of secondary instrument we want to synchronize to primary instrument
			public IDataSeries DataArray;			// Output Data Series synced to primary instrument, hold synchronized copy of secondary instrument
			public BoolSeries Synced;				// BoolSeries output, synced with primary instrument, holds flags
													//   true=output (DataArray) holds synchronized value, false = output does not hold synchronized value
			public int SynchronizedBarCount;		// Total number of synchronized bars held in output, but some or all may be inaccessible if MaximumBarsLookBack is set to 256
			public int SynchronizedBarCountToday;	// Total number of synchronized bars so far today
			public bool DataStarted;
			private bool dailyBars;
//			private bool lastSyncWasToUnclosedBar;
			public DateTime LastDataTime;			// Time of the most recent data received from syncedInstrument
			private DateTime primaryInstrumentCurrentBarTime;
			private DateTime syncedInstrumentCurrentBarTime;
			private int primaryInstrumentCurrentBar;
			private int syncedInstrumentCurrentBar;
			
			public SyncedDataSeries (IndicatorBase indicator, int primaryInstrument, int syncedInstrument, IDataSeries dataArray) :
				this (indicator, primaryInstrument, syncedInstrument, dataArray, indicator.MaximumBarsLookBack)
			{
			}
			public SyncedDataSeries (IndicatorBase indicator, int primaryInstrument, int syncedInstrument, IDataSeries dataArray, MaximumBarsLookBack maxLookback) : base (indicator, maxLookback)
			{
				IndBase = indicator;
				PrimaryInstrument = primaryInstrument;
				SyncedInstrument = syncedInstrument;
				SynchronizedBarCount = 0;
				SynchronizedBarCountToday = 0;
				DataArray = dataArray;
				DataStarted = false;
//				lastSyncWasToUnclosedBar = false;
				LastDataTime = DateTime.MinValue;
				primaryInstrumentCurrentBarTime = DateTime.MinValue;
				syncedInstrumentCurrentBarTime = DateTime.MinValue;
				primaryInstrumentCurrentBar = -1;
				syncedInstrumentCurrentBar = -1;
				
				switch (IndBase.BarsPeriods[PrimaryInstrument].Id)
				{
					case PeriodType.Day:
					case PeriodType.Week:
					case PeriodType.Month:
					case PeriodType.Year:
						dailyBars = true;
						break;
					case PeriodType.Minute:
					case PeriodType.Second:
					{
						dailyBars = false;
						break;
					}
					default:
						throw new ArgumentException("SyncedDataSeries: Instrument Period must be time-based (Minute,Day,Week,Month,Year,or Second).");
						break;
				}
				if (IndBase.BarsPeriods[PrimaryInstrument].Id != IndBase.BarsPeriods[SyncedInstrument].Id)
					throw new ArgumentException("SyncedDataSeries: Primary and Synced Intruments must both have the same Period Type.");
				if (IndBase.BarsPeriods[PrimaryInstrument].Value != IndBase.BarsPeriods[SyncedInstrument].Value)
					throw new ArgumentException("SyncedDataSeries: Primary and Synced Intruments must both have the same Period Value.");
				
				if (PrimaryInstrument == 0)
					Synced = new BoolSeries(IndBase);  // We are syncing to the primary instrument of the instantiating indicator
				else
					throw new ArgumentOutOfRangeException ("PrimaryInstrument", "SyncedDataSeries: PrimaryInstrument must = 0 if no SyncedIndicator base is given.");
			}
	
			public SyncedDataSeries (IndicatorBase indicator, int primaryInstrument, int syncedInstrument, IDataSeries dataArray, IndicatorBase syncedIndicator) :
				this (indicator, primaryInstrument, syncedInstrument, dataArray, syncedIndicator, indicator.MaximumBarsLookBack)
			{
			}
			public SyncedDataSeries (IndicatorBase indicator, int primaryInstrument, int syncedInstrument, IDataSeries dataArray, IndicatorBase syncedIndicator, MaximumBarsLookBack maxLookback) :base (syncedIndicator, maxLookback)
			{	
				IndBase = indicator;
				PrimaryInstrument = primaryInstrument;
				SyncedInstrument = syncedInstrument;
				SynchronizedBarCount = 0;
				SynchronizedBarCountToday = 0;
				DataArray = dataArray;
				DataStarted = false;
//				lastSyncWasToUnclosedBar = false;
				LastDataTime = DateTime.MinValue;
				primaryInstrumentCurrentBarTime = DateTime.MinValue;
				syncedInstrumentCurrentBarTime = DateTime.MinValue;
				primaryInstrumentCurrentBar = -1;
				syncedInstrumentCurrentBar = -1;
				
				switch (IndBase.BarsPeriods[PrimaryInstrument].Id)
				{
					case PeriodType.Day:
					case PeriodType.Week:
					case PeriodType.Month:
					case PeriodType.Year:
						dailyBars = true;
						break;
					case PeriodType.Minute:
					case PeriodType.Second:
					{
						dailyBars = false;
						break;
					}
					default:
						throw new ArgumentException("SyncedDataSeries: Instrument Period must be time-based (Minute,Day,Week,Month,Year,or Second).");
						break;
				}
				if (IndBase.BarsPeriods[PrimaryInstrument].Id != IndBase.BarsPeriods[SyncedInstrument].Id)
					throw new ArgumentException("SyncedDataSeries: Primary and Synced Intruments must both have the same Period Type.");
				if (IndBase.BarsPeriods[PrimaryInstrument].Value != IndBase.BarsPeriods[SyncedInstrument].Value)
					throw new ArgumentException("SyncedDataSeries: Primary and Synced Intruments must both have the same Period Value.");
				
				Synced = new BoolSeries(syncedIndicator);	// Not syncing to primary instrument of instantiating indicator;
															// So create a new BoolSeries synced to the master instrument
			}
			
			public double SyncedGet(int syncedBarsBack)
			{
				int i;
				if (syncedBarsBack >= SynchronizedBarCount)
					return 0;			// not enough synchronized bars - return 0 rather than throw exception
				return SyncedGet(syncedBarsBack, 0, out i);
			}
			
			public double SyncedGet(int syncedBarsBack, int initIndex, out int index)
			{
				index = initIndex;
				while (!Synced[index])  //Find most recent synced Bar
					index++;
				
				for (int counter=0; counter < Math.Min(syncedBarsBack,SynchronizedBarCount); counter++)
				{  //looking back past the first synced bar, keep looking for more
					index++;
					while (!Synced[index])  //Find previous synced Bar
						index++;
				}
				return base[index];
			}
			
			private void setBarAsSynced(int getDataFromBarsBack)
			{
				Synced.Set(true);
				SynchronizedBarCount++;
				SynchronizedBarCountToday++;
				LastDataTime = IndBase.Times[SyncedInstrument][getDataFromBarsBack];
				base.Set(DataArray[getDataFromBarsBack]);
			}
			
			public void Synchronize()
			{
				if ((IndBase.BarsInProgress != PrimaryInstrument) && (IndBase.BarsInProgress != SyncedInstrument))
						return;	// Bars being processed are not for us
				
				if (IndBase.CalculateOnBarClose == true)
				{
					if (IndBase.BarsInProgress == PrimaryInstrument) // Primary Instrument
					{
						primaryInstrumentCurrentBarTime = dailyBars ? IndBase.Times[PrimaryInstrument][0].Date : IndBase.Times[PrimaryInstrument][0];
						syncedInstrumentCurrentBar = IndBase.BarsArray[SyncedInstrument].CurrentBar;
						
						if (!DataStarted)
						{
							if (syncedInstrumentCurrentBar == -1)
							{
								Synced.Set(false);
								return;						// return if no data yet from synced instrument
							}
							else
								DataStarted = true;
						}
						if (IndBase.BarsArray[PrimaryInstrument].FirstBarOfSession)
							SynchronizedBarCountToday = 0;	// it's a new session, zero today's synced bar count
							
						//	Scan through SyncedInstrument's bars, try to find one in sync with the current Primary instrument bar
						//	Start the scan with the unclosed bar [-1] (necessary in real time); if it has the same time as the Primary instrument's Time[0] (which is now closed), it should be done forming
						int barsBack;
						if (syncedInstrumentCurrentBar == IndBase.BarsArray[SyncedInstrument].Count-1)
							barsBack = 0;
						else
							barsBack = -1;
						
						while ((barsBack < syncedInstrumentCurrentBar) && primaryInstrumentCurrentBarTime < (dailyBars ? IndBase.Times[SyncedInstrument][barsBack].Date : IndBase.Times[SyncedInstrument][barsBack]))
								barsBack ++;  // ignore bars that occur in the future (this can happen in real time if primary instrument is missing bars)
						if (primaryInstrumentCurrentBarTime == (dailyBars ? IndBase.Times[SyncedInstrument][barsBack].Date : IndBase.Times[SyncedInstrument][barsBack]))
							{	// Found a synchronized bar
								setBarAsSynced(barsBack);
//								if (barsBack == -1)
//									lastSyncWasToUnclosedBar = true;
							}
						else	// No synchronized bar found
							{
								Synced.Set(false);
								base.Set(DataArray[barsBack]);	// set output to most recent bar that isn't in the future
								LastDataTime = IndBase.Times[SyncedInstrument][barsBack];
							}
					}
					
					else // (IndBase.BarsInProgress == SyncedInstrument)
					{
						// Normally we don't have to do anything when (BarsInProgress == SyncedInstrument) if (CalculateOnBarClose == true)
						// Method output is synced with, and handled by, primary instrument when (BarsInProgress == PrimaryInstrument)
						//
						// Sometimes in a real-time OnBarUpdate for PrimaryIntrument, we find that SyncedInstrument has an unclosed bar (index of [-1]) with the same timestamp,
						// so we use it, which is safe to do because PrimaryInstrument's bar has already closed, so we know that the end of the bar period has passed.
						//
						if (!DataStarted)
						{
							if (IndBase.BarsArray[PrimaryInstrument].CurrentBar == -1)
								return;							// return if no data yet from Primary instrument
							else
								DataStarted = true;
						}	
//						if (lastSyncWasToUnclosedBar)
//						{
//							lastSyncWasToUnclosedBar = false;   // if there were any late arriving ticks, they are now included in the bar that just closed
//							int barsBack=0;
//							while (!Synced[barsBack])			// Scan for most recent synced bar
//								barsBack++;
//							base.Set(barsBack,DataArray[0]);	// Store the final value, in case it has changed since we stored the value from the unclosed bar
//						}
					}
				}
				else  // (CalculateOnBarClose == false)
				{
					if (IndBase.BarsInProgress == PrimaryInstrument) // Primary Instrument
					{
						if (!DataStarted)
						{	
							if (syncedInstrumentCurrentBar == -1)
							{
								Synced.Set(false);
								base.Reset();
								return;							// return if no data yet from synced instrument
							}
							else
								DataStarted = true;
						}		
						if (IndBase.FirstTickOfBar)				// First tick of bar, need to set state of sync flag
						{
							primaryInstrumentCurrentBarTime = dailyBars ? IndBase.Times[PrimaryInstrument][0].Date : IndBase.Times[PrimaryInstrument][0];
							primaryInstrumentCurrentBar = IndBase.BarsArray[PrimaryInstrument].CurrentBar;
							syncedInstrumentCurrentBar = IndBase.BarsArray[SyncedInstrument].CurrentBar;
							syncedInstrumentCurrentBarTime = dailyBars ? IndBase.Times[SyncedInstrument][0].Date : IndBase.Times[SyncedInstrument][0];

							if (IndBase.BarsArray[PrimaryInstrument].FirstBarOfSession)
								SynchronizedBarCountToday = 0;	// it's a new session, zero today's synced bar count
							
							//	When synchronizing daily bars if syncedInstrument has a later session end time than primaryInstrument
							//  if COBC == false, when processing historical data, OnBarUpdate will be called for primaryInstrument
							//  while the syncedInstrument shows the previous day in its current bar.  We have to peek ahead to
							//  the next bar (-1 index) to see if the date matches
							if (   (dailyBars == true)  
								&& (syncedInstrumentCurrentBar != IndBase.BarsArray[SyncedInstrument].Count-1) 
								&& (primaryInstrumentCurrentBarTime == IndBase.Times[SyncedInstrument][-1].Date))
							{
								setBarAsSynced(-1);
								return;
							}
							
							if (primaryInstrumentCurrentBarTime == syncedInstrumentCurrentBarTime)
								setBarAsSynced(0);
							else
							{
								if ((syncedInstrumentCurrentBar > 0) && (primaryInstrumentCurrentBarTime == (dailyBars ? IndBase.Times[SyncedInstrument][1].Date : IndBase.Times[SyncedInstrument][1])))
									setBarAsSynced(1);
								else
								{
									Synced.Set(false);
									base.Set(DataArray[0]);  // store most recent value from prior bars, in case no updates come for rest of bar
									LastDataTime = syncedInstrumentCurrentBarTime;
								}
							}
						}
					}
					else // (IndBase.BarsInProgress == SyncedInstrument)
					{
						if (syncedInstrumentCurrentBar != IndBase.BarsArray[SyncedInstrument].CurrentBar)
						{	// if SyncedInstrument's CurrentBar has changed, it's the first tick of the bar, need to save CurrentBar and Time
							syncedInstrumentCurrentBar = IndBase.BarsArray[SyncedInstrument].CurrentBar;
							syncedInstrumentCurrentBarTime = dailyBars ? IndBase.Times[SyncedInstrument][0].Date : IndBase.Times[SyncedInstrument][0];
						}
						
						if (!DataStarted)
						{
							if (IndBase.BarsArray[PrimaryInstrument].CurrentBar == -1)
								return;						// return if no data yet from primary instrument
							else
								DataStarted = true;
						}
//						if (primaryInstrumentCurrentBarTime == syncedInstrumentCurrentBarTime)
						if (((!dailyBars) && (IndBase.Times[SyncedInstrument][0] == IndBase.Times[PrimaryInstrument][0]))
							|| (dailyBars && (IndBase.ToDay(IndBase.Times[SyncedInstrument][0]) == IndBase.ToDay(IndBase.Times[PrimaryInstrument][0]))))
						{
							if (!Synced[0])
								setBarAsSynced(0);
							else
								base.Set(DataArray[0]);
						}
					}
				}
			}
		}
		#endregion
		
		#region MathematicalMethods	
		private double SyncedSMA(IDataSeries inputSeries, BoolSeries synced, int sMAPeriod)
			//	SyncedSMA: Calculate SMA of a Data Series, using only valid (synchonized) bars
			//
			//	Inputs:
			//		inputSeries: DataSeries on which to calculate SMA
			//		synced: BoolSeries, true for each bar which holds a valid (synchronized) value to use in SMA
			//		sMAPeriod: Period for SMA calculation; there must be >= sMAPeriod synced bars in the input series, or it will throw an exception
			//
		{
			double sMAsum = 0;
			int counter = 0;
			for (int barsBack = 0; counter < sMAPeriod; barsBack++)
			{
				if (barsBack > 255)  // NT now throws an exception, so this statement isn't strictly necessary; left it in to issue helpful error message
					if (MaximumBarsLookBack == MaximumBarsLookBack.TwoHundredFiftySix)
						throw new IndexOutOfRangeException("Index out of range: Try changing 'Maximum bars look back' parameter to 'Infinite'");
					
				if (synced[barsBack])
				{
					sMAsum += inputSeries[barsBack];
					counter++;
				}
			}
			return (sMAsum / System.Convert.ToDouble(sMAPeriod));
		}
		
		private double SyncedCorrelation(IDataSeries inputSeries1, SyncedDataSeries inputSeries2, int corrPeriod)
			//  SyncedCorrelation: Calculates the Pearson Product Moment Correlation of 2 Data Series, using only the bars which are synchronized
			//  Note: Correlations may differ from Excel if NinjaTrader is set to dividend-adjust historical prices
			//
			//	Inputs:
			//		inputSeries1: 1st DataSeries, typically Closes[x]
			//		inputSeries2: SyncedDataSeries, synced to InputSeries1; some bars (flagged by BoolSeries "Synced") hold valid (synchronized) values
			//		corrPeriod: Period for Correlation calculation; there must be >= CorrPeriod synced bars in the input series, or it will throw an exception
			//
		{
			double sum1 = 0; 
			double sum2 = 0;
			double sum3 = 0;
			int index1=0;
			double mean1, mean2, variance1, variance2, denominator;
			mean1 = SyncedSMA(inputSeries1,inputSeries2.Synced,corrPeriod);
			mean2 = SyncedSMA(inputSeries2,inputSeries2.Synced,corrPeriod);
			
			for (int counter = 0; counter < corrPeriod; counter++)
			{
				while (!inputSeries2.Synced[index1])  //Find next synced Bar
					index1++;
				variance1 = inputSeries1[index1] - mean1;
				variance2 = inputSeries2[index1] - mean2;
				sum1 += variance1 * variance2;
				sum2 += variance1 * variance1;
				sum3 += variance2 * variance2;
				index1++;
			}
			denominator = Math.Sqrt(sum2 * sum3);
			if (denominator == 0)
				return (0);  // avoid divide-by-zero error;  not really the correct result (actual correlation is undefined)
			else
				return (sum1 / denominator);
		}
		
		private double KDSMA(double[] inputSeries, int sMAPeriod)
		{   
			double sMAsum = 0;
			for (int index = 0; index < sMAPeriod; index++)
				sMAsum += inputSeries[index];
			return (sMAsum / System.Convert.ToDouble(sMAPeriod));
		}
		
		private double KDCorrelation(double[] inputSeries1, double[] inputSeries2, int corrPeriod)
			//  Note: Correlations may differ from Excel if NinjaTrader is set to dividend-adjust historical prices
		{
			double sum1 = 0; 
			double sum2 = 0;
			double sum3 = 0;
			double mean1, mean2, variance1, variance2, denominator;
			mean1 = KDSMA(inputSeries1,corrPeriod);
			mean2 = KDSMA(inputSeries2,corrPeriod);
			
			for (int index = 0; index < corrPeriod; index++)
			{
				variance1 = inputSeries1[index] - mean1;
				variance2 = inputSeries2[index] - mean2;
				sum1 += variance1 * variance2;
				sum2 += variance1 * variance1;
				sum3 += variance2 * variance2;
			}	
			denominator = Math.Sqrt(sum2 * sum3);
			if (denominator == 0)
				return (0);  // avoid divide-by-zero error;  not really the correct result (actual correlation is undefined)
			else
				return (sum1 / denominator);
		}
		
		
		private double SyncedPctChangeCorrelation(IDataSeries inputSeries1, SyncedDataSeries inputSeries2, int corrPeriod, int changeLookback)
		{
			int index=0;
			double[] pctChangeArray1 = new double[corrPeriod];
			double[] pctChangeArray2 = new double[corrPeriod];
			
			while (!inputSeries2.Synced[index])  //Find most recent synced Bar
				index++;
			double val1 = inputSeries1[index];
			double val2 = inputSeries2[index];
			index++;
			
			for (int counter=corrPeriod-1; counter >= 0; counter--)
			{
				int changeIndex;
				while (!inputSeries2.Synced[index])  //Find previous synced Bar
					index++;
				double chg2 = inputSeries2.SyncedGet(changeLookback-1,index,out changeIndex);  // this will scan back more synced bars if changeLookback > 0
				double chg1 = inputSeries1[changeIndex];
				pctChangeArray1[counter] = (val1 - chg1) / chg1;
				pctChangeArray2[counter] = (val2 - chg2) / chg2;
				val1 = inputSeries1[index];
				val2 = inputSeries2[index];
				index++;
			}		
			
//			return(KDRankCorrelation(pctChangeArray1,pctChangeArray2,corrPeriod));	// maybe rank correlation of PctChange would be better?
			return(KDCorrelation(pctChangeArray1,pctChangeArray2,corrPeriod));
		}
		
		private void KDResidual(double[] inputSeries, int slopePeriod, double[] residualArray)
		{	
			double meanx, variance1, variance2;
			double meany = KDSMA(inputSeries,slopePeriod);
			double sum1 = 0;
			double sum2 = 0;
			meanx = System.Convert.ToDouble(slopePeriod - 1) / 2;
			for (int index = 0; index < slopePeriod; index++)
					{
						variance1 = System.Convert.ToDouble(index) - meanx;
						sum1 += variance1 *(inputSeries[index] - meany);
						sum2 += variance1 * variance1;
					}
			double slope = sum1/sum2;
			double intercept = meany - (slope * meanx);
			for (int index = 0; index < slopePeriod; index ++)
				residualArray[index] = inputSeries[index] - (intercept + (slope * index));
		}
		
		private double SyncedResidualCorrelation(IDataSeries inputSeries1, SyncedDataSeries inputSeries2, int corrPeriod)
		//	Correlation of residuals (residual = error = difference between actual values and linear regression line)
		{
			int index1=0;
			double[] residualArray1 = new double[corrPeriod];
			double[] residualArray2 = new double[corrPeriod];
			double[] dataArray1 = new double[corrPeriod];
			double[] dataArray2 = new double[corrPeriod];
			for (int counter=corrPeriod-1; counter >= 0; counter--)
			{
				while (!inputSeries2.Synced[index1])  //Find next synced Bar
					index1++;
				dataArray1[counter] = inputSeries1[index1];
				dataArray2[counter] = inputSeries2[index1];
				index1++;
			}
			KDResidual(dataArray1,corrPeriod,residualArray1);
			KDResidual(dataArray2,corrPeriod,residualArray2);
//			return(KDRankCorrelation(residualArray1,residualArray2,corrPeriod));	// maybe rank correlation of residuals would be better?
			return(KDCorrelation(residualArray1,residualArray2,corrPeriod));
		}
		
		
		private void RankSort(double[] dataArray, double[] indexArray, int period)
		{
			int index1;
			int[] indexArray2 = new int[period];
			for (index1=0; index1<period; index1++)
				indexArray2[index1] = index1;
			Array.Sort(dataArray,indexArray2);
		
			for (index1=0; index1<period; index1++)
			{
				int index2 = index1+1;
				while ((index2 < period) && (dataArray[index1] == dataArray[Math.Min(index2,period-1)]))
					index2 = index2+1;  //look for series of identical values
				double avgIndex = System.Convert.ToDouble(index1 + index2 + 1) / 2; // Average & convert to 1 origin
				for (int index3 = index2-1; index3 >= index1; index3--)
					indexArray[indexArray2[index3]] = avgIndex;
				index1 = index2-1;
			}
		}
		
		private double KDRankCorrelation(double[] dataArray1, double[] dataArray2, int corrPeriod)
			//  KDRankCorrelation: Calculate the Spearman Rank Correlation of 2 Data Arrays
		{
			double[] indexArray1 = new double[corrPeriod];
			double[] indexArray2 = new double[corrPeriod];
			RankSort(dataArray1,indexArray1,corrPeriod);
			RankSort(dataArray2,indexArray2,corrPeriod);
			return(KDCorrelation(indexArray1,indexArray2,corrPeriod));
		}
		
		private double SyncedRankCorrelation(IDataSeries inputSeries1, SyncedDataSeries inputSeries2, int corrPeriod)
			//  SyncedRankCorrelation: Calculate the Spearman Rank Correlation of 2 Data Series, using only the bars which are synchronized
			//
			//	Inputs:
			//		inputSeries1: 1st DataSeries, typically Closes[x]
			//		inputSeries2: SyncedDataSeries, synced to InputSeries1; some bars (flagged by BoolSeries "Synced") hold valid (synchronized) values
			//		corrPeriod: Period for Correlation calculation; there must be >= CorrPeriod synced bars in the input series, or it will throw an exception
			//
		{
			int index1=0;
			double[] dataArray1 = new double[corrPeriod];
			double[] dataArray2 = new double[corrPeriod];
			for (int counter=corrPeriod-1; counter >= 0; counter--)
			{
				while (!inputSeries2.Synced[index1])  //Find next synced Bar
					index1++;
				dataArray1[counter] = inputSeries1[index1];
				dataArray2[counter] = inputSeries2[index1];
				index1++;
			}
			return(KDRankCorrelation(dataArray1,dataArray2,corrPeriod));
		}
		
		#endregion
		
        protected override void Initialize()
        {
            CalculateOnBarClose	= true;							// Default to true - don't set to false!
			MaximumBarsLookBack = MaximumBarsLookBack.Infinite; // Default to infinite lookback to avoid throwing exceptions
			BarsRequired = 0;
			Add(new Plot(Color.FromKnownColor(KnownColor.CornflowerBlue), PlotStyle.Line, Name));
			switch (BarsPeriod.Id)
			{
				case PeriodType.Day:
				case PeriodType.Week:
				case PeriodType.Month:
				case PeriodType.Year:
				case PeriodType.Minute:
				case PeriodType.Second:
					Add(Symbol2, BarsPeriod.Id, BarsPeriod.Value);
					supportedBarsPeriodType = true;
					break;
				default:
					break;
			}
		}

		protected override void OnStartUp()
		{
			if (!supportedBarsPeriodType)
				throw new ArgumentException("Input series Period must be time-based (Minute,Day,Week,Month,Year,or Second).");
			
			Instrument2Close = new SyncedDataSeries(this, Instrument1, Instrument2, Closes[Instrument2]);
		}
		
        protected override void OnBarUpdate()
        {
			Instrument2Close.Synchronize();
				
			if (BarsInProgress == Instrument1) // Instrument 1
			{
				if (Instrument2Close.SynchronizedBarCount > (CorrType == CorrelationType.PctChange ? Period + PeriodPctChange : Period))	// No output if not enough synchronized bars for calculation
					if (Instrument2Close.Synced[0])  // We have a new synchronized pair of closes; do the correlation calculation
						switch (CorrType)
						{
							case CorrelationType.Pearson:
								Value.Set(SyncedCorrelation(Closes[Instrument1],Instrument2Close,Period)); 
								break;
							case CorrelationType.Rank:
								Value.Set(SyncedRankCorrelation(Closes[Instrument1],Instrument2Close,Period));
								break;
							case CorrelationType.Residual:
								Value.Set(SyncedResidualCorrelation(Closes[Instrument1],Instrument2Close,Period));
								break;
							case CorrelationType.PctChange:
								Value.Set(SyncedPctChangeCorrelation(Closes[Instrument1],Instrument2Close,Period,PeriodPctChange));
								break;
						}
					else
						if (Value.ContainsValue(1))
							Value.Set(Value[1]);	// Current bar is not synchronized; Use the last correlation again as the current one
													// First time through is guaranteed to be synced; if we get here Value[1] is always defined
			}
        }
		
		public override string ToString()
		{
			// Pretty up the chart label
			return Name + "(" + Instruments[Instrument1].FullName + "," + ((Instruments.Length <= Instrument2) ? "" : Instruments[Instrument2].FullName) + "," + corrType.ToString() + "," + Period.ToString() + "," + PeriodPctChange.ToString() + ")";
		}

        #region Properties

		[Description("Symbol 2; i.e. SPY or ES 03-10\nDefault = Secondary chart instrument")]
        [GridCategory("Parameters")]
		// Force this parameter to be displayed first
		[Gui.Design.DisplayName ("\tSymbol 2")]
        public string Symbol2
        {
            get 
			{ 
				if ((symbol2 == "") && (ChartControl != null) && (Instruments != null)) 
					for(int i=0; i<ChartControl.BarsArray.Length; i++)
						if (ChartControl.BarsArray[i].Instrument.FullName != Instruments[0].FullName)
						{
							symbol2 = ChartControl.BarsArray[i].Instrument.FullName;
							break;
						}
				return symbol2; 
			}
            set { symbol2 = value.ToUpper(); }
		}
      
		[Description("Correlation Lookback Period")]
        [GridCategory("Parameters")]
		public int Period
        {
            get { return period; }
            set { period = Math.Max(1, value); }
        }
		
//		[Description("Pearson(Standard), Rank(Spearman),\nResidual(Rank), Percent Change(Rank)")]
		[Description("Pearson(Standard), Rank(Spearman),\nResidual, Percent Change")]
        [GridCategory("Parameters")]
		[Gui.Design.DisplayName("Correlation Type")]
		public CorrelationType CorrType
        {
            get { return corrType; }
            set 
			{ 
				corrType = value; 
				if (Plots.Length > 0)
					switch (corrType)
					{
						case CorrelationType.Pearson:
							Plots[0].Pen.Color = Color.CornflowerBlue;
							break;
						case CorrelationType.Rank:
							Plots[0].Pen.Color = Color.Red;
							break;
						case CorrelationType.Residual:
							Plots[0].Pen.Color = Color.ForestGreen;
							break;
						case CorrelationType.PctChange:
							Plots[0].Pen.Color = Color.Purple;
							break;
					}
			}
        }
		
		[Description("Lookback Bars for PctChange")]
        [GridCategory("Parameters")]
		public int PeriodPctChange
        {
            get { return periodPctChange; }
            set { periodPctChange = Math.Max(1, value); }
        }
		
		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Default
		{
			get { return Values[0]; }
		}
		
        #endregion
    }
}

public enum CorrelationType
{
	Pearson,
	Rank,
	Residual,
	PctChange
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private Correlation[] cacheCorrelation = null;

        private static Correlation checkCorrelation = new Correlation();

        /// <summary>
        /// Correlation of Two Instruments
        /// </summary>
        /// <returns></returns>
        public Correlation Correlation(CorrelationType corrType, int period, int periodPctChange, string symbol2)
        {
            return Correlation(Input, corrType, period, periodPctChange, symbol2);
        }

        /// <summary>
        /// Correlation of Two Instruments
        /// </summary>
        /// <returns></returns>
        public Correlation Correlation(Data.IDataSeries input, CorrelationType corrType, int period, int periodPctChange, string symbol2)
        {
            if (cacheCorrelation != null)
                for (int idx = 0; idx < cacheCorrelation.Length; idx++)
                    if (cacheCorrelation[idx].CorrType == corrType && cacheCorrelation[idx].Period == period && cacheCorrelation[idx].PeriodPctChange == periodPctChange && cacheCorrelation[idx].Symbol2 == symbol2 && cacheCorrelation[idx].EqualsInput(input))
                        return cacheCorrelation[idx];

            lock (checkCorrelation)
            {
                checkCorrelation.CorrType = corrType;
                corrType = checkCorrelation.CorrType;
                checkCorrelation.Period = period;
                period = checkCorrelation.Period;
                checkCorrelation.PeriodPctChange = periodPctChange;
                periodPctChange = checkCorrelation.PeriodPctChange;
                checkCorrelation.Symbol2 = symbol2;
                symbol2 = checkCorrelation.Symbol2;

                if (cacheCorrelation != null)
                    for (int idx = 0; idx < cacheCorrelation.Length; idx++)
                        if (cacheCorrelation[idx].CorrType == corrType && cacheCorrelation[idx].Period == period && cacheCorrelation[idx].PeriodPctChange == periodPctChange && cacheCorrelation[idx].Symbol2 == symbol2 && cacheCorrelation[idx].EqualsInput(input))
                            return cacheCorrelation[idx];

                Correlation indicator = new Correlation();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.CorrType = corrType;
                indicator.Period = period;
                indicator.PeriodPctChange = periodPctChange;
                indicator.Symbol2 = symbol2;
                Indicators.Add(indicator);
                indicator.SetUp();

                Correlation[] tmp = new Correlation[cacheCorrelation == null ? 1 : cacheCorrelation.Length + 1];
                if (cacheCorrelation != null)
                    cacheCorrelation.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheCorrelation = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Correlation of Two Instruments
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.Correlation Correlation(CorrelationType corrType, int period, int periodPctChange, string symbol2)
        {
            return _indicator.Correlation(Input, corrType, period, periodPctChange, symbol2);
        }

        /// <summary>
        /// Correlation of Two Instruments
        /// </summary>
        /// <returns></returns>
        public Indicator.Correlation Correlation(Data.IDataSeries input, CorrelationType corrType, int period, int periodPctChange, string symbol2)
        {
            return _indicator.Correlation(input, corrType, period, periodPctChange, symbol2);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Correlation of Two Instruments
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.Correlation Correlation(CorrelationType corrType, int period, int periodPctChange, string symbol2)
        {
            return _indicator.Correlation(Input, corrType, period, periodPctChange, symbol2);
        }

        /// <summary>
        /// Correlation of Two Instruments
        /// </summary>
        /// <returns></returns>
        public Indicator.Correlation Correlation(Data.IDataSeries input, CorrelationType corrType, int period, int periodPctChange, string symbol2)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.Correlation(input, corrType, period, periodPctChange, symbol2);
        }
    }
}
#endregion
