/// <summary>
/// TSSuperTrendV2 Indicator
/// Version 2.2
/// Visual options for bar colors added by Elliott Wave 12/08.
/// Version 2.3 
/// Fixed VWMA issue
/// Added Sound Alerts
/// http://www.ninjatrader.com/support/forum/showthread.php?t=11844&page=8   line 162 etc
/// </summary>
using System;
using System.ComponentModel;
using System.Drawing;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
using TSSuperTrendV2.Utility;

namespace NinjaTrader.Indicator
{
    [Description("TSSuperTrendV2 Indicator developed by TradingStudies.com (Vertsion 2.3)")]
    public class TSSuperTrendV2 : Indicator
    {
        private int _length = 8;
        private double _multiplier = 2.618;
        private IDataSeries _avg;
        private double _offset;
        private MovingAverageType _maType = MovingAverageType.TEMA;
        private SuperTrendMode _smode = SuperTrendMode.Adaptive;
        private int _smooth = 8;
        private Color _barColorUp = Color.Blue;
        private Color _barColorDown = Color.Red;
        private Color _tempColor;
        private Color _prevColor;
        private double _th;
        private double _tl = double.MaxValue;
        private bool _playAlert = false;
        private string _longAlert = string.Empty;
        private string _shortAlert = string.Empty;
        private int _thisbar = -1;

        private DataSeries eVal;
		private DataSeries eTrend;        
		
        protected override void Initialize()
        {
            Add(new Plot(Color.Green, PlotStyle.Square, "UpTrend"));
            Add(new Plot(Color.Red, PlotStyle.Square, "DownTrend"));
			Plots[0].Pen.Width = 2;
			Plots[1].Pen.Width = 2;
            Overlay = true;

            eVal = new DataSeries(this);
			eTrend = new DataSeries(this);            
        }

        protected override void OnBarUpdate()
        {
            if (CurrentBar < 1)
            {
                if (_smooth > 1 && _avg == null)
                    switch (_maType)
                    {
                        case MovingAverageType.SMA:
                            _avg = SMA(Input, _smooth);
                            break;
                        case MovingAverageType.SMMA:
                            _avg = SMMA(Input, _smooth);
                            break;
                        case MovingAverageType.TMA:
                            _avg = TMA(Input, _smooth);
                            break;
                        case MovingAverageType.WMA:
                            _avg = WMA(Input, _smooth);
                            break;
                        case MovingAverageType.VWMA:
                            _avg = VWMA(Input, _smooth);
                            break;
                        case MovingAverageType.TEMA:
                            _avg = TEMA(Input, _smooth);
                            break;
                        case MovingAverageType.HMA:
                            _avg = HMA(Input, _smooth);
                            break;
                        case MovingAverageType.VMA:
                            _avg = VMA(Input, _smooth, _smooth);
                            break;
                        default:
                            _avg = EMA(Input, _smooth);
                            break;
                    }
                else
                    _avg = Input;

                eVal.Set(0);
                eTrend.Set(0);
                UpTrend.Set(Input[0]);
                DownTrend.Set(Input[0]);
                return;
            }

            switch (_smode)
            {
                case SuperTrendMode.ATR:
                    _offset = ATR(_length)[0] * Multiplier;
                    break;
                case SuperTrendMode.Adaptive:
                    _offset = ATR(_length)[0] * HomodyneDiscriminator(Input)[0] / 10;
                    break;
                default:
                    _offset = Dtt(_length, Multiplier);
                    break;
            }

            if (FirstTickOfBar)
                _prevColor = _tempColor;

            eTrend.Set(Close[0] > DownTrend[1] ? 1 : Close[0] < UpTrend[1] ? -1 : eTrend[1]);

            if (eTrend[0] == 1 && eTrend[1] == -1)
            {
                _th = High[0];
                UpTrend.Set(Math.Max(_avg[0] - _offset, _tl));
                eVal.Set(UpTrend[0]);
                if (Plots[0].PlotStyle == PlotStyle.Line) UpTrend.Set(1, DownTrend[1]);
                _tempColor = _barColorUp;
            }
            else if (eTrend[0] == -1 && eTrend[1] == 1)
            {
                _tl = Low[0];
                DownTrend.Set(Math.Min(_avg[0] + _offset, _th));
                eVal.Set(DownTrend[0]);
                if (Plots[1].PlotStyle == PlotStyle.Line) DownTrend.Set(1, UpTrend[1]);
                _tempColor = _barColorDown;

            }
            else
            {
                if (eTrend[0] == 1)
                {
                    UpTrend.Set((_avg[0] - _offset) > UpTrend[1] ? (_avg[0] - _offset) : UpTrend[1]);
                    eVal.Set(UpTrend[0]);
                    _th = Math.Max(_th, High[0]);
                }
                else
                {
                    DownTrend.Set((_avg[0] + _offset) < DownTrend[1] ? (_avg[0] + _offset) : DownTrend[1]);
                    eVal.Set(DownTrend[0]);
                    _tl = Math.Min(_tl, Low[0]);
                }
                RemoveDrawObject(CurrentBar.ToString());
                _tempColor = _prevColor;
            }
			
			//DrawText(CurrentBar.ToString(), eTrend[0].ToString(), 0, Low[0], Color.White);
        }

        private double Dtt(int nDay, double mult)
        {
            double hh = MAX(High, nDay)[0];
            double hc = MAX(Close, nDay)[0];
            double ll = MIN(Low, nDay)[0];
            double lc = MIN(Close, nDay)[0];
            return mult * Math.Max((hh - lc), (hc - ll));
        }

        #region Properties
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries UpTrend
        {
            get
            {
                Update();
                return Values[0];
            }
        }

        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries DownTrend
        {
            get
            {
                Update();
                return Values[1];
            }
        }

        [Description("SuperTrendMode")]
        [Category("Parameters")]
        [Gui.Design.DisplayName("01. SuperTrend Mode")]
        public SuperTrendMode StMode
        {
            get { return _smode; }
            set { _smode = value; }
        }

        [Description("ATR/DT Period")]
        [Category("Parameters")]
        [Gui.Design.DisplayName("02. Period")]
        public int Length
        {
            get { return _length; }
            set { _length = Math.Max(1, value); }
        }

        [Description("ATR Multiplier")]
        [Category("Parameters")]
        [Gui.Design.DisplayName("03. Multiplier")]
        public double Multiplier
        {
            get { return _multiplier; }
            set { _multiplier = Math.Max(0.0001, value); }
        }

        [Description("Moving Average Type for smoothing")]
        [Category("Parameters")]
        [Gui.Design.DisplayName("04. Moving Average Type")]
        public MovingAverageType MaType
        {
            get { return _maType; }
            set { _maType = value; }
        }

        [Description("Smoothing Period")]
        [Category("Parameters")]
        [Gui.Design.DisplayName("05. SmoothingPeriod (MA)")]
        public int Smooth
        {
            get { return _smooth; }
            set { _smooth = Math.Max(1, value); }
        }

		[Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries Exp_Trend
        {
            get { Update(); return eTrend; }
        }
        [Browsable(false)]	// this line prevents the data series from being displayed in the indicator properties dialog, do not remove
        [XmlIgnore()]		// this line ensures that the indicator can be saved/recovered as part of a chart template, do not remove
        public DataSeries Exp_Val
        {
            get { Update(); return eVal; }
        }

        #endregion
    }
}

namespace TSSuperTrendV2.Utility
{
    public enum SuperTrendMode
    {
        ATR,
        DualThrust,
        Adaptive
    }

    public enum MovingAverageType
    {
        SMA,
        SMMA,
        TMA,
        WMA,
        VWMA,
        TEMA,
        HMA,
        EMA,
        VMA
    }
}
#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private TSSuperTrendV2[] cacheTSSuperTrendV2 = null;

        private static TSSuperTrendV2 checkTSSuperTrendV2 = new TSSuperTrendV2();

        /// <summary>
        /// TSSuperTrendV2 Indicator developed by TradingStudies.com (Vertsion 2.3)
        /// </summary>
        /// <returns></returns>
        public TSSuperTrendV2 TSSuperTrendV2(int length, MovingAverageType maType, double multiplier, int smooth, SuperTrendMode stMode)
        {
            return TSSuperTrendV2(Input, length, maType, multiplier, smooth, stMode);
        }

        /// <summary>
        /// TSSuperTrendV2 Indicator developed by TradingStudies.com (Vertsion 2.3)
        /// </summary>
        /// <returns></returns>
        public TSSuperTrendV2 TSSuperTrendV2(Data.IDataSeries input, int length, MovingAverageType maType, double multiplier, int smooth, SuperTrendMode stMode)
        {
            if (cacheTSSuperTrendV2 != null)
                for (int idx = 0; idx < cacheTSSuperTrendV2.Length; idx++)
                    if (cacheTSSuperTrendV2[idx].Length == length && cacheTSSuperTrendV2[idx].MaType == maType && Math.Abs(cacheTSSuperTrendV2[idx].Multiplier - multiplier) <= double.Epsilon && cacheTSSuperTrendV2[idx].Smooth == smooth && cacheTSSuperTrendV2[idx].StMode == stMode && cacheTSSuperTrendV2[idx].EqualsInput(input))
                        return cacheTSSuperTrendV2[idx];

            lock (checkTSSuperTrendV2)
            {
                checkTSSuperTrendV2.Length = length;
                length = checkTSSuperTrendV2.Length;
                checkTSSuperTrendV2.MaType = maType;
                maType = checkTSSuperTrendV2.MaType;
                checkTSSuperTrendV2.Multiplier = multiplier;
                multiplier = checkTSSuperTrendV2.Multiplier;
                checkTSSuperTrendV2.Smooth = smooth;
                smooth = checkTSSuperTrendV2.Smooth;
                checkTSSuperTrendV2.StMode = stMode;
                stMode = checkTSSuperTrendV2.StMode;

                if (cacheTSSuperTrendV2 != null)
                    for (int idx = 0; idx < cacheTSSuperTrendV2.Length; idx++)
                        if (cacheTSSuperTrendV2[idx].Length == length && cacheTSSuperTrendV2[idx].MaType == maType && Math.Abs(cacheTSSuperTrendV2[idx].Multiplier - multiplier) <= double.Epsilon && cacheTSSuperTrendV2[idx].Smooth == smooth && cacheTSSuperTrendV2[idx].StMode == stMode && cacheTSSuperTrendV2[idx].EqualsInput(input))
                            return cacheTSSuperTrendV2[idx];

                TSSuperTrendV2 indicator = new TSSuperTrendV2();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.Length = length;
                indicator.MaType = maType;
                indicator.Multiplier = multiplier;
                indicator.Smooth = smooth;
                indicator.StMode = stMode;
                Indicators.Add(indicator);
                indicator.SetUp();

                TSSuperTrendV2[] tmp = new TSSuperTrendV2[cacheTSSuperTrendV2 == null ? 1 : cacheTSSuperTrendV2.Length + 1];
                if (cacheTSSuperTrendV2 != null)
                    cacheTSSuperTrendV2.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheTSSuperTrendV2 = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// TSSuperTrendV2 Indicator developed by TradingStudies.com (Vertsion 2.3)
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.TSSuperTrendV2 TSSuperTrendV2(int length, MovingAverageType maType, double multiplier, int smooth, SuperTrendMode stMode)
        {
            return _indicator.TSSuperTrendV2(Input, length, maType, multiplier, smooth, stMode);
        }

        /// <summary>
        /// TSSuperTrendV2 Indicator developed by TradingStudies.com (Vertsion 2.3)
        /// </summary>
        /// <returns></returns>
        public Indicator.TSSuperTrendV2 TSSuperTrendV2(Data.IDataSeries input, int length, MovingAverageType maType, double multiplier, int smooth, SuperTrendMode stMode)
        {
            return _indicator.TSSuperTrendV2(input, length, maType, multiplier, smooth, stMode);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// TSSuperTrendV2 Indicator developed by TradingStudies.com (Vertsion 2.3)
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.TSSuperTrendV2 TSSuperTrendV2(int length, MovingAverageType maType, double multiplier, int smooth, SuperTrendMode stMode)
        {
            return _indicator.TSSuperTrendV2(Input, length, maType, multiplier, smooth, stMode);
        }

        /// <summary>
        /// TSSuperTrendV2 Indicator developed by TradingStudies.com (Vertsion 2.3)
        /// </summary>
        /// <returns></returns>
        public Indicator.TSSuperTrendV2 TSSuperTrendV2(Data.IDataSeries input, int length, MovingAverageType maType, double multiplier, int smooth, SuperTrendMode stMode)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.TSSuperTrendV2(input, length, maType, multiplier, smooth, stMode);
        }
    }
}
#endregion
