#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
using System.Collections;

#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    ///Author:  Fred Walton
	///Date:  July 2009
	///Title:  Price Volume Histogram
	///
	///This Software Source Code ("Price Volume Histogram") is provided under the following License Agreement (the "Agreement") 
	///between you, the end-user ("Licensee"), and author ("Author"). By using this software or storing this program on a computer 
	///hard drive or other media, you are agreeing to be bound by the terms of this Agreement.
	///Author grants a license to download the Price Volume Histogram software from the NinjaTrader® website and to use the 
	///Price Volume Histogram within the NinjaTrader® software platform to any NinjaTrader® registered forum user.
	///You may not have anyone other than yourself alter this software in any way.
	///You may not rent or resell this software or any portion thereof.
	///You may not publicize or distribute any registration code algorithms, information, or registration codes used by this software 
	///without permission of author.
	///This software is provided "as is". Author makes no warranty, expressed or implied, with regard to the software. All implied warranties, including the warranties of merchantability and fitness for a particular use, are hereby excluded.  Under no circumstances shall author of this product be liable for any incidental or consequential damages.
	///
	///Notes:
	///
	///Description:
	///- A graphical representation of trading volumes per price tick.
	///- The indicator was developed to provide an interactive, customizable, big-picture representation of volume distribution across a price range. 
	///- Works on both Historical and Real time data in any period
	///History:
	///  - The indicator was developed to provide an interactive, customizable, big-picture representation of volume distribution across a price range. 
	///  - I’ve recently become intrigued with volume - based trading, including attending Frank Butera’s Balance Trader webinar (www.balancetrader.com) and others.  
	///  - As part of my discovery, I tried the ‘TPO and Volume profile Chart for NinjaTrader from Final software (http://www.fin-alg.com/), but found it required more UI interaction that I wanted 
	///Versions:
	///- V1.3 Jan 29 2010
	///- corrrected HistogramPlotType not saving issue 
	///- added Source Code License Agreement
	///- V1.2 Dec 12 2009
	///- added readonly Version parameter
	///- corrrected colors not saving issue 
	///- V1.1 Dec 7 2009
	///- added VolumeSource parameter to better represent the traded values for dataseries with a large price range 
	///- added PlotWidth parameter to control max width of plot
	///- V1.0 Jul 2009
	///- Initial release
	///
	///Plotting:
	///- All volume tick bars less than the (ThresholdPercent) value are drawn in the (SubThresholdColor).
	///- All volume tick bars from the (ThresholdPercent) value to the maximum value are drawn in a gradient color, ranging from (ThresholdColor) to (MaximumColor)
	///- The Plot is drawn with a maximum width relative to the screen width.
	///  Note: Sometimes the longest bar on the screen is shorter than the specified plot width. This occurs when you are zoomed in on a chart and the max volume is at a price not shown on the chart.
	///
	///Data:
	///- Either all the data in the chart’s dataseries or just the displayed data can be used as the calculation source
	///- The volume data can either be Share Volume or Monetary Value which gives a more acccurate histogram for instruments with a large price variance.
	///
	///Known Issues:
	///- On some occasions when scrolling the previous plot is ghosted with the current plot.
	///Cause: Unknown. Fix: Refresh chart (Press F5)
	///
	///  Testing:
	///  - It is primarily tested with stocks, but seems to work fine on some currencies provided by Gain’s ‘Test’ Forex connection
	///  - On some occasions when scrolling the previous plot is ghosted with the current plot.
	///  - Cause: Unknown. Fix: Refresh chart (Press F5)
	///
	///  Tips:
	///  - Try setting (PlotStyle) to ‘RightJustified’ to concentrate the graphics on the right edge of the chart by the real time bar
	///  - Try setting (PlotStyle) to ‘VerticalGradient’, the (ThresholdPercent) to 80, and the (SubThresholdColor) to ‘Transparent’ to identify support and resistance regions
	///  - Try setting the (DataType) to ‘VisibleOnly’, and scrolling the chart horizontally to view changes over time
	///
	///Disclaimer:
	///No guarantee, either expressed or implied, accompanies this software’s function or suitability for a purpose.  
	///(Do not plan a trip to the moon or an escape from a burning building based on this indicator )
    /// </summary>
    [Description("Volumes Per Tick displayed as histogram variants.")]
    public class PriceVolumeHistogram : Indicator
    {
        #region Variables
        // Wizard generated variables
        // User defined variables (add any user defined variables below)
		private Hashtable _priceRangeVolumes;
		private int _maxVolume;
		private int _maxBarsAgo;
		private int _minBarsAgo;
		private double _minPrice = double.MaxValue;
		private double _maxPrice = 0;
		private bool _isFullUpdate;
		private int _ticksPerPlotRange;

		#endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
			try
			{
				//Print(String.Format("Initialize() ", new object[] { } ));
				CalculateOnBarClose	= false;
				Overlay	= true;
				PriceTypeSupported	= false;
				BarsRequired = 1;
			}
			catch (Exception ex)
			{
				Log(String.Format("Initialize()  Error:{0}", new object[] {ex.Message} ), LogLevel.Error);
			}
        }

		public override void Plot(Graphics graphics, Rectangle bounds, double min, double max)
		{
			//Print(String.Format("Plot() FirstVisibleBar:{0} LastVisibleBar:{1} ",  new object[] { FirstVisibleBar, LastVisibleBar } ));
			RefreshUI(true);
		}
		
		/// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
			try
			{
				//Print(String.Format("OnBarUpdate() CurrentBar:{0} Historical:{1} Bars.Count:{2} Volume:{3} High:{4} Low:{5} ",  new object[] { CurrentBar, Historical, Bars.Count, Volume[0], High[0], Low[0] } ));

				if (CurrentBar == 0)
				{
					_isFullUpdate = true;
				}

				if (CurrentBar == Bars.Count - 1 && !Historical)
				{
					RefreshUI(_isFullUpdate);
					_isFullUpdate = false;
				}

			}
			catch (Exception ex)
			{
				Log(String.Format("OnBarUpdate() Error: {0} Instrument:'{1}' CurrentBar:{2} Date:'{3}' Bars.Count:{4}", new object[] {ex.Message, Instrument.MasterInstrument.Name, CurrentBar, Time[0], Bars.Count}), LogLevel.Error);
			}
        }

		private void RefreshUI(bool isFullUpdate)
		{
			try
			{
				//Print(String.Format("RefreshUI() isFullUpdate:{0}",  new object[] { isFullUpdate } ));

				if (isFullUpdate)
				{
					RefreshPlotSettings();
					
					//process all the data except the last bar
					for (int bar = 1; bar < Bars.Count; bar++)
					{
						if (_maxBarsAgo >= bar && bar >= _minBarsAgo)
						{
							//Print(String.Format("RefreshUI() bar:{0} Volume:{1} High:{2} Low:{3} ",  new object[] { bar, Volume[bar], High[bar], Low[bar] } ));
							AdjustPriceRangeVolumeCache((int)Volume[bar], High[bar], Low[bar], false);
						}
					}
				}
				//Print(String.Format("RefreshUI() Bars.Count:{0} FirstVisibleBar:{1} LastVisibleBar:{2} _maxBarsAgo:{3} _minBarsAgo:{4}", new object[] { Bars.Count, FirstVisibleBar, LastVisibleBar, _maxBarsAgo, _minBarsAgo } ));

				//process the last bar's realtime data
				//temporarily add it to the cache
				if (_minBarsAgo == 0)
					AdjustPriceRangeVolumeCache((int)Volume[0], High[0], Low[0], false);
				//plot with the latest data
				PlotPriceRangeVolumes();
				//remove it from the cache in preparation for the next update
				if (_minBarsAgo == 0)
					AdjustPriceRangeVolumeCache((int)Volume[0], High[0], Low[0], true);

			}
			catch (Exception ex)
			{
				Log(String.Format("OnBarUpdate() Error: {0} Instrument:'{1}' CurrentBar:{2} Date:'{3}' Bars.Count:{4}", new object[] {ex.Message, Instrument.MasterInstrument.Name, CurrentBar, Time[0], Bars.Count}), LogLevel.Error);
			}
		}
		
		private void AdjustPriceRangeVolumeCache(int volume, double high, double low, bool decrement)
		{
			int volumeSum;
			int volumePerPlotRange;
			int priceRangeCount;
			int firstPriceRange;
			string key;
			double averagePrice;
			int adjVolume;
			
			if (_volumeSource == VolumeSourceEnum.MonetaryValue)
			{
				//adjust volume for relative price
				averagePrice = (high + low) / 2;
				volume = (int)(volume *_maxPrice / averagePrice);
			}
			
			//calculate auxiliary values to assist in updating cache
			firstPriceRange = (int) Math.Round(low / TickSize);
			priceRangeCount = (int) Math.Round(high / TickSize - firstPriceRange) + 1;
			volumePerPlotRange = (decrement ?-1 :1) * (int) (Math.Round(1.0 * volume / priceRangeCount));
			
			//ensure the first price range is an exact multiple of _ticksPerPlotRange
			firstPriceRange = Math.Min(firstPriceRange, firstPriceRange - firstPriceRange % _ticksPerPlotRange);

			//Print(String.Format("AdjustPriceRangeVolumeCache() firstPriceRange:{0} priceRangeCount:{1} volumePerPlotRange:{2}", new object[] { firstPriceRange, priceRangeCount, volumePerPlotRange } ));

			//update affected cache values
			for (int i = firstPriceRange; i < firstPriceRange + priceRangeCount; i += _ticksPerPlotRange)
			{
				key = i.ToString();
				if (_priceRangeVolumes[key] == null)
				{
					_priceRangeVolumes.Add(key, 0);
					//Print(String.Format("AdjustPriceRangeVolumeCache() Added:{0} ", new object[] { key } ));
				}
				volumeSum = (int)_priceRangeVolumes[key] + volumePerPlotRange;
				_priceRangeVolumes[key] = volumeSum;
				//update max volume if necessary
				_maxVolume = volumeSum > _maxVolume ?volumeSum :_maxVolume;
				//Print(String.Format("AdjustPriceRangeVolumeCache() key:{0} volumeSum:{1} _maxVolume:{2}", new object[] { key, volumeSum, _maxVolume } ));
			}
		}
		
		private void PlotPriceRangeVolumes()
		{
			int maxWidth;
			int width;
			int leftBarsAgo;
			int rightBarsAgo;
			int maxBarsAgo;
			int minBarsAgo;
			double volumePercent;
			double yBase;
			Color color;
			double gradient;
			
			leftBarsAgo = 0;
			rightBarsAgo = 0;
			gradient = 0.0;
			color = Color.Black;
			
			//Sometimes on the first load the colors are not initialized. Bug? 
			if (_maximumColor == Color.Empty)
			{
				_maximumColor = Color.Yellow; 
				_thresholdColor = Color.Gray;
				_subThresholdColor = Color.DimGray;
			}

			maxBarsAgo = FirstVisibleBar;
			minBarsAgo = Math.Max(LastVisibleBar, 0);
			
			maxWidth = maxBarsAgo - minBarsAgo;
			//Print(String.Format("PlotPriceRangeVolumes() _ticksPerPlotRange:{0} MaxVolume:{1} _priceRangeVolumes.Count:{2}", new object[] { _ticksPerPlotRange, _maxVolume, _priceRangeVolumes.Count }));
			
			foreach (DictionaryEntry de in _priceRangeVolumes)
			{
				volumePercent = 1.0 * (int)de.Value / _maxVolume;
				if (volumePercent < _thresholdPercent / 100.0)
				{
					color = _subThresholdColor;
					//Print(String.Format("PlotPriceRangeVolumes() _subThresholdColor: R:{0} G:{1} B:{2}", new object[] { color.R, color.G, color.B } ));
				}
				else
				{

					gradient = (1.0 - volumePercent) / (1.0 - _thresholdPercent / 100.0);
					//Print(String.Format("PlotPriceRangeVolumes() gradient:{0} R:{1} G:{2} B:{3}", new object[] { gradient, Math.Abs(_maximumColor.R - (int) (gradient * (_maximumColor.R - _thresholdColor.R))), Math.Abs(_maximumColor.G - (int) (gradient * (_maximumColor.G - _thresholdColor.G))), Math.Abs(_maximumColor.B - (int) (gradient * (_maximumColor.B - _thresholdColor.B))) } ));
					
					color = Color.FromArgb(Math.Abs(_maximumColor.R - (int) (gradient * (_maximumColor.R - _thresholdColor.R))), 
														Math.Abs(_maximumColor.G - (int) (gradient * (_maximumColor.G - _thresholdColor.G))), 
														Math.Abs(_maximumColor.B - (int) (gradient * (_maximumColor.B - _thresholdColor.B)))); 
					//Print(String.Format("PlotPriceRangeVolumes() Gradient color: R:{0} G:{1} B:{2}", new object[] { color.R, color.G, color.B } ));
				}
					
				width = (int) Math.Round(volumePercent * maxWidth * _plotWidth / 100);

				yBase = Int32.Parse(de.Key.ToString()) * TickSize - _ticksPerPlotRange /100 / 2.0;
				switch (_plotType)
				{
					case HistogramPlotTypeEnum.LeftJustified:
					{
						leftBarsAgo = FirstVisibleBar; //_maxBarsAgo;
						rightBarsAgo = leftBarsAgo - width;
						break;
					}
					case (HistogramPlotTypeEnum.RightJustified):
					{
						leftBarsAgo = minBarsAgo + width;
						rightBarsAgo = minBarsAgo;
						break;
					}
					case (HistogramPlotTypeEnum.VerticalGradient):
					{
						leftBarsAgo = maxBarsAgo;
						rightBarsAgo = minBarsAgo;
						break;
					}
				}
					
				//Print(String.Format("PlotPriceRangeVolumes() Key:{0} Volume:{1} VolumePercent:{2} Width:{3} LeftBarsAgo:{4} RightBarsAgo:{5} yBase:{6}, yUpper:{7}, Opacity:{8}", new object[] { de.Key, (int) de.Value, volumePercent, width, leftBarsAgo, rightBarsAgo, yBase, yBase + TickSize, _opacity } ));
				DrawRectangle("pv" + de.Key, false, leftBarsAgo, yBase, rightBarsAgo, yBase + TickSize * _ticksPerPlotRange, Color.Empty, color, _opacity);
			}
		}
		
		private void RefreshPlotSettings()
		{

			_priceRangeVolumes = new Hashtable();
			_maxVolume = 0;
			
			switch (_dataRangeType)
			{
				case HistogramDataRangeTypeEnum.AllData:
				{
					_maxBarsAgo = Bars.Count - 1;
					_minBarsAgo = 0;
					break;
				}
				case HistogramDataRangeTypeEnum.VisibleDataOnly:
				{
					_maxBarsAgo = FirstVisibleBar;
					_minBarsAgo = Math.Max(LastVisibleBar, 0);
					break;
				}
			}
			
			//determine the min/max prices of the plottable data
			for (int i = _minBarsAgo; i <= _maxBarsAgo; i++)
			{
				_minPrice = Math.Min(Low[i], _minPrice); 
				_maxPrice = Math.Max(High[i], _maxPrice); 
			}
			
			//determine the optimum number of ticks per displayable line so the preformance doesn't degrade
			_ticksPerPlotRange = (int) ((_maxPrice -  _minPrice) / TickSize / ChartControl.Controls["pnlchart"].Height);
			_ticksPerPlotRange = Math.Max(_ticksPerPlotRange, 1);
			//Print(String.Format("RefreshPlotSettings() minPrice:{0} maxPrice:{1} ticksPerPlotRange:{2} TickCount:{3} ChartPixelHeight:{4}", new object[] { minPrice, maxPrice , _ticksPerPlotRange, (maxPrice - minPrice) / TickSize, ChartControl.Controls["pnlchart"].Height } ));
		}
		
        #region Properties
		[XmlIgnore()]
        [Description("Version")]
        [Category("Parameters")]
        public string _Version
        {
            get { return "V 1.3"; }
            set {  }
        }

		private Color _maximumColor = Color.Yellow; // Default setting for MaximumColor
		[XmlIgnore()]
        [Description("Color for maximum Tick volume bar")]
        [Category("Parameters")]
        public Color MaximumColor
        {
            get { return _maximumColor; }
            set { _maximumColor = value; }
        }
		//required for serialization
		[Browsable(false)]
		public string MaximumColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(_maximumColor); }
			set { _maximumColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}

		private Color _thresholdColor = Color.Gray; // Default setting for ThresholdColor
		[XmlIgnore()]
        [Description("Color for Tick volume bars which are at the threshold percentage of maximum")]
        [Category("Parameters")]
        public Color ThresholdColor
        {
            get { return _thresholdColor; }
            set { _thresholdColor = value; }
        }
		//required for serialization
		[Browsable(false)]
		public string ThresholdColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(_thresholdColor); }
			set { _thresholdColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}

		private Color _subThresholdColor = Color.DimGray; // Default setting for SubThresholdColor
		[XmlIgnore()]
        [Description("Color for Tick volume bars which are below the threshold percentage of maximum")]
        [Category("Parameters")]
        public Color SubThresholdColor
        {
            get { return _subThresholdColor; }
            set { _subThresholdColor = value; }
        }
		//required for serialization
		[Browsable(false)]
		public string SubThresholdColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(_subThresholdColor); }
			set { _subThresholdColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}

		private int _opacity = 5; // Default setting for Opacity
        [Description("Opacity of Tick volume bars. \nRange: 1-8 \nColor will be blended with chart background color")]
        [Category("Parameters")]
        public int Opacity
        {
            get { return _opacity; }
            set { _opacity = Math.Min(Math.Max(1, value), 8); }
        }
        
        private int _thresholdPercent = 70; // Default setting for ThresholdPercent
        [Description("Percent of Maximum volume value. \nControls plot characteristics.")]
        [Category("Parameters")]
        public int ThresholdPercent
        {
            get { return _thresholdPercent; }
            set { _thresholdPercent = Math.Min(Math.Max(0, value), 99); }
        }

        private int _plotWidth = 100; // Default setting for Opacity
        [Description("Width of screen (Percentage) used to draw histogram")]
        [Category("Parameters")]
        public int PlotWidth
        {
            get { return _plotWidth; }
            set { _plotWidth = Math.Min(100, Math.Max(1, value)); }
        }

        private HistogramDataRangeTypeEnum _dataRangeType = HistogramDataRangeTypeEnum.AllData; // Default setting for Opacity
        [Description("Data to use for generating the histogram. \n AllData: Use chart's entire Dataseries \n VisibleOnly: Use the data underlying the bars displayed on the chart")]
        [Category("Parameters")]
        public HistogramDataRangeTypeEnum DataRangeType
        {
            get { return _dataRangeType; }
            set { _dataRangeType = value; }
        }
		
		private HistogramPlotTypeEnum _plotType = HistogramPlotTypeEnum.LeftJustified; // Default setting for PriceRangeMultiple
        [Description("Type of display to use when plotting results. \n LeftJustified: Gaussian distribution based at chart's left edge \n RightJustified: Gaussian distribution based at chart's right edge \n VerticalGradient: Full width lines whose color relates to relative volume")]
        [Category("Parameters")]
        public HistogramPlotTypeEnum PlotType
        {
            get { return _plotType; }
            set { _plotType = value; }
        }

		private VolumeSourceEnum _volumeSource = VolumeSourceEnum.ShareVolume; // Default setting for VolumeSource
        [Description("Data used to calculate histogram \n ShareVolume: Numbers of shares \n MonetaryValue: Share price * Share volume ")]
        [Category("Parameters")]
        public VolumeSourceEnum VolumeSource
        {
            get { return _volumeSource; }
            set { _volumeSource = value; }
        }
		#endregion
    }
}

public enum HistogramPlotTypeEnum
{
	LeftJustified = 1
	,RightJustified
	,VerticalGradient
}

public enum HistogramDataRangeTypeEnum
{
	AllData = 1
	,VisibleDataOnly
}

public enum VolumeSourceEnum
{
	ShareVolume = 1,
	MonetaryValue	
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private PriceVolumeHistogram[] cachePriceVolumeHistogram = null;

        private static PriceVolumeHistogram checkPriceVolumeHistogram = new PriceVolumeHistogram();

        /// <summary>
        /// Volumes Per Tick displayed as histogram variants.
        /// </summary>
        /// <returns></returns>
        public PriceVolumeHistogram PriceVolumeHistogram(string _Version, HistogramDataRangeTypeEnum dataRangeType, Color maximumColor, int opacity, HistogramPlotTypeEnum plotType, int plotWidth, Color subThresholdColor, Color thresholdColor, int thresholdPercent, VolumeSourceEnum volumeSource)
        {
            return PriceVolumeHistogram(Input, _Version, dataRangeType, maximumColor, opacity, plotType, plotWidth, subThresholdColor, thresholdColor, thresholdPercent, volumeSource);
        }

        /// <summary>
        /// Volumes Per Tick displayed as histogram variants.
        /// </summary>
        /// <returns></returns>
        public PriceVolumeHistogram PriceVolumeHistogram(Data.IDataSeries input, string _Version, HistogramDataRangeTypeEnum dataRangeType, Color maximumColor, int opacity, HistogramPlotTypeEnum plotType, int plotWidth, Color subThresholdColor, Color thresholdColor, int thresholdPercent, VolumeSourceEnum volumeSource)
        {
            if (cachePriceVolumeHistogram != null)
                for (int idx = 0; idx < cachePriceVolumeHistogram.Length; idx++)
                    if (cachePriceVolumeHistogram[idx]._Version == _Version && cachePriceVolumeHistogram[idx].DataRangeType == dataRangeType && cachePriceVolumeHistogram[idx].MaximumColor == maximumColor && cachePriceVolumeHistogram[idx].Opacity == opacity && cachePriceVolumeHistogram[idx].PlotType == plotType && cachePriceVolumeHistogram[idx].PlotWidth == plotWidth && cachePriceVolumeHistogram[idx].SubThresholdColor == subThresholdColor && cachePriceVolumeHistogram[idx].ThresholdColor == thresholdColor && cachePriceVolumeHistogram[idx].ThresholdPercent == thresholdPercent && cachePriceVolumeHistogram[idx].VolumeSource == volumeSource && cachePriceVolumeHistogram[idx].EqualsInput(input))
                        return cachePriceVolumeHistogram[idx];

            lock (checkPriceVolumeHistogram)
            {
                checkPriceVolumeHistogram._Version = _Version;
                _Version = checkPriceVolumeHistogram._Version;
                checkPriceVolumeHistogram.DataRangeType = dataRangeType;
                dataRangeType = checkPriceVolumeHistogram.DataRangeType;
                checkPriceVolumeHistogram.MaximumColor = maximumColor;
                maximumColor = checkPriceVolumeHistogram.MaximumColor;
                checkPriceVolumeHistogram.Opacity = opacity;
                opacity = checkPriceVolumeHistogram.Opacity;
                checkPriceVolumeHistogram.PlotType = plotType;
                plotType = checkPriceVolumeHistogram.PlotType;
                checkPriceVolumeHistogram.PlotWidth = plotWidth;
                plotWidth = checkPriceVolumeHistogram.PlotWidth;
                checkPriceVolumeHistogram.SubThresholdColor = subThresholdColor;
                subThresholdColor = checkPriceVolumeHistogram.SubThresholdColor;
                checkPriceVolumeHistogram.ThresholdColor = thresholdColor;
                thresholdColor = checkPriceVolumeHistogram.ThresholdColor;
                checkPriceVolumeHistogram.ThresholdPercent = thresholdPercent;
                thresholdPercent = checkPriceVolumeHistogram.ThresholdPercent;
                checkPriceVolumeHistogram.VolumeSource = volumeSource;
                volumeSource = checkPriceVolumeHistogram.VolumeSource;

                if (cachePriceVolumeHistogram != null)
                    for (int idx = 0; idx < cachePriceVolumeHistogram.Length; idx++)
                        if (cachePriceVolumeHistogram[idx]._Version == _Version && cachePriceVolumeHistogram[idx].DataRangeType == dataRangeType && cachePriceVolumeHistogram[idx].MaximumColor == maximumColor && cachePriceVolumeHistogram[idx].Opacity == opacity && cachePriceVolumeHistogram[idx].PlotType == plotType && cachePriceVolumeHistogram[idx].PlotWidth == plotWidth && cachePriceVolumeHistogram[idx].SubThresholdColor == subThresholdColor && cachePriceVolumeHistogram[idx].ThresholdColor == thresholdColor && cachePriceVolumeHistogram[idx].ThresholdPercent == thresholdPercent && cachePriceVolumeHistogram[idx].VolumeSource == volumeSource && cachePriceVolumeHistogram[idx].EqualsInput(input))
                            return cachePriceVolumeHistogram[idx];

                PriceVolumeHistogram indicator = new PriceVolumeHistogram();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator._Version = _Version;
                indicator.DataRangeType = dataRangeType;
                indicator.MaximumColor = maximumColor;
                indicator.Opacity = opacity;
                indicator.PlotType = plotType;
                indicator.PlotWidth = plotWidth;
                indicator.SubThresholdColor = subThresholdColor;
                indicator.ThresholdColor = thresholdColor;
                indicator.ThresholdPercent = thresholdPercent;
                indicator.VolumeSource = volumeSource;
                Indicators.Add(indicator);
                indicator.SetUp();

                PriceVolumeHistogram[] tmp = new PriceVolumeHistogram[cachePriceVolumeHistogram == null ? 1 : cachePriceVolumeHistogram.Length + 1];
                if (cachePriceVolumeHistogram != null)
                    cachePriceVolumeHistogram.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cachePriceVolumeHistogram = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Volumes Per Tick displayed as histogram variants.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.PriceVolumeHistogram PriceVolumeHistogram(string _Version, HistogramDataRangeTypeEnum dataRangeType, Color maximumColor, int opacity, HistogramPlotTypeEnum plotType, int plotWidth, Color subThresholdColor, Color thresholdColor, int thresholdPercent, VolumeSourceEnum volumeSource)
        {
            return _indicator.PriceVolumeHistogram(Input, _Version, dataRangeType, maximumColor, opacity, plotType, plotWidth, subThresholdColor, thresholdColor, thresholdPercent, volumeSource);
        }

        /// <summary>
        /// Volumes Per Tick displayed as histogram variants.
        /// </summary>
        /// <returns></returns>
        public Indicator.PriceVolumeHistogram PriceVolumeHistogram(Data.IDataSeries input, string _Version, HistogramDataRangeTypeEnum dataRangeType, Color maximumColor, int opacity, HistogramPlotTypeEnum plotType, int plotWidth, Color subThresholdColor, Color thresholdColor, int thresholdPercent, VolumeSourceEnum volumeSource)
        {
            return _indicator.PriceVolumeHistogram(input, _Version, dataRangeType, maximumColor, opacity, plotType, plotWidth, subThresholdColor, thresholdColor, thresholdPercent, volumeSource);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Volumes Per Tick displayed as histogram variants.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.PriceVolumeHistogram PriceVolumeHistogram(string _Version, HistogramDataRangeTypeEnum dataRangeType, Color maximumColor, int opacity, HistogramPlotTypeEnum plotType, int plotWidth, Color subThresholdColor, Color thresholdColor, int thresholdPercent, VolumeSourceEnum volumeSource)
        {
            return _indicator.PriceVolumeHistogram(Input, _Version, dataRangeType, maximumColor, opacity, plotType, plotWidth, subThresholdColor, thresholdColor, thresholdPercent, volumeSource);
        }

        /// <summary>
        /// Volumes Per Tick displayed as histogram variants.
        /// </summary>
        /// <returns></returns>
        public Indicator.PriceVolumeHistogram PriceVolumeHistogram(Data.IDataSeries input, string _Version, HistogramDataRangeTypeEnum dataRangeType, Color maximumColor, int opacity, HistogramPlotTypeEnum plotType, int plotWidth, Color subThresholdColor, Color thresholdColor, int thresholdPercent, VolumeSourceEnum volumeSource)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.PriceVolumeHistogram(input, _Version, dataRangeType, maximumColor, opacity, plotType, plotWidth, subThresholdColor, thresholdColor, thresholdPercent, volumeSource);
        }
    }
}
#endregion
