#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// A market breadth indicator that is based on the difference between the number of advancing and declining issues on the NYSE.
    /// </summary>
    [Description("A market breadth indicator that is based on the difference between the number of advancing and declining issues on the NYSE.")]
    public class McClellanOscillator : Indicator
    {
        #region Variables
        
            private int 	slowPeriod 			= 39;
			private int 	fastPeriod 			= 19;
		
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
			Add("^ADD", BarsPeriod.Id, BarsPeriod.Value);
			Add(new Plot(Color.FromKnownColor(KnownColor.Lime), PlotStyle.Line, "McClellanUpper"));
			Add(new Plot(Color.FromKnownColor(KnownColor.Red), PlotStyle.Line, "McClellanLower"));
			
			Plots[0].Min = 0; 
			Plots[1].Max = 0;
			
			DrawOnPricePanel 	= false;
            Overlay				= false;
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {	
			if (CurrentBarArray[0] > 0)
			{
				if (CurrentBarArray[1] > 0)
				{
					double McClellanUpper = Math.Max(((EMA(BarsArray[1], FastPeriod)[0]) - (EMA(BarsArray[1], SlowPeriod)[0])), 0);
     				double McClellanLower = Math.Min(((EMA(BarsArray[1], FastPeriod)[0]) - (EMA(BarsArray[1], SlowPeriod)[0])), 0);
     
					McClellanPlotUpper.Set(McClellanUpper);
					McClellanPlotLower.Set(McClellanLower);
			
        		}
			}
				
			DrawRegion("Upper" + CurrentBar, Bars.BarsSinceSession, 0, McClellanPlotUpper, 0, Color.Lime, Color.Lime, 10);
			DrawRegion("Lower" + CurrentBar, Bars.BarsSinceSession, 0, McClellanPlotLower, 0, Color.Red, Color.Red, 10);
		}

        #region Properties
        [Browsable(false)]	
        [XmlIgnore()]		
        public DataSeries McClellanPlotUpper
        {
            get { return Values[0]; }
        }
		
		[Browsable(false)]	 
        [XmlIgnore()]		
        public DataSeries McClellanPlotLower
        {
            get { return Values[1]; }
        }
		
        [Description("Number of bars used for calculation.")]
        [GridCategory("Parameters")]
        public int SlowPeriod
        {
            get { return slowPeriod; }
            set { slowPeriod = Math.Max(1, value); }
        }
		
		[Description("Number of bars used for calculation.")]
        [GridCategory("Parameters")]
        public int FastPeriod
        {
            get { return fastPeriod; }
            set { fastPeriod = Math.Max(1, value); }
        }
		
		
		
		
		
		
		
		
		
        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private McClellanOscillator[] cacheMcClellanOscillator = null;

        private static McClellanOscillator checkMcClellanOscillator = new McClellanOscillator();

        /// <summary>
        /// A market breadth indicator that is based on the difference between the number of advancing and declining issues on the NYSE.
        /// </summary>
        /// <returns></returns>
        public McClellanOscillator McClellanOscillator(int fastPeriod, int slowPeriod)
        {
            return McClellanOscillator(Input, fastPeriod, slowPeriod);
        }

        /// <summary>
        /// A market breadth indicator that is based on the difference between the number of advancing and declining issues on the NYSE.
        /// </summary>
        /// <returns></returns>
        public McClellanOscillator McClellanOscillator(Data.IDataSeries input, int fastPeriod, int slowPeriod)
        {
            if (cacheMcClellanOscillator != null)
                for (int idx = 0; idx < cacheMcClellanOscillator.Length; idx++)
                    if (cacheMcClellanOscillator[idx].FastPeriod == fastPeriod && cacheMcClellanOscillator[idx].SlowPeriod == slowPeriod && cacheMcClellanOscillator[idx].EqualsInput(input))
                        return cacheMcClellanOscillator[idx];

            lock (checkMcClellanOscillator)
            {
                checkMcClellanOscillator.FastPeriod = fastPeriod;
                fastPeriod = checkMcClellanOscillator.FastPeriod;
                checkMcClellanOscillator.SlowPeriod = slowPeriod;
                slowPeriod = checkMcClellanOscillator.SlowPeriod;

                if (cacheMcClellanOscillator != null)
                    for (int idx = 0; idx < cacheMcClellanOscillator.Length; idx++)
                        if (cacheMcClellanOscillator[idx].FastPeriod == fastPeriod && cacheMcClellanOscillator[idx].SlowPeriod == slowPeriod && cacheMcClellanOscillator[idx].EqualsInput(input))
                            return cacheMcClellanOscillator[idx];

                McClellanOscillator indicator = new McClellanOscillator();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.FastPeriod = fastPeriod;
                indicator.SlowPeriod = slowPeriod;
                Indicators.Add(indicator);
                indicator.SetUp();

                McClellanOscillator[] tmp = new McClellanOscillator[cacheMcClellanOscillator == null ? 1 : cacheMcClellanOscillator.Length + 1];
                if (cacheMcClellanOscillator != null)
                    cacheMcClellanOscillator.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheMcClellanOscillator = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// A market breadth indicator that is based on the difference between the number of advancing and declining issues on the NYSE.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.McClellanOscillator McClellanOscillator(int fastPeriod, int slowPeriod)
        {
            return _indicator.McClellanOscillator(Input, fastPeriod, slowPeriod);
        }

        /// <summary>
        /// A market breadth indicator that is based on the difference between the number of advancing and declining issues on the NYSE.
        /// </summary>
        /// <returns></returns>
        public Indicator.McClellanOscillator McClellanOscillator(Data.IDataSeries input, int fastPeriod, int slowPeriod)
        {
            return _indicator.McClellanOscillator(input, fastPeriod, slowPeriod);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// A market breadth indicator that is based on the difference between the number of advancing and declining issues on the NYSE.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.McClellanOscillator McClellanOscillator(int fastPeriod, int slowPeriod)
        {
            return _indicator.McClellanOscillator(Input, fastPeriod, slowPeriod);
        }

        /// <summary>
        /// A market breadth indicator that is based on the difference between the number of advancing and declining issues on the NYSE.
        /// </summary>
        /// <returns></returns>
        public Indicator.McClellanOscillator McClellanOscillator(Data.IDataSeries input, int fastPeriod, int slowPeriod)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.McClellanOscillator(input, fastPeriod, slowPeriod);
        }
    }
}
#endregion
