// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	/// <summary>
	/// The MACD_AvgUpDwn (Moving Average Convergence/Divergence) is a trend following momentum indicator that shows the relationship between two moving averages of prices.
	/// </summary>
	[Description("The MACD_AvgUpDwn (Moving Average Convergence/Divergence) is a trend following momentum indicator that shows the relationship between two moving averages of prices.")]
	public class MACD_AvgUpDwn : Indicator
	{
		#region Variables
		private int					fast	= 12;
		private int					slow	= 26;
		private int					smooth	= 9;
		private	DataSeries		fastEma;
		private	DataSeries		slowEma;
		#endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void Initialize()
		{
			Add(new Plot(new Pen(Color.Green, 9), PlotStyle.Bar, "MacdU"));
			Add(new Plot(new Pen(Color.Red, 9), PlotStyle.Bar, "MacdD"));
			Add(new Plot(Color.Black, "Avg"));
			Add(new Plot(Color.Transparent, PlotStyle.Line, "Diff"));
			
			//Add(new Plot(Color.Green, "Macd"));
			//Add(new Plot(Color.DarkViolet, "Avg"));
			//Add(new Plot(new Pen(Color.Navy, 2), PlotStyle.Bar, "Diff"));

			Add(new Line(Color.DarkGray, 0, "Zero line"));

			fastEma	= new DataSeries(this);
			slowEma	= new DataSeries(this);
		}

		/// <summary>
		/// Calculates the indicator value(s) at the current index.
		/// </summary>
		protected override void OnBarUpdate()
		{
			if (CurrentBar == 0)
			{
				fastEma.Set(Input[0]);
				slowEma.Set(Input[0]);
				MacdU.Set(0);
				MacdD.Set(0);
				Avg.Set(0);
				Diff.Set(0);
			}
			else
			{
				fastEma.Set((2.0 / (1 + Fast)) * Input[0] + (1 - (2.0 / (1 + Fast))) * fastEma[1]);
				slowEma.Set((2.0 / (1 + Slow)) * Input[0] + (1 - (2.0 / (1 + Slow))) * slowEma[1]);

				double macd		= fastEma[0] - slowEma[0];
				double macdAvg	= (2.0 / (1 + Smooth)) * macd + (1 - (2.0 / (1 + Smooth))) * Avg[1];
				
				
				Avg.Set(macdAvg);
				
				if( Avg[0]> Avg[1])
				{
					MacdU.Set(macd);
				}
				else
				{
					MacdD.Set(macd);
				}
				
				Diff.Set(macd - macdAvg);
			}
		}

		#region Properties
		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries MacdU
		{
			get { return Values[0]; }
		}
		/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries MacdD
		{
			get { return Values[1]; }
		}
			/// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Avg
		{
			get { return Values[2]; }
		}
        /// <summary>
		/// </summary>
		[Browsable(false)]
		[XmlIgnore()]
		public DataSeries Diff
		{
			get { return Values[3]; }
		}

		/// <summary>
		/// </summary>
		[Description("Number of bars for fast EMA")]
		[GridCategory("Parameters")]
		public int Fast
		{
			get { return fast; }
			set { fast = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[Description("Number of bars for slow EMA")]
		[GridCategory("Parameters")]
		public int Slow
		{
			get { return slow; }
			set { slow = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[Description("Number of bars for smoothing")]
		[GridCategory("Parameters")]
		public int Smooth
		{
			get { return smooth; }
			set { smooth = Math.Max(1, value); }
		}
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private MACD_AvgUpDwn[] cacheMACD_AvgUpDwn = null;

        private static MACD_AvgUpDwn checkMACD_AvgUpDwn = new MACD_AvgUpDwn();

        /// <summary>
        /// The MACD_AvgUpDwn (Moving Average Convergence/Divergence) is a trend following momentum indicator that shows the relationship between two moving averages of prices.
        /// </summary>
        /// <returns></returns>
        public MACD_AvgUpDwn MACD_AvgUpDwn(int fast, int slow, int smooth)
        {
            return MACD_AvgUpDwn(Input, fast, slow, smooth);
        }

        /// <summary>
        /// The MACD_AvgUpDwn (Moving Average Convergence/Divergence) is a trend following momentum indicator that shows the relationship between two moving averages of prices.
        /// </summary>
        /// <returns></returns>
        public MACD_AvgUpDwn MACD_AvgUpDwn(Data.IDataSeries input, int fast, int slow, int smooth)
        {
            if (cacheMACD_AvgUpDwn != null)
                for (int idx = 0; idx < cacheMACD_AvgUpDwn.Length; idx++)
                    if (cacheMACD_AvgUpDwn[idx].Fast == fast && cacheMACD_AvgUpDwn[idx].Slow == slow && cacheMACD_AvgUpDwn[idx].Smooth == smooth && cacheMACD_AvgUpDwn[idx].EqualsInput(input))
                        return cacheMACD_AvgUpDwn[idx];

            lock (checkMACD_AvgUpDwn)
            {
                checkMACD_AvgUpDwn.Fast = fast;
                fast = checkMACD_AvgUpDwn.Fast;
                checkMACD_AvgUpDwn.Slow = slow;
                slow = checkMACD_AvgUpDwn.Slow;
                checkMACD_AvgUpDwn.Smooth = smooth;
                smooth = checkMACD_AvgUpDwn.Smooth;

                if (cacheMACD_AvgUpDwn != null)
                    for (int idx = 0; idx < cacheMACD_AvgUpDwn.Length; idx++)
                        if (cacheMACD_AvgUpDwn[idx].Fast == fast && cacheMACD_AvgUpDwn[idx].Slow == slow && cacheMACD_AvgUpDwn[idx].Smooth == smooth && cacheMACD_AvgUpDwn[idx].EqualsInput(input))
                            return cacheMACD_AvgUpDwn[idx];

                MACD_AvgUpDwn indicator = new MACD_AvgUpDwn();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.Fast = fast;
                indicator.Slow = slow;
                indicator.Smooth = smooth;
                Indicators.Add(indicator);
                indicator.SetUp();

                MACD_AvgUpDwn[] tmp = new MACD_AvgUpDwn[cacheMACD_AvgUpDwn == null ? 1 : cacheMACD_AvgUpDwn.Length + 1];
                if (cacheMACD_AvgUpDwn != null)
                    cacheMACD_AvgUpDwn.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheMACD_AvgUpDwn = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// The MACD_AvgUpDwn (Moving Average Convergence/Divergence) is a trend following momentum indicator that shows the relationship between two moving averages of prices.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.MACD_AvgUpDwn MACD_AvgUpDwn(int fast, int slow, int smooth)
        {
            return _indicator.MACD_AvgUpDwn(Input, fast, slow, smooth);
        }

        /// <summary>
        /// The MACD_AvgUpDwn (Moving Average Convergence/Divergence) is a trend following momentum indicator that shows the relationship between two moving averages of prices.
        /// </summary>
        /// <returns></returns>
        public Indicator.MACD_AvgUpDwn MACD_AvgUpDwn(Data.IDataSeries input, int fast, int slow, int smooth)
        {
            return _indicator.MACD_AvgUpDwn(input, fast, slow, smooth);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// The MACD_AvgUpDwn (Moving Average Convergence/Divergence) is a trend following momentum indicator that shows the relationship between two moving averages of prices.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.MACD_AvgUpDwn MACD_AvgUpDwn(int fast, int slow, int smooth)
        {
            return _indicator.MACD_AvgUpDwn(Input, fast, slow, smooth);
        }

        /// <summary>
        /// The MACD_AvgUpDwn (Moving Average Convergence/Divergence) is a trend following momentum indicator that shows the relationship between two moving averages of prices.
        /// </summary>
        /// <returns></returns>
        public Indicator.MACD_AvgUpDwn MACD_AvgUpDwn(Data.IDataSeries input, int fast, int slow, int smooth)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.MACD_AvgUpDwn(input, fast, slow, smooth);
        }
    }
}
#endregion
