#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// (c) Finn Jurbol. For more free indicators visit www.jurbol.com
namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Enter the description of your new custom indicator here
    /// </summary>
    [Description("Jurbol BBMacd. Combination of Bollinger Bands and Macd. Programmed for NT 7")]
    public class JurbolBBmacd : Indicator
    {
        #region Variables
        private int					fast		= 12;
		private int					slow		= 26;
		private int					smooth		= 1;
		private	double				numStdDev	= 0.882;
		private int			   		period		= 10;
		private int			   		opacity		= 3;
		private int			   		arrowSize		= 14;
		private double			   		arrowOffset		= 0.5;
		private Color       bullColor   = Color.Blue;
		private Color		bearColor = Color.Maroon;
		private Color		neutralColor = Color.Yellow;
		private Color		fillColor = Color.CadetBlue;
		private Color       bullArrowColor   = Color.Cyan;
		private Color		bearArrowColor = Color.Magenta;
        #endregion

        /// <summary>
        /// This method is used to configure the indicator and is called once before any bar data is loaded.
        /// </summary>
        protected override void Initialize()
        {
            Add(new Plot(Color.FromKnownColor(KnownColor.Orange), PlotStyle.Dot, "Macd"));
			Add(new Plot(Color.FromKnownColor(KnownColor.Black), PlotStyle.Line, "Avg"));
			Add(new Plot(Color.FromKnownColor(KnownColor.LimeGreen), PlotStyle.Line, "BUpper"));
			Add(new Plot(Color.FromKnownColor(KnownColor.Red), PlotStyle.Line, "BLower"));
			
			
			Plots[0].Pen.Width = 4;
			Plots[1].Pen.Width = 1;
			Plots[2].Pen.Width = 2;
			Plots[3].Pen.Width = 2;
			
			
			Add(new Line(Color.DarkGray, 0, "Zero line"));
           
			DisplayInDataBox 	= false; 
			PaintPriceMarkers 	= true; 
            CalculateOnBarClose	= false;
            Overlay				= false;
            PriceTypeSupported	= true;
			DrawOnPricePanel	= false;
        }

        /// <summary>
        /// Called on each bar update event (incoming tick)
        /// </summary>
        protected override void OnBarUpdate()
        {
			if (CurrentBar < 1) 
				return;
            // Use this method for calculating your indicator values. Assign a value to each
            // plot below by replacing 'Close[0]' with your own formula.
         DrawRegion("Fill_Band", CurrentBar, 0, BollingerUpper, BollingerLower, Color.Transparent, fillColor, opacity);
			Macd.Set(MACD(fast, slow, smooth)[0]);
			Avg.Set(MACD(fast, slow, smooth)[0]);

			BollingerUpper.Set(Bollinger(MACD(fast, slow, smooth), numStdDev, period).Upper[0]);
			BollingerLower.Set(Bollinger(MACD(fast, slow, smooth), numStdDev, period).Lower[0]);
			
			
			
			if(Rising(Macd))
			{	
			PlotColors[0][0] = bullColor;
			}
			else if ( Falling(Macd))
			{	
			PlotColors[0][0] = bearColor;
			}
			else
			{	
			PlotColors[0][0] = neutralColor;
			}
			Font arrowFont = new Font("Wingdings", arrowSize, FontStyle.Bold);
			if (CrossAbove(MACD(fast, slow, smooth), 0, 1))
			{
		    DrawText("CrossingUp"+CurrentBar, false, "é", 0, Macd[0]+arrowOffset*TickSize,0, bullArrowColor, arrowFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
			}
			else if(CrossBelow(MACD(fast, slow, smooth), 0, 1))
			{
		     DrawText("CrossingDn"+CurrentBar, false, "ê", 0, Macd[0]-arrowOffset*TickSize,0, bearArrowColor, arrowFont, StringAlignment.Center, Color.Transparent, Color.Transparent, 0);
			}
			
 

        }

        #region Properties
		[Description("Number of bars for fast EMA")]
		[Category("Parameters")]
		[Gui.Design.DisplayNameAttribute("01. Fast")]
		public int Fast
		{
			get { return fast; }
			set { fast = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[Description("Number of bars for slow EMA")]
		[Category("Parameters")]
		[Gui.Design.DisplayNameAttribute("01. Slow")]
		public int Slow
		{
			get { return slow; }
			set { slow = Math.Max(1, value); }
		}

		/// <summary>
		/// </summary>
		[Description("Number of bars for smoothing")]
		[Category("Parameters")]
		[Gui.Design.DisplayNameAttribute("03. Smooth")]
		public int Smooth
		{
			get { return smooth; }
			set { smooth = Math.Max(1, value); }
		}
		
			[Description("Number of standard deviations")]
		[GridCategory("Parameters")]
		[Gui.Design.DisplayNameAttribute("04. BB std. dev.")]
		public double NumStdDev
		{
			get { return numStdDev; }
			set { numStdDev = Math.Max(0, value); }
		}

		/// <summary>
		/// </summary>
		[Description("Numbers of bars used for calculations")]
		[GridCategory("Parameters")]
		[Gui.Design.DisplayNameAttribute("05. BB Period")]
		public int Period
		{
			get { return period; }
			set { period = Math.Max(1, value); }
		}
		
		[Description("Fill Opacity")]
        [GridCategory("Parameters")]
		[Gui.Design.DisplayNameAttribute("06. FillColor Opacity")]
        public int Opacity
        {
            get { return opacity; }
            set { opacity = value; }
        }
		
		[Description("ArrowSize")]
        [GridCategory("Parameters")]
		[Gui.Design.DisplayNameAttribute("06. Arrow Size")]
        public int ArrowSize
        {
            get { return arrowSize; }
            set { arrowSize = value; }
        }
		
		[Description("ArrowOffset")]
        [GridCategory("Parameters")]
		[Gui.Design.DisplayNameAttribute("07. Arrow Offset from MacdLine")]
        public double ArrowOffset
        {
            get { return arrowOffset; }
            set { arrowOffset = value; }
        }
		
		
        [Browsable(false)]	
        [XmlIgnore()]		
        public DataSeries Macd
        {
            get { return Values[0]; }
        }

         [Browsable(false)]	
        [XmlIgnore()]		
        public DataSeries Avg
        {
            get { return Values[1]; }
        }
		  [Browsable(false)]	
        [XmlIgnore()]		
        public DataSeries BollingerUpper
        {
            get { return Values[2]; }
        }
		
		  [Browsable(false)]	
        [XmlIgnore()]		
        public DataSeries BollingerLower
        {
            get { return Values[3]; }
        }
		//----Visual---------------------------------------------------
		
		[XmlIgnore()]
        [Description("Color for Up Bars")]
        [Category("Visual")]
		[Gui.Design.DisplayNameAttribute("01. Color Bull Dots")]
        public Color BullColor
        {
            get { return bullColor; }
            set { bullColor = value; }
        }
		[Browsable(false)]
		public string BullColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(BullColor); }
			set { BullColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		[XmlIgnore()]
        [Description("Color for Down Bars")]
        [Category("Visual")]
		[Gui.Design.DisplayNameAttribute("02. Color Bear Dots")]
        public Color BearColor
        {
            get { return bearColor; }
            set { bearColor = value; }
        }
		[Browsable(false)]
		public string BearColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(BearColor); }
			set { BearColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		[XmlIgnore()]
        [Description("Color for NN Bars")]
        [Category("Visual")]
		[Gui.Design.DisplayNameAttribute("03. Color Neutral Dots")]
        public Color NeutralColor
        {
            get { return neutralColor; }
            set { neutralColor = value; }
        }
		[Browsable(false)]
		public string NeutralColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(NeutralColor); }
			set { NeutralColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		[XmlIgnore()]
        [Description("Color for fill")]
        [Category("Visual")]
		[Gui.Design.DisplayNameAttribute("04. Fill Color")]
        public Color FillColor
        {
            get { return fillColor; }
            set { fillColor = value; }
        }
		[Browsable(false)]
		public string FillColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(FillColor); }
			set { FillColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		[XmlIgnore()]
        [Description("Arrow Crossing up Color")]
        [Category("Visual")]
		[Gui.Design.DisplayNameAttribute("05. Color Up Arrows")]
        public Color BullArrowColor
        {
            get { return bullArrowColor; }
            set { bullArrowColor = value; }
        }
		[Browsable(false)]
		public string BullArrowColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(BullArrowColor); }
			set { BullArrowColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
		[XmlIgnore()]
        [Description("Arrow Crossing udn Color")]
        [Category("Visual")]
		[Gui.Design.DisplayNameAttribute("06. Color Dn Arrows")]
        public Color BearArrowColor
        {
            get { return bearArrowColor; }
            set { bearArrowColor = value; }
        }
		[Browsable(false)]
		public string BearArrowColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(BearArrowColor); }
			set { BearArrowColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}
		
        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private JurbolBBmacd[] cacheJurbolBBmacd = null;

        private static JurbolBBmacd checkJurbolBBmacd = new JurbolBBmacd();

        /// <summary>
        /// Jurbol BBMacd. Combination of Bollinger Bands and Macd. Programmed for NT 7
        /// </summary>
        /// <returns></returns>
        public JurbolBBmacd JurbolBBmacd(double arrowOffset, int arrowSize, int fast, double numStdDev, int opacity, int period, int slow, int smooth)
        {
            return JurbolBBmacd(Input, arrowOffset, arrowSize, fast, numStdDev, opacity, period, slow, smooth);
        }

        /// <summary>
        /// Jurbol BBMacd. Combination of Bollinger Bands and Macd. Programmed for NT 7
        /// </summary>
        /// <returns></returns>
        public JurbolBBmacd JurbolBBmacd(Data.IDataSeries input, double arrowOffset, int arrowSize, int fast, double numStdDev, int opacity, int period, int slow, int smooth)
        {
            if (cacheJurbolBBmacd != null)
                for (int idx = 0; idx < cacheJurbolBBmacd.Length; idx++)
                    if (Math.Abs(cacheJurbolBBmacd[idx].ArrowOffset - arrowOffset) <= double.Epsilon && cacheJurbolBBmacd[idx].ArrowSize == arrowSize && cacheJurbolBBmacd[idx].Fast == fast && Math.Abs(cacheJurbolBBmacd[idx].NumStdDev - numStdDev) <= double.Epsilon && cacheJurbolBBmacd[idx].Opacity == opacity && cacheJurbolBBmacd[idx].Period == period && cacheJurbolBBmacd[idx].Slow == slow && cacheJurbolBBmacd[idx].Smooth == smooth && cacheJurbolBBmacd[idx].EqualsInput(input))
                        return cacheJurbolBBmacd[idx];

            lock (checkJurbolBBmacd)
            {
                checkJurbolBBmacd.ArrowOffset = arrowOffset;
                arrowOffset = checkJurbolBBmacd.ArrowOffset;
                checkJurbolBBmacd.ArrowSize = arrowSize;
                arrowSize = checkJurbolBBmacd.ArrowSize;
                checkJurbolBBmacd.Fast = fast;
                fast = checkJurbolBBmacd.Fast;
                checkJurbolBBmacd.NumStdDev = numStdDev;
                numStdDev = checkJurbolBBmacd.NumStdDev;
                checkJurbolBBmacd.Opacity = opacity;
                opacity = checkJurbolBBmacd.Opacity;
                checkJurbolBBmacd.Period = period;
                period = checkJurbolBBmacd.Period;
                checkJurbolBBmacd.Slow = slow;
                slow = checkJurbolBBmacd.Slow;
                checkJurbolBBmacd.Smooth = smooth;
                smooth = checkJurbolBBmacd.Smooth;

                if (cacheJurbolBBmacd != null)
                    for (int idx = 0; idx < cacheJurbolBBmacd.Length; idx++)
                        if (Math.Abs(cacheJurbolBBmacd[idx].ArrowOffset - arrowOffset) <= double.Epsilon && cacheJurbolBBmacd[idx].ArrowSize == arrowSize && cacheJurbolBBmacd[idx].Fast == fast && Math.Abs(cacheJurbolBBmacd[idx].NumStdDev - numStdDev) <= double.Epsilon && cacheJurbolBBmacd[idx].Opacity == opacity && cacheJurbolBBmacd[idx].Period == period && cacheJurbolBBmacd[idx].Slow == slow && cacheJurbolBBmacd[idx].Smooth == smooth && cacheJurbolBBmacd[idx].EqualsInput(input))
                            return cacheJurbolBBmacd[idx];

                JurbolBBmacd indicator = new JurbolBBmacd();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.ArrowOffset = arrowOffset;
                indicator.ArrowSize = arrowSize;
                indicator.Fast = fast;
                indicator.NumStdDev = numStdDev;
                indicator.Opacity = opacity;
                indicator.Period = period;
                indicator.Slow = slow;
                indicator.Smooth = smooth;
                Indicators.Add(indicator);
                indicator.SetUp();

                JurbolBBmacd[] tmp = new JurbolBBmacd[cacheJurbolBBmacd == null ? 1 : cacheJurbolBBmacd.Length + 1];
                if (cacheJurbolBBmacd != null)
                    cacheJurbolBBmacd.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheJurbolBBmacd = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Jurbol BBMacd. Combination of Bollinger Bands and Macd. Programmed for NT 7
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.JurbolBBmacd JurbolBBmacd(double arrowOffset, int arrowSize, int fast, double numStdDev, int opacity, int period, int slow, int smooth)
        {
            return _indicator.JurbolBBmacd(Input, arrowOffset, arrowSize, fast, numStdDev, opacity, period, slow, smooth);
        }

        /// <summary>
        /// Jurbol BBMacd. Combination of Bollinger Bands and Macd. Programmed for NT 7
        /// </summary>
        /// <returns></returns>
        public Indicator.JurbolBBmacd JurbolBBmacd(Data.IDataSeries input, double arrowOffset, int arrowSize, int fast, double numStdDev, int opacity, int period, int slow, int smooth)
        {
            return _indicator.JurbolBBmacd(input, arrowOffset, arrowSize, fast, numStdDev, opacity, period, slow, smooth);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Jurbol BBMacd. Combination of Bollinger Bands and Macd. Programmed for NT 7
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.JurbolBBmacd JurbolBBmacd(double arrowOffset, int arrowSize, int fast, double numStdDev, int opacity, int period, int slow, int smooth)
        {
            return _indicator.JurbolBBmacd(Input, arrowOffset, arrowSize, fast, numStdDev, opacity, period, slow, smooth);
        }

        /// <summary>
        /// Jurbol BBMacd. Combination of Bollinger Bands and Macd. Programmed for NT 7
        /// </summary>
        /// <returns></returns>
        public Indicator.JurbolBBmacd JurbolBBmacd(Data.IDataSeries input, double arrowOffset, int arrowSize, int fast, double numStdDev, int opacity, int period, int slow, int smooth)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.JurbolBBmacd(input, arrowOffset, arrowSize, fast, numStdDev, opacity, period, slow, smooth);
        }
    }
}
#endregion
