//
// Index Price of Multiple Instruments - For NinjaTrader 7 & higher
//
// Creates an index from multiple instruments - the weighted sum of the prices, divided by a divisor
//
//    Parameters:
//		SymbolList - a string of comma-separated symbols  (i.e. "DIA,SPY,GLD" or "YM 03-11,ES 03-11,GC 03-11"
//
//		WeightList - a string of comma-separated weights for each symbol (i.e. "0.333,0.507,0.233" or "1,-1" or "1")
//					 Number of weights must match the number of symbols  (exception: if 1 weight is given (i.e. "1"), it will be used for all symbols)
//					 For a price-weighted index such as DJIA, all symbols are evenly weighted at 1.0.
//					 For a market-cap weighted index such as SOX, each symbol is weighted by number of shares.
//
//		Divisor    - a number (double) which is divided into the sum of (( Close of symbol[i]) * (weight of symbol[i])).  A small number for DJIA, big for a market-cap weighted index.
//
//		NoOutputWhenMissingInput - a bool which, if true, causes "No Output" (i.e. blank space on chart) to be generated if any of the inputs are not present for that bar
//								-      but beware that an indicator with missing output cannot be used as input to another indicator (i.e. SMA of the index)
//								- if false (default), the previous bar's output value will be repeated if any input is missing for that bar.
//
//		OldestDataMinutes -	Input is considered missing if older than this many minutes. If set to 0 (Default), Input is considered missing if older than 1 bar. (Default)  
//
//
// Usage Examples:
//		(1) Plot a pair trade spread:  Long 3 INTC, Short 7 AMD
//				SymbolList: INTC,AMD
//				WeightList: 3,-7
//				Divisor: 1
//
//		(2) Plot the spread between two futures months:
//				SymbolList: ES 06-11,ES 09-11
//				WeightList: 1,-1
//				Divisor: 1
//
//		(3) Plot the value of a portfolio:  100 SBUX, 300 ACN, 100 AAPL, 200 CAT, 400 T, 200 GLD
//				SymbolList: SBUX,ACN,AAPL,CAT,T,GLD
//				WeightList: 100,300,100,200,400,200
//				Divisor: 1
//
//	Reproducing a major index is not especially useful because index quotes and history are easily available from data providers.
//  The following examples show how they are created, so you can create your own index that has meaning to you.
//
//		(4) Plot the Dow Jones Industrial Average  (a price-weighted index)
//				SymbolList: MMM,AA,AXP,T,BAC,BA,CAT,CVX,CSCO,KO,DD,XOM,GE,HPQ,HD,INTC,IBM,JNJ,JPM,KFT,MCD,MRK,MSFT,PFE,PG,TRV,UTX,VZ,WMT,DIS
//				WeightList: 1  (DJIA is price weighted, all components have equal weight)
//				Divisor: 0.132129493  (the Dow Divisor as of 7/2/10, see http://www.cmegroup.com/trading/equity-index/files/djia-history-divisor.pdf)
//
//      (5) Plot SOX (a market-cap weighted index, weights are shares outstanding on 2/20/11, shares and divisor are from from https://indexes.nasdaqomx.com/MonthEndFiles.aspx)
//				SymbolList: ALTR,AMAT,AMD,ATHR,AVGO,BRCM,CREE,CRUS,HITT,INTC,KLAC,LLTC,LRCX,MKSI,MRVL,MU,NETL,NSM,NVDA,NVLS,POWI,RBCN,SNDK,STM,TER,TSM,TXN,VECO,WFR,XLNX
//              WeightList: 364951315,2060521585,1396802719,146660859,490415275,575686296,196473483,141628229,63130703,1212702808,342283544,392826431,251960161,103124900,
//                          1105594919,1763931357,130849166,490825711,941628062,184777433,57263098,46993684,287133221,153857427,371654361,1191268991,805417714,81755766,465958033,472202863
//              Divisor: 822180367
//
//
// The Input Series for the indicator (BarsInProgress == 0) is the "clock" which drives the output and to which the other instruments are synchronized.  
// The Input Series is not included in the calculation unless its symbol is in the SymbolList.
// The Input Series must be time-based (Minute,Day, Week, Month, Year, Second).
// If you wish to use this indicator with a non-time-based chart (i.e. volume bars), you must add a time-based series to the chart, and select that as the input series to the indicator.
// You should make sure an active instrument is selected as the Input Series.
//
// Kevin Doren  4/9/2011
//				5/4/2011 - fixes for NT7 compatibility
//

#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

namespace NinjaTrader.Indicator
{
    /// <summary>
    /// Index Price of Two Instruments
    /// </summary>
    [Description("Weighted Average of a List of Symbols")]
    public class Index : Indicator
    {
        #region Variables
			string symbolList = "";
			string weightList = "";
			double divisor = 1;
//          Example: Dow Jones Industrial Average (Dow 30):
//			string symbolList = "MMM,AA,AXP,T,BAC,BA,CAT,CVX,CSCO,KO,DD,XOM,GE,HPQ,HD,INTC,IBM,JNJ,JPM,KFT,MCD,MRK,MSFT,PFE,PG,TRV,UTX,VZ,WMT,DIS";  // List of symbols: Default to DJIA
//			string weightList = "1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1"; // Weights for each symbol: Default DJIA is equally (price) weighted - could just use "1" as the weightList
//			double divisor = 0.132129493;			// Divisor to be appied to index calculation, Default: DJIA divisor as of 7/2/10, see http://en.wikipedia.org/wiki/DJIA_divisor
			string[] symbols;						// array of strings containing the symbols
			int numSymbols;							// number of Symbols  (set to symbols.Length)
			double[] weights;						// array of weights, one for each symbol
			SyncedDataSeries[] syncedSeries;		// array of SyncedDataSeries, one for each symbol 
			const int symbolListBegin = 1;          // BIP Index of first symbol in syncedSeries
			
			bool noOutputWhenMissingInput = false;  // true = produce blank plot if any input is missing a bar
													// BEWARE: this causes problems if this indicator is used as input to another indicator
			double oldestDataMinutes = 0;			// Oldest data to use in index calculation, in minutes (but will not go back to previous day)
			bool outputStarted = false;				// flag which is set to "true" when the indicator starts producing ouput
			bool lastBarNotSynced = false;			// flag which is set to "true" in real time if the last bar had a missing input
			bool inputStarted = false;				// flag which is set to "true" when all inputs have seen data
			bool supportedBarsPeriodType = false;		// set to true once we know this indicator has been applied to time-based bars
        #endregion		
		
		#region SynchronizationMethods
/*
  Synchronization of Multi-Instrument Indicators & Strategies - For NinjaTrader 7
  Kevin Doren - Version 4/9/2011

  7/29/2010:  Added "SynchronizedBarCountToday" field
  9/8/2010:   Added "SyncedGet" method
  4/1/2011:   Added "LastDataTime" field, streamlined & simplified some code, added check for time-based Period
  4/9/2011:   Fixed problem with syncing daily bars with different session end times while COBC=false w/ historical data
			
  Multi-instrument indicators (and strategies, for the same concepts apply) by their nature are looking
  at relationships between instruments.  Any calculations or comparisons we do need to be time-synchronized - we want
  to be using data that occurred at the same time.  For certain kinds of analysis, such as intraday
  analysis of heavily-traded instruments during regular trading hours on historical data, the techniques
  shown here probably aren't necessary.  But for real time analysis, especially with instruments that
  might have missing bars, synchronization techniques are neeeded to avoid the misleading results that
  come from comparing data that occurred at different times when the analysis only has meaning if they occurred at the same time.

  There are 3 scenarios: Historical, real-time with (CalculateOnBarClose == true), 
  and real-time with (CalculateOnBarClose == false).  Each has its issues.

  The simplest case is Historical.  In this case, NT7 will synchronize the bars, which means that when
  OnBarUpdate is called for a Bar Series with a particular timestamp, any other Bar Series which
  has a bar with the same timestamp will have that bar set to be the current bar.  This means that
  Closes[PrimaryInstrument][0] and Closes[SyncedInstrument][0] will both point to bars with the same timestamp IF Closes[1] has a
  bar with that timestamp.  If not, Close[SyncedInstrument][0] will point to the most recent bar preceding it.
  So even Close[PrimaryInstrument][0] and Close[SyncedInstrument][0] are not necessarily synchronized.  As you look back in time,
  it can get even more out of sync, because bars might be missing from either series.

  For an indicator like correlation, the calculation makes no sense unless the bars are all synchronized.
  The solution implemented here is to create 2 additional Data Series, each synchronized to the Primary 
  Instrument, because the indicator output is synced to the Primary Instrument.
  The first (base class) Data Series ("SyncedDataSeries") is a synchronized copy of the secondary instrument, Closes[SyncedInstrument].
  The second Data Series is a BoolSeries field (".Synced") which flags the bars that are synced.
		
  In other words, Closes[PrimaryInstrument][x] and SyncedDataSeries class SyncedDataSeries[x] will refer to the same time,
  and the field SyncedDataSeries.Synced[x] will be set to true if a tick from SyncedInstrument occurred during that bar.

  When operating with Historical data, if a synchronized bar exists, it will be Closes[SyncedInstrument][0].
  In real time, with (CalculateOnBarClose==false), it gets more complicated, because NT7 is event-driven,
  which mean that bars don't close because of time.  They close because a tick arrives with a timestamp of
  a later bar, at which time OnBarUpdate is called and the new bar starts forming.  This could take a long time.
  So when we are processing a bar for the primary instrument, we need to also check the unclosed bar of the secondary
  instrument, which we can do by referencing Times[1][-1]; this is safe to do because we know the bar's time
  has expired - we are running OnBarUpdate for the primary instrument for the same timestamp, which can
  only happen because a tick arrived for a later bar.

  Also, in real time, it's possible that the synced instrument has received bars after the time of the
  bar we are currently processing.  A synchronized bar could exist at Closes[SyncedInstrument][1], Closes[SyncedInstrument][2], or earlier.
  So we need to scan back through the seconday instrument's closes to see if the Synced bar is there.
  
  The (base class) Data Series "SyncedDataSeries" is filled from OnBarUpdate for the primary instrument, and it will hold the
  synchronized bars we found using the above techniques.  We set a flag in the "Synced" data series to show which
  locations are synced.  Routines which require lookbacks can be written to perform their calculations using only synced data.

  There is an additional issue when using daily bars.  As of this writing, daily bars carry a timestamp of the end of the
  NT7 session template.  This can change depending on which session template is applied to the instrument, either in the
  instrument manager or in the chart properties.  There is no issue if both instruments have the same session template,
  then their daily bars will have the same timestamp and thus be synchronized.  But if the secondary instrument has a timestamp
  later than the primary instrument, the days will be out of sync unless we strip away the times and just use the dates.
*/		
		public class SyncedDataSeries : DataSeries
		{
			public IndicatorBase IndBase;
			public int PrimaryInstrument;			// BarsArray Index of primary instrument (usually 0 in indicators)
													//   but could be higher than 0 for multi-instrument, multi-timeframe indicators
			public int SyncedInstrument;			// BarsArray Index of secondary instrument we want to synchronize to primary instrument
			public IDataSeries DataArray;			// Output Data Series synced to primary instrument, hold synchronized copy of secondary instrument
			public BoolSeries Synced;				// BoolSeries output, synced with primary instrument, holds flags
													//   true=output (DataArray) holds synchronized value, false = output does not hold synchronized value
			public int SynchronizedBarCount;		// Total number of synchronized bars held in output, but some or all may be inaccessible if MaximumBarsLookBack is set to 256
			public int SynchronizedBarCountToday;	// Total number of synchronized bars so far today
			public bool DataStarted;
			private bool dailyBars;
//			private bool lastSyncWasToUnclosedBar;
			public DateTime LastDataTime;			// Time of the most recent data received from syncedInstrument
			private DateTime primaryInstrumentCurrentBarTime;
			private DateTime syncedInstrumentCurrentBarTime;
			private int primaryInstrumentCurrentBar;
			private int syncedInstrumentCurrentBar;
			
			public SyncedDataSeries (IndicatorBase indicator, int primaryInstrument, int syncedInstrument, IDataSeries dataArray) :
				this (indicator, primaryInstrument, syncedInstrument, dataArray, indicator.MaximumBarsLookBack)
			{
			}
			public SyncedDataSeries (IndicatorBase indicator, int primaryInstrument, int syncedInstrument, IDataSeries dataArray, MaximumBarsLookBack maxLookback) : base (indicator, maxLookback)
			{
				IndBase = indicator;
				PrimaryInstrument = primaryInstrument;
				SyncedInstrument = syncedInstrument;
				SynchronizedBarCount = 0;
				SynchronizedBarCountToday = 0;
				DataArray = dataArray;
				DataStarted = false;
//				lastSyncWasToUnclosedBar = false;
				LastDataTime = DateTime.MinValue;
				primaryInstrumentCurrentBarTime = DateTime.MinValue;
				syncedInstrumentCurrentBarTime = DateTime.MinValue;
				primaryInstrumentCurrentBar = -1;
				syncedInstrumentCurrentBar = -1;
				
				switch (IndBase.BarsPeriods[PrimaryInstrument].Id)
				{
					case PeriodType.Day:
					case PeriodType.Week:
					case PeriodType.Month:
					case PeriodType.Year:
						dailyBars = true;
						break;
					case PeriodType.Minute:
					case PeriodType.Second:
					{
						dailyBars = false;
						break;
					}
					default:
						throw new ArgumentException("SyncedDataSeries: Instrument Period must be time-based (Minute,Day,Week,Month,Year,or Second).");
						break;
				}
				if (IndBase.BarsPeriods[PrimaryInstrument].Id != IndBase.BarsPeriods[SyncedInstrument].Id)
					throw new ArgumentException("SyncedDataSeries: Primary and Synced Intruments must both have the same Period Type.");
				if (IndBase.BarsPeriods[PrimaryInstrument].Value != IndBase.BarsPeriods[SyncedInstrument].Value)
					throw new ArgumentException("SyncedDataSeries: Primary and Synced Intruments must both have the same Period Value.");
				
				if (PrimaryInstrument == 0)
					Synced = new BoolSeries(IndBase);  // We are syncing to the primary instrument of the instantiating indicator
				else
					throw new ArgumentOutOfRangeException ("PrimaryInstrument", "SyncedDataSeries: PrimaryInstrument must = 0 if no SyncedIndicator base is given.");
			}
	
			public SyncedDataSeries (IndicatorBase indicator, int primaryInstrument, int syncedInstrument, IDataSeries dataArray, IndicatorBase syncedIndicator) :
				this (indicator, primaryInstrument, syncedInstrument, dataArray, syncedIndicator, indicator.MaximumBarsLookBack)
			{
			}
			public SyncedDataSeries (IndicatorBase indicator, int primaryInstrument, int syncedInstrument, IDataSeries dataArray, IndicatorBase syncedIndicator, MaximumBarsLookBack maxLookback) :base (syncedIndicator, maxLookback)
			{	
				IndBase = indicator;
				PrimaryInstrument = primaryInstrument;
				SyncedInstrument = syncedInstrument;
				SynchronizedBarCount = 0;
				SynchronizedBarCountToday = 0;
				DataArray = dataArray;
				DataStarted = false;
//				lastSyncWasToUnclosedBar = false;
				LastDataTime = DateTime.MinValue;
				primaryInstrumentCurrentBarTime = DateTime.MinValue;
				syncedInstrumentCurrentBarTime = DateTime.MinValue;
				primaryInstrumentCurrentBar = -1;
				syncedInstrumentCurrentBar = -1;
				
				switch (IndBase.BarsPeriods[PrimaryInstrument].Id)
				{
					case PeriodType.Day:
					case PeriodType.Week:
					case PeriodType.Month:
					case PeriodType.Year:
						dailyBars = true;
						break;
					case PeriodType.Minute:
					case PeriodType.Second:
					{
						dailyBars = false;
						break;
					}
					default:
						throw new ArgumentException("SyncedDataSeries: Instrument Period must be time-based (Minute,Day,Week,Month,Year,or Second).");
						break;
				}
				if (IndBase.BarsPeriods[PrimaryInstrument].Id != IndBase.BarsPeriods[SyncedInstrument].Id)
					throw new ArgumentException("SyncedDataSeries: Primary and Synced Intruments must both have the same Period Type.");
				if (IndBase.BarsPeriods[PrimaryInstrument].Value != IndBase.BarsPeriods[SyncedInstrument].Value)
					throw new ArgumentException("SyncedDataSeries: Primary and Synced Intruments must both have the same Period Value.");
				
				Synced = new BoolSeries(syncedIndicator);	// Not syncing to primary instrument of instantiating indicator;
															// So create a new BoolSeries synced to the master instrument
			}
			
			public double SyncedGet(int syncedBarsBack)
			{
				int i;
				if (syncedBarsBack >= SynchronizedBarCount)
					return 0;			// not enough synchronized bars - return 0 rather than throw exception
				return SyncedGet(syncedBarsBack, 0, out i);
			}
			
			public double SyncedGet(int syncedBarsBack, int initIndex, out int index)
			{
				index = initIndex;
				while (!Synced[index])  //Find most recent synced Bar
					index++;
				
				for (int counter=0; counter < Math.Min(syncedBarsBack,SynchronizedBarCount); counter++)
				{  //looking back past the first synced bar, keep looking for more
					index++;
					while (!Synced[index])  //Find previous synced Bar
						index++;
				}
				return base[index];
			}
			
			private void setBarAsSynced(int getDataFromBarsBack)
			{
				Synced.Set(true);
				SynchronizedBarCount++;
				SynchronizedBarCountToday++;
				LastDataTime = IndBase.Times[SyncedInstrument][getDataFromBarsBack];
				base.Set(DataArray[getDataFromBarsBack]);
			}
			
			public void Synchronize()
			{
				if ((IndBase.BarsInProgress != PrimaryInstrument) && (IndBase.BarsInProgress != SyncedInstrument))
						return;	// Bars being processed are not for us
				
				if (IndBase.CalculateOnBarClose == true)
				{
					if (IndBase.BarsInProgress == PrimaryInstrument) // Primary Instrument
					{
						primaryInstrumentCurrentBarTime = dailyBars ? IndBase.Times[PrimaryInstrument][0].Date : IndBase.Times[PrimaryInstrument][0];
						syncedInstrumentCurrentBar = IndBase.BarsArray[SyncedInstrument].CurrentBar;
						
						if (!DataStarted)
						{
							if (syncedInstrumentCurrentBar == -1)
							{
								Synced.Set(false);
								return;						// return if no data yet from synced instrument
							}
							else
								DataStarted = true;
						}
						if (IndBase.BarsArray[PrimaryInstrument].FirstBarOfSession)
							SynchronizedBarCountToday = 0;	// it's a new session, zero today's synced bar count
							
						//	Scan through SyncedInstrument's bars, try to find one in sync with the current Primary instrument bar
						//	Start the scan with the unclosed bar [-1] (necessary in real time); if it has the same time as the Primary instrument's Time[0] (which is now closed), it should be done forming
						int barsBack;
						if (syncedInstrumentCurrentBar == IndBase.BarsArray[SyncedInstrument].Count-1)
							barsBack = 0;
						else
							barsBack = -1;
						
						while ((barsBack < syncedInstrumentCurrentBar) && primaryInstrumentCurrentBarTime < (dailyBars ? IndBase.Times[SyncedInstrument][barsBack].Date : IndBase.Times[SyncedInstrument][barsBack]))
								barsBack ++;  // ignore bars that occur in the future (this can happen in real time if primary instrument is missing bars)
						if (primaryInstrumentCurrentBarTime == (dailyBars ? IndBase.Times[SyncedInstrument][barsBack].Date : IndBase.Times[SyncedInstrument][barsBack]))
							{	// Found a synchronized bar
								setBarAsSynced(barsBack);
//								if (barsBack == -1)
//									lastSyncWasToUnclosedBar = true;
							}
						else	// No synchronized bar found
							{
								Synced.Set(false);
								base.Set(DataArray[barsBack]);	// set output to most recent bar that isn't in the future
								LastDataTime = IndBase.Times[SyncedInstrument][barsBack];
							}
					}
					
					else // (IndBase.BarsInProgress == SyncedInstrument)
					{
						// Normally we don't have to do anything when (BarsInProgress == SyncedInstrument) if (CalculateOnBarClose == true)
						// Method output is synced with, and handled by, primary instrument when (BarsInProgress == PrimaryInstrument)
						//
						// Sometimes in a real-time OnBarUpdate for PrimaryIntrument, we find that SyncedInstrument has an unclosed bar (index of [-1]) with the same timestamp,
						// so we use it, which is safe to do because PrimaryInstrument's bar has already closed, so we know that the end of the bar period has passed.
						//
						if (!DataStarted)
						{
							if (IndBase.BarsArray[PrimaryInstrument].CurrentBar == -1)
								return;							// return if no data yet from Primary instrument
							else
								DataStarted = true;
						}	
//						if (lastSyncWasToUnclosedBar)
//						{
//							lastSyncWasToUnclosedBar = false;   // if there were any late arriving ticks, they are now included in the bar that just closed
//							int barsBack=0;
//							while (!Synced[barsBack])			// Scan for most recent synced bar
//								barsBack++;
//							base.Set(barsBack,DataArray[0]);	// Store the final value, in case it has changed since we stored the value from the unclosed bar
//						}
					}
				}
				else  // (CalculateOnBarClose == false)
				{
					if (IndBase.BarsInProgress == PrimaryInstrument) // Primary Instrument
					{
						if (!DataStarted)
						{	
							if (syncedInstrumentCurrentBar == -1)
							{
								Synced.Set(false);
								base.Reset();
								return;							// return if no data yet from synced instrument
							}
							else
								DataStarted = true;
						}		
						if (IndBase.FirstTickOfBar)				// First tick of bar, need to set state of sync flag
						{
							primaryInstrumentCurrentBarTime = dailyBars ? IndBase.Times[PrimaryInstrument][0].Date : IndBase.Times[PrimaryInstrument][0];
							primaryInstrumentCurrentBar = IndBase.BarsArray[PrimaryInstrument].CurrentBar;
							syncedInstrumentCurrentBar = IndBase.BarsArray[SyncedInstrument].CurrentBar;
							syncedInstrumentCurrentBarTime = dailyBars ? IndBase.Times[SyncedInstrument][0].Date : IndBase.Times[SyncedInstrument][0];

							if (IndBase.BarsArray[PrimaryInstrument].FirstBarOfSession)
								SynchronizedBarCountToday = 0;	// it's a new session, zero today's synced bar count
							
							//	When synchronizing daily bars if syncedInstrument has a later session end time than primaryInstrument
							//  if COBC == false, when processing historical data, OnBarUpdate will be called for primaryInstrument
							//  while the syncedInstrument shows the previous day in its current bar.  We have to peek ahead to
							//  the next bar (-1 index) to see if the date matches
							if (   (dailyBars == true)  
								&& (syncedInstrumentCurrentBar != IndBase.BarsArray[SyncedInstrument].Count-1) 
								&& (primaryInstrumentCurrentBarTime == IndBase.Times[SyncedInstrument][-1].Date))
							{
								setBarAsSynced(-1);
								return;
							}
							
							if (primaryInstrumentCurrentBarTime == syncedInstrumentCurrentBarTime)
								setBarAsSynced(0);
							else
							{
								if ((syncedInstrumentCurrentBar > 0) && (primaryInstrumentCurrentBarTime == (dailyBars ? IndBase.Times[SyncedInstrument][1].Date : IndBase.Times[SyncedInstrument][1])))
									setBarAsSynced(1);
								else
								{
									Synced.Set(false);
									base.Set(DataArray[0]);  // store most recent value from prior bars, in case no updates come for rest of bar
									LastDataTime = syncedInstrumentCurrentBarTime;
								}
							}
						}
					}
					else // (IndBase.BarsInProgress == SyncedInstrument)
					{
						if (syncedInstrumentCurrentBar != IndBase.BarsArray[SyncedInstrument].CurrentBar)
						{	// if SyncedInstrument's CurrentBar has changed, it's the first tick of the bar, need to save CurrentBar and Time
							syncedInstrumentCurrentBar = IndBase.BarsArray[SyncedInstrument].CurrentBar;
							syncedInstrumentCurrentBarTime = dailyBars ? IndBase.Times[SyncedInstrument][0].Date : IndBase.Times[SyncedInstrument][0];
						}
						
						if (!DataStarted)
						{
							if (IndBase.BarsArray[PrimaryInstrument].CurrentBar == -1)
								return;						// return if no data yet from primary instrument
							else
								DataStarted = true;
						}
//						if (primaryInstrumentCurrentBarTime == syncedInstrumentCurrentBarTime)
						if (((!dailyBars) && (IndBase.Times[SyncedInstrument][0] == IndBase.Times[PrimaryInstrument][0]))
							|| (dailyBars && (IndBase.ToDay(IndBase.Times[SyncedInstrument][0]) == IndBase.ToDay(IndBase.Times[PrimaryInstrument][0]))))
						{
							if (!Synced[0])
								setBarAsSynced(0);
							else
								base.Set(DataArray[0]);
						}
					}
				}
			}
		}
		#endregion

		private bool allInputsSynced(int barsBack)
		{	// output is valid only when called with (BarsInProgress == 0)
			for (int i=0; i < numSymbols; i++)
			{
				if (!syncedSeries[i].Synced[barsBack])		// Return false if any input isn't synced
				{
					if (OldestDataMinutes == 0)				// If OldestDataMinutes = 0 it's not synced
						return false;
					else									// otherwise call it synced if most recent data arrived within that time
						if (syncedSeries[i].LastDataTime.AddMinutes(OldestDataMinutes) <  Times[0][barsBack])	
							return false;
				}
			}
			return true;
		}
		
		private bool allInputsOpenedToday()
		{	// output is valid only when called with (BarsInProgress == 0)
			for (int i=0; i < numSymbols; i++)
			{
				if (syncedSeries[i].SynchronizedBarCountToday == 0)			// Return false if any input hasn't opened today
					return false;
			}
			return true;
		}
		
		private void produceMissingOutput(int barsBack)
		{
			// We are here if one of the inputs skipped a bar.
			// We have 3 options:   (1) Do the calculation anyway with the most recent data
			//						(2) repeat the last indicator output
			//						(3) we could also send "no output"  ( Value.Reset(0) ) but that causes problems if this indicator is input to another indicator 
			
			// If we just do (1) the calculation using the most recent data; that could produce (very) misleading output because the most recent date could be (very) old
			
			if (NoOutputWhenMissingInput)
				Value.Reset(barsBack);										// Send "No Output" for the bar (be sure you never use this indicator as input to another indicator!
			else
			{	// Repeat the previous output if it exists
				if (outputStarted && (CurrentBar >= barsBack)) 
				{
					if (Value.ContainsValue(barsBack+1))
						Value.Set(barsBack,Value[barsBack+1]);				// repeat the previous output
					else  // previous bar contained nothing, do the same here
						Value.Reset(barsBack);
				}
			}
		}
		
		private double indexCalc(int barsBack)
		{
			double sum = 0;
			for (int i=0; i < numSymbols; i++)
			{
				sum += syncedSeries[i][barsBack] * weights[i];				// add up the current synchronized closing prices multiplied by weights
			}
			return sum / Divisor;											// divide by divisor
		}
		
		protected override void Initialize()
        {
			BarsRequired = 0;  												// OK to plot on first bar
			symbols = SymbolList.Split(',');								// create new array of symbols by Comma-Separating the list of symbols
			numSymbols = (symbolList == "") ? 0 : symbols.Length;				// set number of symbols
			
			Add(new Plot(Color.FromKnownColor(KnownColor.ForestGreen), PlotStyle.Line, "Index"));
			
			switch (BarsPeriod.Id)
			{
				case PeriodType.Day:
				case PeriodType.Week:
				case PeriodType.Month:
				case PeriodType.Year:
				case PeriodType.Minute:
				case PeriodType.Second:
					supportedBarsPeriodType = true;
					for (int i=0; i < numSymbols; i++)
					{
						symbols[i] = symbols[i].Trim();
						Add(symbols[i], BarsPeriod.Id, BarsPeriod.Value);
					}
					break;
				default:
					break;
			}
		}

		protected override void OnStartUp()
		{
			if (!supportedBarsPeriodType)
				throw new ArgumentException("Input series Period must be time-based (Minute,Day,Week,Month,Year,or Second).");
			
			weights = new double[numSymbols];								// create new array with 1 weight for each symbol
			string[] weightStringArray = WeightList.Split(',');				// create new array of weights (strings) by Comma-Separating the list of weights
			if    ((weightStringArray.Length != numSymbols) 			// check for same number of symbols and weights
				&& (weightStringArray.Length != 1))						// OK if only 1 weight, we will use it for all symbols
				{
					throw new ArgumentException (string.Format("Unequal number of symbols & weights - '{0}' Symbols and '{1}' Weights ", symbols.Length, weightStringArray.Length), "SymbolList or WeightList");
				}
			else
				for (int i=0; i < numSymbols; i++)
				{
					try {
						weights[i] = Convert.ToDouble(weightStringArray[Math.Min(i,weightStringArray.Length-1)]);		// If only 1 weight given, use it for all symbols 
					}   
					catch (FormatException) {
						throw new ArgumentException (string.Format("Unable to convert '{0}' to a Double.", weightStringArray[i]), "WeightList");
					}               
					catch (OverflowException) {
						throw new ArgumentOutOfRangeException (string.Format("'{0}' is outside the range of a Double.", weightStringArray[i]), "WeightList");
					}
				}
			
			syncedSeries = new SyncedDataSeries[numSymbols];			// instantiate array of SyncedDataSeries, one for eash instrument
			for (int i=0; i < numSymbols; i++)							// instantiate each SyncedDataSerie in the array
			{
				syncedSeries[i] = new SyncedDataSeries(this, 0, i+symbolListBegin, Closes[i+symbolListBegin]);		// sync each series to the primary instrument
			}
		}
		
        protected override void OnBarUpdate()
        {	
			if (BarsInProgress != 0)
			{
				syncedSeries[BarsInProgress - symbolListBegin].Synchronize();	// Secondary instrument - call synchronization method just for this instrument (subtract 1 from BarsInProgress to get index)
				if (outputStarted && (CalculateOnBarClose == false) && (!Historical) && allInputsOpenedToday() && allInputsSynced(0))
					Value.Set(indexCalc(0));		// if real-time and COBC==false, calculate output on every tick of index components
				return;
			}

			// BarsInProgress == 0
			for (int i=0; i < numSymbols; i++)
			{
				syncedSeries[i].Synchronize();							// Primary instrument - call synchronization method for each instrument in list
			}
			
			if (!inputStarted)
			{
				for (int i=0; i < numSymbols; i++)
				{
					if (!syncedSeries[i].DataStarted)					// Return if we haven't seen data from ALL instruments
						return;
				}
				inputStarted = true;									// flag that all instruments have started sending data so we don't have to check again
			}

			if (CalculateOnBarClose == true)
			{
				if (allInputsOpenedToday() && allInputsSynced(0))
				{
					Value.Set(indexCalc(0));
					outputStarted = true;
				}
				else
					produceMissingOutput(0);
			}
			else  //  (CalculateOnBarClose == false)
			{
				//  We are here if we are processing every tick of the primary instrument (because we want the chart to update in real time)
				//  Current data for symbol "i" is held in syncedSeries[i][0]
				//
				//  Handling missing bars is more complicated in real time:
				//  Since we are processing every tick, we can't know if an instrument has skipped a bar until the bar is over.
				//
				//  Each SyncedDataSeries has a field named "SynchronizedBarCountToday" which tells how many synchronized bars it has seen today.
				//  A Value of 0 means no bars yet today (i.e. it hasn't opened).
				
				if ((!outputStarted) && (!allInputsSynced(0)))
						return;					// don't process anything until we've seen at least one synchronized bar
				
				if (!allInputsOpenedToday()) 
					{ // not all instruments have opened yet
						produceMissingOutput(0); 
						return; 
					}

				if (Historical)
				{	// Historical data, we missed a bar if the current bar wasn't synced
					if  (!allInputsSynced(0))
					{ 	// Missing an input in current bar, produce "Missing Output" until we see some synchronized data
						produceMissingOutput(0);
						return; 
					}
				}
				else // !Historical
				{	// Real-time tick data, we only know we missed a bar if the previous bar wasn't synced
					if (FirstTickOfBar)
					{   
						lastBarNotSynced = !allInputsSynced(1);
						if  (lastBarNotSynced)
							produceMissingOutput(1);	// Previous bar had a missing input, set it to "Missing Output"
					}
					if (lastBarNotSynced && (!allInputsSynced(0)))
					{	// No data in previous bar or current bar, produce "Missing Output" until we see some synchronized data
							produceMissingOutput(0);
							return; 
					}
				}

				//  The code below will use the last Close value of every instrument, no matter how old it is, if we get here without checking for missing bars.
				Value.Set(indexCalc(0));
				outputStarted = true;
			}
        }
		
		public override string ToString()
		{
				return Name + "(\"" + SymbolList + "\",\"" + WeightList + "\"," + Divisor + "," + NoOutputWhenMissingInput.ToString() + "," + OldestDataMinutes.ToString() + ")"; 
		}

        #region Properties

		
		[Description("SymbolList (Comma-Separated)\nExamples: ")]
        [GridCategory("Parameters")]
		[Gui.Design.DisplayName ("\t\t\t\t\t\t\tSymbolList")]
        public string SymbolList
        {
			get { return symbolList; }
            set { symbolList = value.ToUpper(); }
		}
		
		[Description("WeightList (Comma-Separated): 1 for each Symbol; Negative values are OK.\nIf there is only 1 weight, it is used for all symbols.")]
        [GridCategory("Parameters")]
		[Gui.Design.DisplayName ("\t\t\t\t\t\tWeightList")]
        public string WeightList
        {
			get { return weightList; }
            set { weightList = value; }
		}
		
		[Description("Divisor\nIndicator Output = (sum of Symbol[i] * Weight[i]) / Divisor")]
        [GridCategory("Parameters")]
		[Gui.Design.DisplayName ("\t\t\t\t\tDivisor")]
        public double Divisor
        {
            get { return divisor; }
            set { divisor = value; }
        }
		
		[Description("False: Repeat last output if missing an input. (Default)\nTrue: No Plot if missing an input (Don't use as input to another indicator)")]
        [GridCategory("Parameters")]
		[Gui.Design.DisplayName ("\t\t\t\tNoOutputWhenMissingInput")]
        public bool NoOutputWhenMissingInput
        {
            get { return noOutputWhenMissingInput; }
            set { noOutputWhenMissingInput = value; }
        }
		
		[Description("Input is considered missing if older than this many minutes.\n0 = Input is considered missing if older than 1 bar. (Default)")]
        [GridCategory("Parameters")]
		[Gui.Design.DisplayName ("\t\t\tOldestDataMinutes")]
        public double OldestDataMinutes
        {
            get { return oldestDataMinutes; }
            set { oldestDataMinutes = Math.Max(0, value); }
        }
		
        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private Index[] cacheIndex = null;

        private static Index checkIndex = new Index();

        /// <summary>
        /// Weighted Average of a List of Symbols
        /// </summary>
        /// <returns></returns>
        public Index Index(double divisor, bool noOutputWhenMissingInput, double oldestDataMinutes, string symbolList, string weightList)
        {
            return Index(Input, divisor, noOutputWhenMissingInput, oldestDataMinutes, symbolList, weightList);
        }

        /// <summary>
        /// Weighted Average of a List of Symbols
        /// </summary>
        /// <returns></returns>
        public Index Index(Data.IDataSeries input, double divisor, bool noOutputWhenMissingInput, double oldestDataMinutes, string symbolList, string weightList)
        {
            if (cacheIndex != null)
                for (int idx = 0; idx < cacheIndex.Length; idx++)
                    if (Math.Abs(cacheIndex[idx].Divisor - divisor) <= double.Epsilon && cacheIndex[idx].NoOutputWhenMissingInput == noOutputWhenMissingInput && Math.Abs(cacheIndex[idx].OldestDataMinutes - oldestDataMinutes) <= double.Epsilon && cacheIndex[idx].SymbolList == symbolList && cacheIndex[idx].WeightList == weightList && cacheIndex[idx].EqualsInput(input))
                        return cacheIndex[idx];

            lock (checkIndex)
            {
                checkIndex.Divisor = divisor;
                divisor = checkIndex.Divisor;
                checkIndex.NoOutputWhenMissingInput = noOutputWhenMissingInput;
                noOutputWhenMissingInput = checkIndex.NoOutputWhenMissingInput;
                checkIndex.OldestDataMinutes = oldestDataMinutes;
                oldestDataMinutes = checkIndex.OldestDataMinutes;
                checkIndex.SymbolList = symbolList;
                symbolList = checkIndex.SymbolList;
                checkIndex.WeightList = weightList;
                weightList = checkIndex.WeightList;

                if (cacheIndex != null)
                    for (int idx = 0; idx < cacheIndex.Length; idx++)
                        if (Math.Abs(cacheIndex[idx].Divisor - divisor) <= double.Epsilon && cacheIndex[idx].NoOutputWhenMissingInput == noOutputWhenMissingInput && Math.Abs(cacheIndex[idx].OldestDataMinutes - oldestDataMinutes) <= double.Epsilon && cacheIndex[idx].SymbolList == symbolList && cacheIndex[idx].WeightList == weightList && cacheIndex[idx].EqualsInput(input))
                            return cacheIndex[idx];

                Index indicator = new Index();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.Divisor = divisor;
                indicator.NoOutputWhenMissingInput = noOutputWhenMissingInput;
                indicator.OldestDataMinutes = oldestDataMinutes;
                indicator.SymbolList = symbolList;
                indicator.WeightList = weightList;
                Indicators.Add(indicator);
                indicator.SetUp();

                Index[] tmp = new Index[cacheIndex == null ? 1 : cacheIndex.Length + 1];
                if (cacheIndex != null)
                    cacheIndex.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheIndex = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Weighted Average of a List of Symbols
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.Index Index(double divisor, bool noOutputWhenMissingInput, double oldestDataMinutes, string symbolList, string weightList)
        {
            return _indicator.Index(Input, divisor, noOutputWhenMissingInput, oldestDataMinutes, symbolList, weightList);
        }

        /// <summary>
        /// Weighted Average of a List of Symbols
        /// </summary>
        /// <returns></returns>
        public Indicator.Index Index(Data.IDataSeries input, double divisor, bool noOutputWhenMissingInput, double oldestDataMinutes, string symbolList, string weightList)
        {
            return _indicator.Index(input, divisor, noOutputWhenMissingInput, oldestDataMinutes, symbolList, weightList);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Weighted Average of a List of Symbols
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.Index Index(double divisor, bool noOutputWhenMissingInput, double oldestDataMinutes, string symbolList, string weightList)
        {
            return _indicator.Index(Input, divisor, noOutputWhenMissingInput, oldestDataMinutes, symbolList, weightList);
        }

        /// <summary>
        /// Weighted Average of a List of Symbols
        /// </summary>
        /// <returns></returns>
        public Indicator.Index Index(Data.IDataSeries input, double divisor, bool noOutputWhenMissingInput, double oldestDataMinutes, string symbolList, string weightList)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.Index(input, divisor, noOutputWhenMissingInput, oldestDataMinutes, symbolList, weightList);
        }
    }
}
#endregion
