using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;

namespace NinjaTrader.Indicator
{
    public class i22m01: Indicator
    {
		const int MAX_DRAW_COUNT = 4000;
		float[] dashPattern = {1, 5};
		float radiusStep = 2;
		string alertFilenameBid = "", alertFilenameAsk = "";

		int drawCount = 0, length = 1, drawBarsAgo = 0, tradesAmount = 50;
		int[] volumeLevels = {0, 200, 250, 300, 350};
		Color colorAboveAsk = Color.Red, colorUnderBid = Color.Green;
		
		DrawObject[] drawObjects = new DrawObject[MAX_DRAW_COUNT];

		ILine line;
		
		i21 volumes;
		
		class DrawObject {
			private DateTime time;
			private double price;
			private SolidBrush brush;
			private int level;
			
			public DrawObject(DateTime time, double price, SolidBrush brush, int level) {
				this.time = time;
				this.price = price;
				this.brush = brush;
				this.level = level;
			}
			
			public DateTime Time {
				get { return time; }
			}

			public double Price {
				get { return price; }
			}

			public SolidBrush Brush {
				get { return brush; }
			}
			
			public int Level {
				get { return level; }
			}
		}
		
		protected override void Initialize()
		{
			Overlay = true;

			Add(PeriodType.Custom2, 1);
		}

		protected override void OnStartUp()
		{
			volumes = i21(Inputs[1], tradesAmount);

			if (CalculateOnBarClose)
				drawBarsAgo = -1;
		}

		double getIfContain(DataSeries data, int barsAgo)
		{
			return data.ContainsValue(barsAgo) ? data[barsAgo] : 0;
		}

		double getIfContain(DataSeries data)
		{
			return getIfContain(data, 0);
		}
		
		int getVolumeLevel(double volume) {
			for (int i = 1; i < volumeLevels.Length; i++)
				if (volume < volumeLevels[i])
					return i;
			
			return volumeLevels.Length - 1;
		}
		
		void inform(double volume, Color color) {
			int level = getVolumeLevel(volume);
			line = DrawLine(String.Format("Line {0}", drawCount),
				false, drawBarsAgo, Close[0], -length, Close[0],
				color, DashStyle.Custom, 1);
			line.Pen.DashPattern = dashPattern;
			//DrawText(String.Format("Text {0}", drawCount),
				//volume.ToString(), drawBarsAgo + level,
				//Close[0] + TickSize * 2, Color.Black);

			if (color == colorUnderBid && alertFilenameBid != "")
				Alert(Name + " alert", NinjaTrader.Cbi.Priority.Medium,
					volume.ToString(), alertFilenameBid, 10, Color.White, Color.Black);
			else if (color == colorAboveAsk && alertFilenameAsk != "")
				Alert(Name + " alert", NinjaTrader.Cbi.Priority.Medium,
					volume.ToString(), alertFilenameAsk, 10, Color.White, Color.Black);

			drawObjects[drawCount++] = new i22m01.DrawObject(
				Time[0], Close[0], new SolidBrush(Color.FromArgb(150, color)), level);
		}

		public override void Plot(Graphics graphics, Rectangle bounds, double min, double max)
		{
			base.Plot(graphics, bounds, min, max);
			
			SmoothingMode oldSmoothingMode = graphics.SmoothingMode;
			graphics.SmoothingMode = SmoothingMode.AntiAlias;
			for (int i = 0; i < MAX_DRAW_COUNT; i++)
				if (drawObjects[i] != null) {
					float x = ChartControl.GetXByBarIdx(Bars, Bars.GetBar(drawObjects[i].Time));
					float y = ChartControl.GetYByValue(Bars, drawObjects[i].Price);
					graphics.FillEllipse(drawObjects[i].Brush,
						x - radiusStep * drawObjects[i].Level, y - radiusStep * drawObjects[i].Level,
						radiusStep * 2 * drawObjects[i].Level, radiusStep * 2 * drawObjects[i].Level);
				}
			graphics.SmoothingMode = oldSmoothingMode;
		}

		protected override void OnBarUpdate()
		{
			if (Historical)
				return;
			
			if (BarsInProgress == 1) {
				Dictionary<double, Color> volumesWithColor = new Dictionary<double, Color>();
				volumesWithColor[getIfContain(volumes.AskVolume)] = colorAboveAsk;
				volumesWithColor[getIfContain(volumes.BidVolume)] = colorUnderBid;
				
				double v = volumesWithColor.Keys.Max();
				if (v > 0) {
					Color c = volumesWithColor[v];
					inform(v, c);
				}

				if (drawCount > MAX_DRAW_COUNT)
					drawCount = 0;
			}
		}
		
        [XmlIgnore()]
        [GridCategory("Параметры")]
		[Gui.Design.DisplayNameAttribute("Цвет области, Ask")]
		[Description("Выброс на цене равной или выше Ask")]
		public Color ColorAboveAsk
		{
			get { return colorAboveAsk; }
			set { colorAboveAsk = (Color) value; }
		}

		[Browsable(false)]
		public string ColorAboveAskSerialize
		{
			get { return Gui.Design.SerializableColor.ToString(colorAboveAsk); }
			set { colorAboveAsk = Gui.Design.SerializableColor.FromString(value); }
		}

        [XmlIgnore()]
        [GridCategory("Параметры")]
		[Gui.Design.DisplayNameAttribute("Цвет области, Bid")]
		[Description("Выброс на цене равной или ниже Bid")]
		public Color ColorUnderBid
		{
			get { return colorUnderBid; }
			set { colorUnderBid = (Color) value; }
		}

		[Browsable(false)]
		public string ColorUnderBidSerialize
		{
			get { return Gui.Design.SerializableColor.ToString(colorUnderBid); }
			set { colorUnderBid = Gui.Design.SerializableColor.FromString(value); }
		}

		[GridCategory("Параметры")]
		[Gui.Design.DisplayNameAttribute("1 уровень объёма")]
		public int VolumeLevel1
		{
			get { return volumeLevels[1]; }
			set { volumeLevels[1] = Math.Max(1, value); }
		}

        [GridCategory("Параметры")]
		[Gui.Design.DisplayNameAttribute("2 уровень объёма")]
		public int VolumeLevel2
		{
			get { return volumeLevels[2]; }
			set { volumeLevels[2] = Math.Max(1, value); }
		}

        [GridCategory("Параметры")]
		[Gui.Design.DisplayNameAttribute("3 уровень объёма")]
		public int VolumeLevel3
		{
			get { return volumeLevels[3]; }
			set { volumeLevels[3] = Math.Max(1, value); }
		}

        [GridCategory("Параметры")]
		[Gui.Design.DisplayNameAttribute("4 уровень объёма")]
		public int VolumeLevel4
		{
			get { return volumeLevels[4]; }
			set { volumeLevels[4] = Math.Max(1, value); }
		}

		[GridCategory("Параметры")]
		[Gui.Design.DisplayNameAttribute("Длина линии")]
		public int Length
		{
			get { return length; }
			set { length = Math.Max(1, value); }
		}

        [GridCategory("Параметры")]
		[Gui.Design.DisplayNameAttribute("Звук оповещения, Bid")]
		[Description("Имя wav-файла для оповещения")]
		public string AlertFilenameBid
		{
			get { return alertFilenameBid; }
			set { alertFilenameBid = value; }
		}

        [GridCategory("Параметры")]
		[Gui.Design.DisplayNameAttribute("Звук оповещения, Ask")]
		[Description("Имя wav-файла для оповещения")]
		public string AlertFilenameAsk
		{
			get { return alertFilenameAsk; }
			set { alertFilenameAsk = value; }
		}

		[GridCategory("Параметры")]
		[Gui.Design.DisplayNameAttribute("Количество сделок")]
		[Description("Количество идущих подряд сделок")]
		public int TradesAmount
		{
			get { return tradesAmount; }
			set { tradesAmount = Math.Max(1, value); }
		}

	}
}


#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private i22m01[] cachei22m01 = null;

        private static i22m01 checki22m01 = new i22m01();

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public i22m01 i22m01(string alertFilenameAsk, string alertFilenameBid, Color colorAboveAsk, Color colorUnderBid, int length, int tradesAmount, int volumeLevel1, int volumeLevel2, int volumeLevel3, int volumeLevel4)
        {
            return i22m01(Input, alertFilenameAsk, alertFilenameBid, colorAboveAsk, colorUnderBid, length, tradesAmount, volumeLevel1, volumeLevel2, volumeLevel3, volumeLevel4);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public i22m01 i22m01(Data.IDataSeries input, string alertFilenameAsk, string alertFilenameBid, Color colorAboveAsk, Color colorUnderBid, int length, int tradesAmount, int volumeLevel1, int volumeLevel2, int volumeLevel3, int volumeLevel4)
        {
            if (cachei22m01 != null)
                for (int idx = 0; idx < cachei22m01.Length; idx++)
                    if (cachei22m01[idx].AlertFilenameAsk == alertFilenameAsk && cachei22m01[idx].AlertFilenameBid == alertFilenameBid && cachei22m01[idx].ColorAboveAsk == colorAboveAsk && cachei22m01[idx].ColorUnderBid == colorUnderBid && cachei22m01[idx].Length == length && cachei22m01[idx].TradesAmount == tradesAmount && cachei22m01[idx].VolumeLevel1 == volumeLevel1 && cachei22m01[idx].VolumeLevel2 == volumeLevel2 && cachei22m01[idx].VolumeLevel3 == volumeLevel3 && cachei22m01[idx].VolumeLevel4 == volumeLevel4 && cachei22m01[idx].EqualsInput(input))
                        return cachei22m01[idx];

            lock (checki22m01)
            {
                checki22m01.AlertFilenameAsk = alertFilenameAsk;
                alertFilenameAsk = checki22m01.AlertFilenameAsk;
                checki22m01.AlertFilenameBid = alertFilenameBid;
                alertFilenameBid = checki22m01.AlertFilenameBid;
                checki22m01.ColorAboveAsk = colorAboveAsk;
                colorAboveAsk = checki22m01.ColorAboveAsk;
                checki22m01.ColorUnderBid = colorUnderBid;
                colorUnderBid = checki22m01.ColorUnderBid;
                checki22m01.Length = length;
                length = checki22m01.Length;
                checki22m01.TradesAmount = tradesAmount;
                tradesAmount = checki22m01.TradesAmount;
                checki22m01.VolumeLevel1 = volumeLevel1;
                volumeLevel1 = checki22m01.VolumeLevel1;
                checki22m01.VolumeLevel2 = volumeLevel2;
                volumeLevel2 = checki22m01.VolumeLevel2;
                checki22m01.VolumeLevel3 = volumeLevel3;
                volumeLevel3 = checki22m01.VolumeLevel3;
                checki22m01.VolumeLevel4 = volumeLevel4;
                volumeLevel4 = checki22m01.VolumeLevel4;

                if (cachei22m01 != null)
                    for (int idx = 0; idx < cachei22m01.Length; idx++)
                        if (cachei22m01[idx].AlertFilenameAsk == alertFilenameAsk && cachei22m01[idx].AlertFilenameBid == alertFilenameBid && cachei22m01[idx].ColorAboveAsk == colorAboveAsk && cachei22m01[idx].ColorUnderBid == colorUnderBid && cachei22m01[idx].Length == length && cachei22m01[idx].TradesAmount == tradesAmount && cachei22m01[idx].VolumeLevel1 == volumeLevel1 && cachei22m01[idx].VolumeLevel2 == volumeLevel2 && cachei22m01[idx].VolumeLevel3 == volumeLevel3 && cachei22m01[idx].VolumeLevel4 == volumeLevel4 && cachei22m01[idx].EqualsInput(input))
                            return cachei22m01[idx];

                i22m01 indicator = new i22m01();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.AlertFilenameAsk = alertFilenameAsk;
                indicator.AlertFilenameBid = alertFilenameBid;
                indicator.ColorAboveAsk = colorAboveAsk;
                indicator.ColorUnderBid = colorUnderBid;
                indicator.Length = length;
                indicator.TradesAmount = tradesAmount;
                indicator.VolumeLevel1 = volumeLevel1;
                indicator.VolumeLevel2 = volumeLevel2;
                indicator.VolumeLevel3 = volumeLevel3;
                indicator.VolumeLevel4 = volumeLevel4;
                Indicators.Add(indicator);
                indicator.SetUp();

                i22m01[] tmp = new i22m01[cachei22m01 == null ? 1 : cachei22m01.Length + 1];
                if (cachei22m01 != null)
                    cachei22m01.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cachei22m01 = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.i22m01 i22m01(string alertFilenameAsk, string alertFilenameBid, Color colorAboveAsk, Color colorUnderBid, int length, int tradesAmount, int volumeLevel1, int volumeLevel2, int volumeLevel3, int volumeLevel4)
        {
            return _indicator.i22m01(Input, alertFilenameAsk, alertFilenameBid, colorAboveAsk, colorUnderBid, length, tradesAmount, volumeLevel1, volumeLevel2, volumeLevel3, volumeLevel4);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public Indicator.i22m01 i22m01(Data.IDataSeries input, string alertFilenameAsk, string alertFilenameBid, Color colorAboveAsk, Color colorUnderBid, int length, int tradesAmount, int volumeLevel1, int volumeLevel2, int volumeLevel3, int volumeLevel4)
        {
            return _indicator.i22m01(input, alertFilenameAsk, alertFilenameBid, colorAboveAsk, colorUnderBid, length, tradesAmount, volumeLevel1, volumeLevel2, volumeLevel3, volumeLevel4);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.i22m01 i22m01(string alertFilenameAsk, string alertFilenameBid, Color colorAboveAsk, Color colorUnderBid, int length, int tradesAmount, int volumeLevel1, int volumeLevel2, int volumeLevel3, int volumeLevel4)
        {
            return _indicator.i22m01(Input, alertFilenameAsk, alertFilenameBid, colorAboveAsk, colorUnderBid, length, tradesAmount, volumeLevel1, volumeLevel2, volumeLevel3, volumeLevel4);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public Indicator.i22m01 i22m01(Data.IDataSeries input, string alertFilenameAsk, string alertFilenameBid, Color colorAboveAsk, Color colorUnderBid, int length, int tradesAmount, int volumeLevel1, int volumeLevel2, int volumeLevel3, int volumeLevel4)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.i22m01(input, alertFilenameAsk, alertFilenameBid, colorAboveAsk, colorUnderBid, length, tradesAmount, volumeLevel1, volumeLevel2, volumeLevel3, volumeLevel4);
        }
    }
}
#endregion
