#region Using declarations
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Gui.Design;
using System.Windows.Forms;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    
    [Description("Tools for Vertical Scrolling,Draging and AutoScrolling the NT Charts")]
    public class VerticalScrollTools : Indicator
    {
        #region Variables
            private            System.Windows.Forms.VScrollBar vsbar = null;
		    private            System.Windows.Forms.Button asbutton = null;
		    private            System.Windows.Forms.Button vdbutton = null;
		    private int        speed = 2;
		    private int        extra_margin = 2;
		    private int        ct;
		    private int        cb;
		    private double     pr_min;
		    private double     pr_max;
		    private double     old_prmin;
			private double     old_prmax;
		    private double     scale;
		    private double     margin_size;  //used by maximum/minimum price in dataseries and also by auto scroll feature.
		    private double     un_margin;    // this is due to maximum value of scroll that can't be achieved by user interaction.
		                                     // by MSDN , it is equivalent to (vsbar.largechange - 1).
		    private double     space_max;
		    private double     space_min;
		    private bool       auto_scroll = true;
		    private bool       vertical_drag = true;
		    private double     as_margin = 0;   // margin to cover in autoscroll/verticaldrag mode
		    private bool       show_as = true;
		    private bool       show_vs = false;
		    private bool       show_vd = false;
		    private AutoScrollButtonPosition asbp = AutoScrollButtonPosition.TOP_RIGHT;
		    private AutoScrollType ast = AutoScrollType.Extreem;
		    private int        center_margin_ticks = 2;    // margin in ticks within which center AutoScroll type will keep current close.
		    private double     center_margin;
		    private int        old_mouse_y;
		    private bool       scale_init;
        #endregion

       
        protected override void Initialize()
        {
            Overlay				= true;
			CalculateOnBarClose = false;
			AutoScale           = true;
			scale_init          = false;
        }
		
        #region StartUp/Buttons/Bar
        protected override void OnStartUp()
		{
			base.OnStartUp();
			this.ChartControl.YAxisRangeTypeRight = YAxisRangeType.Fixed;
			
			// Vertical Scroll Bar
			vsbar = new VScrollBar();
			vsbar.Dock = System.Windows.Forms.DockStyle.Right;
			vsbar.Width = 25;
			vsbar.Minimum = 0;
			vsbar.Maximum = 100;
			vsbar.SmallChange = speed;
			vsbar.Value = 0;
			vsbar.Name = "vsBar";
			if(show_vs)
			{
				vsbar.Show();
			}
			else vsbar.Hide();
			this.ChartControl.Controls.Add(vsbar);
			
			vsbar.Scroll += new System.Windows.Forms.ScrollEventHandler(vsbar_Scroll);
			this.ChartControl.ChartPanel.MouseUp += new MouseEventHandler(MouseDrag_OnScale);
			this.ChartControl.ChartPanel.MouseDown += new MouseEventHandler(MouseDown_OnChart);
			this.ChartControl.ChartPanel.MouseMove += new MouseEventHandler(MouseDrag_OnChart);
			
			//Auto Scroll Button 
			    asbutton = new System.Windows.Forms.Button();
				switch (asbp)
				{
					case AutoScrollButtonPosition.TOP_LEFT:
					{
						asbutton.Location = new System.Drawing.Point(this.ChartControl.CanvasLeft + 8,this.ChartControl.CanvasTop + 15);
						break;
					}
					
					case AutoScrollButtonPosition.TOP_RIGHT:
					{
						asbutton.Location = new System.Drawing.Point(this.ChartControl.CanvasRight - 75,this.ChartControl.CanvasTop + 3);
						break;
					}
				}
			    asbutton.Size = new System.Drawing.Size(27,27);
			    asbutton.BackColor = Color.GreenYellow;
			    asbutton.Name = "asButton";
			    asbutton.Text = "AS";
				if(show_as)
				{
					asbutton.Show();
				}
				else asbutton.Hide();
			    this.ChartControl.ChartPanel.Controls.Add(asbutton);
			    asbutton.Click += ToggleAutoScroll;
			
			//Vertical Drag Button
				vdbutton = new System.Windows.Forms.Button();
				switch (asbp)
				{
					case AutoScrollButtonPosition.TOP_LEFT:
					{
						vdbutton.Location = new System.Drawing.Point(this.ChartControl.CanvasLeft + 38,this.ChartControl.CanvasTop + 15);
						break;
					}
					
					case AutoScrollButtonPosition.TOP_RIGHT:
					{
						vdbutton.Location = new System.Drawing.Point(this.ChartControl.CanvasRight - 105, this.ChartControl.CanvasTop +3);
						break;
					}
				}
				vdbutton.Size = new System.Drawing.Size(27,27);
				vdbutton.BackColor = Color.GreenYellow;
				vdbutton.Name = "vdButton";
				vdbutton.Text = "VD";
				if(show_vd)
				{
					vdbutton.Show();
				}
				else vdbutton.Hide();
				this.ChartControl.ChartPanel.Controls.Add(vdbutton);
				vdbutton.Click += ToggleVerticalDrag;
			
				
			old_prmin = double.MaxValue;
			old_prmax = double.MinValue;
			margin_size = (double)(extra_margin*TickSize);
			center_margin = (double)(center_margin_ticks*TickSize)/2;
			UpdateScale();
		}
		#endregion
		
        protected override void OnBarUpdate()
        {
			if(CurrentBar < 0)return;
			pr_min = MIN(Low,Count)[0] - margin_size - un_margin;
			pr_max = MAX(High,Count)[0] + margin_size;
			if( pr_min < old_prmin || pr_max > old_prmax )
			{
				SetScrollLimits();
				old_prmin = pr_min;
				old_prmax = pr_max;
			}
			
			if(!scale_init && CurrentBar == Count - 1)
			{
				InitScale();
				UpdateScale();
				scale_init = true;
			}
			
			AutoScroll();
			
		}
		
		
		#region Event Handelers
		private void MouseDrag_OnScale(object sender, MouseEventArgs e)
		{
			if( this.ChartControl.CanvasRight < e.X  && e.X < this.ChartControl.Size.Width)
			{
				
				
				if(e.Button == MouseButtons.Left)
				{
					UpdateScale();
					
					SetScrollLimits();
					this.ChartControl.ChartPanel.Invalidate();
				}
				
			}
		}
		
		private void MouseDown_OnChart(object sender, MouseEventArgs e)
		{
			if(e.Button == MouseButtons.Left)
			{
				if(!MouseOnChart(e.X,e.Y)) return;
				else old_mouse_y = e.Y;
			}
		}
		
		
		private void MouseDrag_OnChart(object sender, MouseEventArgs e)
		{
			if(e.Button == MouseButtons.Left && vertical_drag)
			{
				if(!MouseOnChart(e.X,e.Y)) return;
				else
				{
					if(auto_scroll && vertical_drag)
					{
					auto_scroll = false;
					asbutton.BackColor = Color.LightSalmon;
					}
					as_margin = (double) ( e.Y - old_mouse_y)*scale;
					this.ChartControl.FixedPanelMinRight = this.ChartControl.FixedPanelMinRight + as_margin;
					this.ChartControl.FixedPanelMaxRight = this.ChartControl.FixedPanelMaxRight + as_margin;
					as_margin = 0;
					SetScrollLimits();
					old_mouse_y = e.Y;
				}
			}
			this.ChartControl.ChartPanel.Invalidate();
		}
		
		
		private void vsbar_Scroll(object sender, ScrollEventArgs se)
		{	
			if(auto_scroll)
			{
				auto_scroll = false; 
				asbutton.BackColor = Color.LightSalmon;
			}
			DoScroll(vsbar.Value);
		}
		
		private void ToggleAutoScroll(object sender, EventArgs e)
		{
			if(!auto_scroll)
			{
				auto_scroll = true;
				asbutton.BackColor = Color.GreenYellow;
				AutoScroll();          // added to make shift of chart when clicks.
			}
			
			else if (auto_scroll)
			{
				auto_scroll = false;
				asbutton.BackColor = Color.LightSalmon;
			}
			
			this.ChartControl.ChartPanel.Invalidate();
		}
		
		private void ToggleVerticalDrag(object sender, EventArgs e)
		{
			if(!vertical_drag)
			{
				vertical_drag = true;
				vdbutton.BackColor = Color.GreenYellow;
			}
			
			else if (vertical_drag)
			{
				vertical_drag = false;
				vdbutton.BackColor = Color.LightSalmon;
			}
			
			this.ChartControl.ChartPanel.Invalidate();
		}
		
		
		protected bool MouseOnChart(int x, int y)
		{
			int temp_y;
			temp_y = this.ChartControl.GetYByValue(this,this.ChartControl.FixedPanelMinRight);   // restricted only in price panel
			if( x < this.ChartControl.CanvasLeft || x >= this.ChartControl.CanvasRight || y < this.ChartControl.CanvasTop || y > temp_y) return false;
			else return true;
		}
			
		#endregion
		
		#region InitScale/UpdateScale/Scrolls
		
		protected void InitScale()
		{
			int lmb_idx;       // left most bar index
			int rmb_idx;       // right most bar index
			
			lmb_idx = CurrentBar - this.FirstBarIndexPainted;
			rmb_idx = CurrentBar - this.LastBarIndexPainted;
			
			double screen_high = double.MinValue;
			double screen_low =  double.MaxValue;
			
			for (int i = rmb_idx ; i <= lmb_idx ; i++)
				{
					if( High[i] > screen_high ) screen_high = High[i];
					if( Low[i] < screen_low ) screen_low = Low[i];
				}
			
			
			/*this.ChartControl.YAxisRangeTypeRight = YAxisRangeType.Automatic;
			this.ChartControl.AutoScaleDateRangeTypeRight = AutoScaleDateRangeType.ScreenDateRange;
			this.ChartControl.AutoScaleMarginTypeRight = AutoScaleMarginType.Percent;
			this.ChartControl.FixedPanelMinRight =  screen_low - this.ChartControl.AutoScaleMarginLowerRight;
			this.ChartControl.FixedPanelMaxRight =  screen_high + this.ChartControl.AutoScaleMarginUpperRight;
			this.ChartControl.YAxisRangeTypeRight = YAxisRangeType.Fixed;*/
				
			this.ChartControl.FixedPanelMinRight =  screen_low - 10*TickSize;
			this.ChartControl.FixedPanelMaxRight =  screen_high + 10*TickSize;
			this.ChartControl.ChartPanel.Invalidate();
		}
		
		protected void UpdateScale()
		{
			ct = this.ChartControl.CanvasTop;
			cb = this.ChartControl.CanvasBottom;
			scale = (this.ChartControl.FixedPanelMaxRight - this.ChartControl.FixedPanelMinRight)/ ( cb - ct );
			un_margin = (double) ((vsbar.LargeChange - 1)*scale);
		}
		
		protected void SetScrollLimits()
		{
			space_max = Math.Max(pr_max,this.ChartControl.FixedPanelMaxRight);
			space_min = Math.Min(pr_min,this.ChartControl.FixedPanelMinRight);
			
			vsbar.SuspendLayout();
			vsbar.Minimum = ct + (int)(( this.ChartControl.FixedPanelMaxRight - space_max )/scale);
			vsbar.Maximum = ct + (int)(( this.ChartControl.FixedPanelMinRight - space_min )/scale);
			vsbar.Value = Math.Max(vsbar.Minimum,0);
			vsbar.ResumeLayout();
		}
		
		protected void DoScroll(int vsvalue)
		{
			
			this.ChartControl.FixedPanelMaxRight = space_max - scale*(double)(vsvalue-vsbar.Minimum);
			this.ChartControl.FixedPanelMinRight = space_min + scale*(double)(vsbar.Maximum - vsvalue);
			this.ChartControl.ChartPanel.Invalidate();
		}
		
			
		protected void AutoScroll()
		{
			int rmb_index;
			rmb_index = CurrentBar - this.LastBarIndexPainted;
			if(this.ChartControl.FixedPanelMaxRight - this.ChartControl.FixedPanelMinRight < High[rmb_index] - Low[rmb_index] + 2*margin_size + un_margin)return;
			if(auto_scroll)
			{
				switch(ast)
				{
					case AutoScrollType.Extreem:
					{
						if(this.ChartControl.FixedPanelMaxRight - High[rmb_index] < margin_size)
				          {
					         as_margin = (margin_size - this.ChartControl.FixedPanelMaxRight + High[rmb_index]);
					         this.ChartControl.FixedPanelMaxRight = this.ChartControl.FixedPanelMaxRight + as_margin;
					         this.ChartControl.FixedPanelMinRight = this.ChartControl.FixedPanelMinRight + as_margin;
					         
				          }
				
				        if(Low[rmb_index] - this.ChartControl.FixedPanelMinRight  < margin_size + un_margin)
				          {
					         as_margin = (margin_size + un_margin - Low[rmb_index] + this.ChartControl.FixedPanelMinRight);
					         this.ChartControl.FixedPanelMinRight = this.ChartControl.FixedPanelMinRight - as_margin;
					         this.ChartControl.FixedPanelMaxRight = this.ChartControl.FixedPanelMaxRight - as_margin;
					         
				           }
						break;
					}
					
					case AutoScrollType.Center:
					{
						double center_price;
						center_price = (this.ChartControl.FixedPanelMaxRight + this.ChartControl.FixedPanelMinRight)/2.0;
						if( Close[rmb_index] > center_price + center_margin)
						{
							as_margin = Close[rmb_index] - center_price - center_margin;
							this.ChartControl.FixedPanelMinRight = this.ChartControl.FixedPanelMinRight + as_margin;
							this.ChartControl.FixedPanelMaxRight = this.ChartControl.FixedPanelMaxRight + as_margin;
							
						}
						
						if( Close[rmb_index] < center_price - center_margin)
						{
							as_margin = center_price - center_margin - Close[rmb_index];
							this.ChartControl.FixedPanelMinRight = this.ChartControl.FixedPanelMinRight - as_margin;
							this.ChartControl.FixedPanelMaxRight = this.ChartControl.FixedPanelMaxRight - as_margin;
							
						}
						break;
					}
				}
				
				as_margin = 0;
				SetScrollLimits();
			}
		}
		
		#endregion
		
		#region Plot/Termination
		public override void Plot(Graphics g,Rectangle bounds,double min,double max)
		{
			switch (asbp)
				{
					case AutoScrollButtonPosition.TOP_LEFT:
					{
						if(asbutton != null)
						{
							asbutton.Location = new System.Drawing.Point(this.ChartControl.CanvasLeft + 8,this.ChartControl.CanvasTop + 15);
						}
						
						if(vdbutton != null)
						{
							vdbutton.Location = new System.Drawing.Point(this.ChartControl.CanvasLeft + 38,this.ChartControl.CanvasTop + 15);
						}
						break;
					}
					
					case AutoScrollButtonPosition.TOP_RIGHT:
					{
						if(asbutton != null)
						{
							asbutton.Location = new System.Drawing.Point(this.ChartControl.CanvasRight - 75,this.ChartControl.CanvasTop + 3);
						}
						
						if(vdbutton != null)
						{
							vdbutton.Location = new System.Drawing.Point(this.ChartControl.CanvasRight - 105, this.ChartControl.CanvasTop +3);
						}
						break;
					}
				}
		}
			
		
		protected override void OnTermination()
		{
			vsbar.Scroll -= vsbar_Scroll;
			this.ChartControl.ChartPanel.MouseUp -= MouseDrag_OnScale;
			this.ChartControl.ChartPanel.MouseDown -= MouseDown_OnChart;
			this.ChartControl.ChartPanel.MouseMove -= MouseDrag_OnChart;
			
			this.ChartControl.Controls.Remove(vsbar);
			vsbar.Dispose();
			if(asbutton != null)
			{
				asbutton.Click -= ToggleAutoScroll;
			    this.ChartControl.ChartPanel.Controls.Remove(asbutton);
			    asbutton.Dispose();
			}
			
			if(vdbutton != null)
			{
				vdbutton.Click -= ToggleVerticalDrag;
				this.ChartControl.ChartPanel.Controls.Remove(vdbutton);
				vdbutton.Dispose();
			}
			
			this.ChartControl.YAxisRangeTypeRight = YAxisRangeType.Automatic;
			base.OnTermination();
		}
		
		#endregion
		
        #region Properties
		[Description("Scroll speed by scroll buttons. Select suitable to Chart and Instrument.")]
        [Category("Parameters")]
		[Gui.Design.DisplayName("Scroll Speed")]
        public int Speed
        {
            get { return speed; }
            set { speed = Math.Max(1, value); }
        }
		
		[Description("Select extra ticks to have on beyond maximum amd minimum dataseries prices.")]
        [Category("Parameters")]
		[Gui.Design.DisplayName("Extra Ticks")]
        public int Extra_margin
        {
            get { return extra_margin; }
            set { extra_margin = Math.Max(1, value); }
        }
		
		[Description("Select Center Margin Ticks for Center AutoScroll.")]
		[Category("Parameters")]
		[Gui.Design.DisplayName("Center AutoScroll Margin")]
		public int Center_margin_ticks
		{
			get { return center_margin_ticks; }
			set { center_margin_ticks = Math.Max(2, value); }
		}
		
		[Description("Show or Hide the Auto Scroll Button. Auto Scroll works only with show.")]
        [Category("Parameters")]
		[Gui.Design.DisplayName("Show AutoScroll Button")]
        public bool Show_as
        {
            get { return show_as; }
            set { show_as = value; }
        }
		
		[Description("Select Location of Auto Scroll Button on Chart")]
        [Category("Parameters")]
		[Gui.Design.DisplayName("Auto Scroll Button Position")]
        public AutoScrollButtonPosition Asbp
        {
            get { return asbp; }
            set { asbp = value; }
        }
		
		[Description("Select Auto Scroll Type. Extreem -> normal auto scroll from boundaries, Center -> keeps close of current bar in center.")]
		[Category("Parameters")]
		[Gui.Design.DisplayName("AutoScroll Type")]
		public AutoScrollType Ast
		{
			get { return ast; }
			set { ast = value; }
		}
		
		
		[Description("Show or Hide Vertical Scroll Bar.")]
        [Category("Parameters")]
		[Gui.Design.DisplayName("Show VerticalScroll Bar")]
        public bool Show_vs
        {
            get { return show_vs; }
            set { show_vs = value; }
        }
		
		
		[Description("Show or Hide Vertical Drag Button.")]
        [Category("Parameters")]
		[Gui.Design.DisplayName("Show VerticalDrag Button")]
        public bool Show_vd
        {
            get { return show_vd; }
            set { show_vd = value; }
        }

        #endregion
        

    }
}

       #region Enums
public enum AutoScrollButtonPosition
			{
				TOP_LEFT,
				TOP_RIGHT,
			}
			
public enum AutoScrollType
{
	Center,
	Extreem,
}
       #endregion

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private VerticalScrollTools[] cacheVerticalScrollTools = null;

        private static VerticalScrollTools checkVerticalScrollTools = new VerticalScrollTools();

        /// <summary>
        /// Tools for Vertical Scrolling,Draging and AutoScrolling the NT Charts
        /// </summary>
        /// <returns></returns>
        public VerticalScrollTools VerticalScrollTools(AutoScrollButtonPosition asbp, AutoScrollType ast, int center_margin_ticks, int extra_margin, bool show_as, bool show_vd, bool show_vs, int speed)
        {
            return VerticalScrollTools(Input, asbp, ast, center_margin_ticks, extra_margin, show_as, show_vd, show_vs, speed);
        }

        /// <summary>
        /// Tools for Vertical Scrolling,Draging and AutoScrolling the NT Charts
        /// </summary>
        /// <returns></returns>
        public VerticalScrollTools VerticalScrollTools(Data.IDataSeries input, AutoScrollButtonPosition asbp, AutoScrollType ast, int center_margin_ticks, int extra_margin, bool show_as, bool show_vd, bool show_vs, int speed)
        {
            if (cacheVerticalScrollTools != null)
                for (int idx = 0; idx < cacheVerticalScrollTools.Length; idx++)
                    if (cacheVerticalScrollTools[idx].Asbp == asbp && cacheVerticalScrollTools[idx].Ast == ast && cacheVerticalScrollTools[idx].Center_margin_ticks == center_margin_ticks && cacheVerticalScrollTools[idx].Extra_margin == extra_margin && cacheVerticalScrollTools[idx].Show_as == show_as && cacheVerticalScrollTools[idx].Show_vd == show_vd && cacheVerticalScrollTools[idx].Show_vs == show_vs && cacheVerticalScrollTools[idx].Speed == speed && cacheVerticalScrollTools[idx].EqualsInput(input))
                        return cacheVerticalScrollTools[idx];

            lock (checkVerticalScrollTools)
            {
                checkVerticalScrollTools.Asbp = asbp;
                asbp = checkVerticalScrollTools.Asbp;
                checkVerticalScrollTools.Ast = ast;
                ast = checkVerticalScrollTools.Ast;
                checkVerticalScrollTools.Center_margin_ticks = center_margin_ticks;
                center_margin_ticks = checkVerticalScrollTools.Center_margin_ticks;
                checkVerticalScrollTools.Extra_margin = extra_margin;
                extra_margin = checkVerticalScrollTools.Extra_margin;
                checkVerticalScrollTools.Show_as = show_as;
                show_as = checkVerticalScrollTools.Show_as;
                checkVerticalScrollTools.Show_vd = show_vd;
                show_vd = checkVerticalScrollTools.Show_vd;
                checkVerticalScrollTools.Show_vs = show_vs;
                show_vs = checkVerticalScrollTools.Show_vs;
                checkVerticalScrollTools.Speed = speed;
                speed = checkVerticalScrollTools.Speed;

                if (cacheVerticalScrollTools != null)
                    for (int idx = 0; idx < cacheVerticalScrollTools.Length; idx++)
                        if (cacheVerticalScrollTools[idx].Asbp == asbp && cacheVerticalScrollTools[idx].Ast == ast && cacheVerticalScrollTools[idx].Center_margin_ticks == center_margin_ticks && cacheVerticalScrollTools[idx].Extra_margin == extra_margin && cacheVerticalScrollTools[idx].Show_as == show_as && cacheVerticalScrollTools[idx].Show_vd == show_vd && cacheVerticalScrollTools[idx].Show_vs == show_vs && cacheVerticalScrollTools[idx].Speed == speed && cacheVerticalScrollTools[idx].EqualsInput(input))
                            return cacheVerticalScrollTools[idx];

                VerticalScrollTools indicator = new VerticalScrollTools();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.Asbp = asbp;
                indicator.Ast = ast;
                indicator.Center_margin_ticks = center_margin_ticks;
                indicator.Extra_margin = extra_margin;
                indicator.Show_as = show_as;
                indicator.Show_vd = show_vd;
                indicator.Show_vs = show_vs;
                indicator.Speed = speed;
                Indicators.Add(indicator);
                indicator.SetUp();

                VerticalScrollTools[] tmp = new VerticalScrollTools[cacheVerticalScrollTools == null ? 1 : cacheVerticalScrollTools.Length + 1];
                if (cacheVerticalScrollTools != null)
                    cacheVerticalScrollTools.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheVerticalScrollTools = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Tools for Vertical Scrolling,Draging and AutoScrolling the NT Charts
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.VerticalScrollTools VerticalScrollTools(AutoScrollButtonPosition asbp, AutoScrollType ast, int center_margin_ticks, int extra_margin, bool show_as, bool show_vd, bool show_vs, int speed)
        {
            return _indicator.VerticalScrollTools(Input, asbp, ast, center_margin_ticks, extra_margin, show_as, show_vd, show_vs, speed);
        }

        /// <summary>
        /// Tools for Vertical Scrolling,Draging and AutoScrolling the NT Charts
        /// </summary>
        /// <returns></returns>
        public Indicator.VerticalScrollTools VerticalScrollTools(Data.IDataSeries input, AutoScrollButtonPosition asbp, AutoScrollType ast, int center_margin_ticks, int extra_margin, bool show_as, bool show_vd, bool show_vs, int speed)
        {
            return _indicator.VerticalScrollTools(input, asbp, ast, center_margin_ticks, extra_margin, show_as, show_vd, show_vs, speed);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Tools for Vertical Scrolling,Draging and AutoScrolling the NT Charts
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.VerticalScrollTools VerticalScrollTools(AutoScrollButtonPosition asbp, AutoScrollType ast, int center_margin_ticks, int extra_margin, bool show_as, bool show_vd, bool show_vs, int speed)
        {
            return _indicator.VerticalScrollTools(Input, asbp, ast, center_margin_ticks, extra_margin, show_as, show_vd, show_vs, speed);
        }

        /// <summary>
        /// Tools for Vertical Scrolling,Draging and AutoScrolling the NT Charts
        /// </summary>
        /// <returns></returns>
        public Indicator.VerticalScrollTools VerticalScrollTools(Data.IDataSeries input, AutoScrollButtonPosition asbp, AutoScrollType ast, int center_margin_ticks, int extra_margin, bool show_as, bool show_vd, bool show_vs, int speed)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.VerticalScrollTools(input, asbp, ast, center_margin_ticks, extra_margin, show_as, show_vd, show_vs, speed);
        }
    }
}
#endregion
