#region Using declarations
using System;
using System.ComponentModel;
using System.Drawing;
using NinjaTrader.Data;
using System.IO;
using System.Globalization;
using System.Text;


#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
	#region File Classes
		//class to read file backwards. found it on
		//http://social.msdn.microsoft.com/Forums/en-US/csharpgeneral/thread/9acdde1a-03cd-4018-9f87-6e201d8f5d09
		//we need it to find last written date without parsing all the file
		public class GomBackwardReader
		{
			private string path;
			private FileStream fs = null;
			
			public GomBackwardReader(string path)
			{
				this.path = path;
				fs = new FileStream(path, FileMode.Open, FileAccess.Read, FileShare.ReadWrite);
				fs.Seek(0, SeekOrigin.End);
			}

			public string Readline()
			{
				byte[] line;	
				byte[] text = new byte[1];	
				long position = 0;	
				int count;
	
				fs.Seek(0, SeekOrigin.Current);
				position = fs.Position;
	
				//do we have trailing \r\n?
				if (fs.Length > 1)	
				{
					byte[] vagnretur = new byte[2];	
					fs.Seek(-2, SeekOrigin.Current);	
					fs.Read(vagnretur, 0, 2);	
					if (ASCIIEncoding.ASCII.GetString(vagnretur).Equals("\r\n"))	
					{		
						//move it back
						fs.Seek(-2, SeekOrigin.Current);		
						position = fs.Position;		
					}	
				}
	
				while (fs.Position > 0)	
				{	
					text.Initialize();
	
					//read one char
					fs.Read(text, 0, 1);
	
					string asciiText = ASCIIEncoding.ASCII.GetString(text);
	
					//moveback to the charachter before
					fs.Seek(-2, SeekOrigin.Current);
	
					if (asciiText.Equals("\n"))	
					{	
						fs.Read(text,0,1);	
						asciiText = ASCIIEncoding.ASCII.GetString(text);
	
						if (asciiText.Equals("\r"))	
						{	
							fs.Seek(1, SeekOrigin.Current);	
							break;	
						}	
					}
				}
	
				count = int.Parse((position - fs.Position ).ToString());	
				line = new byte[count];
				fs.Read(line, 0, count);	
				fs.Seek(-count, SeekOrigin.Current);
	
				return ASCIIEncoding.ASCII.GetString(line);
	
			}

			public bool SOF	{ get { return fs.Position == 0; } }

			public void Dispose()
			{
				if (fs !=null)
				{	fs.Close();
					fs.Dispose();
				}
			}

		}
				
		abstract class GomFileManager
		{	
			
			static public string dateformat="yyMMddHHmmss";
			static public DateTime nullDT = new DateTime(0L);
			protected CultureInfo curCulture = (CultureInfo)CultureInfo.InvariantCulture.Clone();
            

			protected string _fileExtensionName;
			public string FileExtensionName {get  {return _fileExtensionName;} }

			protected FileStream  datafileread,datafilewrite ;
			
			protected string _InstrName;
			protected double _tickSize;
            protected bool _writable=false;
            protected string _fileName=null;
            protected bool _isBinary = false;
	
			protected StreamReader sr; 
			protected StreamWriter sw;

            protected BinaryReader br;
            protected BinaryWriter bw;

            protected bool _compressticks = true;

			
			private DateTime _lastTimeInFile=new DateTime(0L);
			public DateTime LastTimeInFile {get  {return _lastTimeInFile;} set {_lastTimeInFile=value;}}

			public  struct GomDataType
			{
				public DateTime time;
				public TickTypeEnum ticktype;
				public double price;
				public int volume;
			}
			
			abstract public  void RecordTick(DateTime date, TickTypeEnum tickType, double price, int volume);
			abstract public void GetNextLineDataFormatted(ref GomDataType gomdata);
			abstract protected DateTime GetMaxTimeInFile();

			protected void SwapCulture()
			{
                if (curCulture.NumberFormat.NumberDecimalSeparator == ".")
                    curCulture.NumberFormat.NumberDecimalSeparator = ",";
                else
                    curCulture.NumberFormat.NumberDecimalSeparator = ".";
			}
			
			
			public static TickTypeEnum GetIntTickType(double bid,double ask,double price)
			{
				TickTypeEnum tickType;
				
				if (ask<bid) // should not happen but does
				{ 
					if (price<ask) 		    tickType=TickTypeEnum.BelowBid;
					else if (price==ask) 	tickType=TickTypeEnum.AtAsk;
					else if (price<bid) 	tickType=TickTypeEnum.BetweenBidAsk;
					else if (price==bid) 	tickType=TickTypeEnum.AtBid;
					else 					tickType=TickTypeEnum.AboveAsk;
				}
				else if (bid<ask) //normal case
				{
					if (price<bid) 		    tickType=TickTypeEnum.BelowBid;
					else if (price==bid)	tickType=TickTypeEnum.AtBid;
					else if (price<ask)	    tickType=TickTypeEnum.BetweenBidAsk;
					else if (price==ask) 	tickType=TickTypeEnum.AtAsk;
					else					tickType=TickTypeEnum.AboveAsk;
				}
				else //bid==ask, should not happen
				{
					if (price<bid)		    tickType=TickTypeEnum.BelowBid;
					else if (price>ask)	    tickType=TickTypeEnum.AboveAsk;
					else				    tickType=tickType=TickTypeEnum.BetweenBidAsk;
				}
			
			return tickType;
			}
			

			
			public GomFileManager(string name,double tickSize,bool isInstrName)
			{
                if (isInstrName)
                    _InstrName = name;
                else
                    _fileName = name;

				_tickSize=tickSize;
                curCulture.NumberFormat.NumberGroupSeparator = "";
			}

			
			public virtual string GetFileName()
			{
                if (_fileName == null)
                {
                    //file name is instrument name
                    string nomfic = _InstrName;
                    char[] invalidFileChars = Path.GetInvalidFileNameChars();

                    // remove invalid chars (*,/,\ etx)
                    foreach (char invalidFChar in invalidFileChars)
                        nomfic = nomfic.Replace(invalidFChar.ToString(), "");

                    string extension;
                    if (_isBinary)
                        extension = ".dat";
                    else
                        extension = ".txt";
					
					string folder=Environment.GetEnvironmentVariable("GOMFOLDER");
					
					if (folder == null)
						folder=Environment.GetFolderPath(Environment.SpecialFolder.MyDocuments);
					
                    return (folder + "\\" + nomfic + "." + _fileExtensionName + extension);
                }
                else
                    return _fileName;
			}
				
			public bool opendatafile()
			{		
				bool writeOK=false;


                string FileName = GetFileName() ;

                if (_writable)
                {
				    if  (!File.Exists(FileName))
				    {
					    try 
                        { 
                            datafilewrite=File.Open(FileName,FileMode.Create,FileAccess.Write,FileShare.Read);
                            writeOK = true;    
                        }
					    catch (Exception) { writeOK=false;}
				    }				
				    else
				    {
					    try 
                        { 
                            datafilewrite=File.Open(FileName,FileMode.Append,FileAccess.Write,FileShare.Read);
                            writeOK = true;
                        }
					    catch (Exception) { writeOK=false; }
				    }
    				
				    if (writeOK) 
				    { 						
					    LastTimeInFile=GetMaxTimeInFile();
                        if (_isBinary)
                            bw = new BinaryWriter(datafilewrite);
                        else
                            sw = new StreamWriter(datafilewrite);
				    }
                }

				try
				{
					datafileread=File.Open(FileName,FileMode.Open,FileAccess.Read,FileShare.ReadWrite);
				}
				catch (Exception)
				{
					FileStream fstemp=File.Create(FileName);
					fstemp.Close();
					fstemp.Dispose();
					datafileread=File.Open(FileName,FileMode.Open,FileAccess.Read,FileShare.ReadWrite);

				}
                if (_isBinary)
                    br = new BinaryReader(datafileread);
                else
                    sr= new StreamReader(datafileread);	
				
				return writeOK;
			}
			
			private void freehandles()
			{			
				if  (sw != null)
				{
					sw.Close();
					sw.Dispose();
				}
				if  (sr != null)
				{	
					sr.Close();
					sr.Dispose();
				}

                if (bw != null)
                {
                    bw.Close();
                }
                if (br != null)
                {
                    br.Close();
                }

				
				if  (datafileread != null)
				{				
					datafileread.Close();
					datafileread.Dispose();
				}
				
				if  (datafilewrite != null)
				{				
					datafilewrite.Close();
					datafilewrite.Dispose();
				}
			}
			
			public virtual void Dispose()
			 {
				freehandles();
			 }

        }
        #region existing stuff
        class GomFileManagerFlat:GomFileManager //YYMMAAHHMMSS \t TickType \t Price \t Volume
		{
			private DateTime lastTime = new DateTime(0L);
			private TickTypeEnum lastTickType=0;
			private double	lastPrice;
			private int accumulatedVolume=0;
			
			//constructor
			public GomFileManagerFlat(string name,double tickSize,bool isInstrName):base(name,tickSize,isInstrName)
            {
                _fileExtensionName = "Flat";
//                _writable = true;
            }


            public GomFileManagerFlat(string name, double tickSize, bool isInstrName,bool compressticks,bool writedata)
                : this(name, tickSize, isInstrName)
            {
                _compressticks = compressticks;
				_writable=writedata;
            }

			
			// 3 abstract methods to implement
			public  override void RecordTick(DateTime time, TickTypeEnum tickType, double price, int volume)
			{
                bool cancompress = ((time == lastTime) && (tickType == lastTickType) && (price == lastPrice) && _compressticks);
				
                if (cancompress)
 					accumulatedVolume += volume;
				else				
				{	
					if ((lastTime.Ticks>0L) && (lastTime>LastTimeInFile))
					{	
						sw.WriteLine(lastTime.ToUniversalTime().ToString(dateformat)+"\t"+(int)lastTickType+"\t"+lastPrice.ToString("G10",CultureInfo.InvariantCulture)+"\t"+accumulatedVolume);
						sw.Flush();	
					}
					lastTime=time;
					lastTickType=tickType;
					lastPrice=price;
					accumulatedVolume=volume;
				}
			}
			
			private string GetNextLinePivotFormatted()
			{
				string retString=null;
				
				if (!sr.EndOfStream)
				{	
					retString=sr.ReadLine();					
				}			
				return retString;
			}

			public override void GetNextLineDataFormatted(ref GomDataType gomData)
			{
				string retString=GetNextLinePivotFormatted();

				if (retString != null)
				{
					string[] split = retString.Split('\t');

					gomData.time = DateTime.ParseExact(split[0], dateformat, CultureInfo.InvariantCulture).ToLocalTime();
					gomData.ticktype = (TickTypeEnum)Enum.Parse(typeof(TickTypeEnum), split[1]);
					try
					{
						gomData.price = Double.Parse(split[2], curCulture);
					}
					catch (Exception)
					{
						SwapCulture();
						gomData.price = Double.Parse(split[2], curCulture);
					}
					gomData.volume = Int32.Parse(split[3]);
				}
				else
					gomData.time = nullDT;

			}
	
			
			protected override DateTime GetMaxTimeInFile()
			{	
				DateTime retTime=new DateTime(0L);
				string stringRead,lastTimeInFileString="";

				GomBackwardReader br = new GomBackwardReader(GetFileName());
				
				if (!br.SOF)
				{	
					stringRead=br.Readline();
	
					if ((stringRead != null)&& (stringRead.Length>12))						
					{
						lastTimeInFileString=stringRead.Substring(0,12); // get the date						
						retTime=DateTime.ParseExact(lastTimeInFileString,dateformat,CultureInfo.CurrentCulture).ToLocalTime();	
					}
				}
				
				br.Dispose();					
				return retTime;
			}

            public override void Dispose()
            {
                if (lastTime != nullDT)
                    this.RecordTick(lastTime.AddMinutes(1), TickTypeEnum.Unknown, 0, 0);
                base.Dispose();
            }
		}
		
		// DYYMMAAHHMM00 \t Price  on every minute
		// SS TickType \t Delta per reference price of the minute in ticks \t \t Volume  on every tick.
		// SS is omitted is same as previous line
		
		 class GomFileManagerShort:GomFileManager
		{
			private DateTime lastTime = nullDT;
			private int lastMinute;
			private int lastSecond;
			private double pivotPrice;
			private TickTypeEnum lastTickType=0;
			private double	lastPrice;
			private int accumulatedVolume=0;
			private bool newminutestarted=true;

			private DateTime lastReadMinute = nullDT;
			private DateTime lastReadSecond;
			private double	lastReadPivot;
			
			
			
			//constructor
            public GomFileManagerShort(string name, double tickSize, bool isInstrName)
                : base(name, tickSize, isInstrName)
            {
                _fileExtensionName = "Short";
 //               _writable = true;
            }


            public GomFileManagerShort(string name, double tickSize, bool isInstrName, bool compressticks,bool writedata)
                : this(name, tickSize, isInstrName)
            {
                _compressticks = compressticks;
				_writable=writedata;
            }

			// 3 abstract methods to implement
			public override void RecordTick(DateTime time, TickTypeEnum tickType, double price, int volume)
			{
				int newMinute=0;

                bool cancompress = ((time == lastTime) && (tickType == lastTickType) && (price == lastPrice)&&_compressticks );

                if (cancompress)
                    accumulatedVolume += volume;

                else					
				{	
					if ((lastTime.Ticks>0L) && (lastTime>LastTimeInFile))
					{	
						
						if ((lastTime.Second != lastSecond) || newminutestarted)
						{	
							if (newminutestarted)
								{	
									sw.WriteLine(lastMinute.ToString("D10")+"\t"+pivotPrice.ToString("G10",CultureInfo.InvariantCulture));
									newminutestarted=false;
								}

                            sw.WriteLine(lastTime.Second.ToString("D2") + "\t" + (int)lastTickType + "\t" + Convert.ToInt32((lastPrice - pivotPrice) / _tickSize) + "\t" + accumulatedVolume);
							lastSecond=lastTime.Second;
							
						}
						else
						{	
							sw.WriteLine((int)lastTickType+"\t"+Convert.ToInt32((lastPrice-pivotPrice)/_tickSize)+"\t"+accumulatedVolume);
						}
							
						newMinute=Int32.Parse(time.ToUniversalTime().ToString("yyMMddHHmm"));
						if (newMinute != lastMinute)
						{	
							pivotPrice=price-5*_tickSize;  // 5 tick offset to avoid - signs				
							lastMinute=newMinute;
							newminutestarted=true;
						}
						sw.Flush();	
					}
					
					lastTime=time;
					
					if (lastMinute==0)
					{   
						lastMinute=Int32.Parse(lastTime.ToUniversalTime().ToString("yyMMddHHmm"));
						pivotPrice=price-5*_tickSize;;
						newminutestarted=true;
						lastSecond=lastTime.Second;
					}

					lastTickType=tickType;
					lastPrice=price;
					accumulatedVolume=volume;
				}
			}



            public override void GetNextLineDataFormatted(ref GomDataType gomData)
            {

                string readString = null;

                string[] split;

                gomData.time = nullDT;

                if (!sr.EndOfStream)
                {
                    readString = sr.ReadLine();
                    if (readString != null)
                    {
                        split = readString.Split('\t');

                        if (split.Length == 2)
                        {
                            lastReadMinute = DateTime.ParseExact(split[0] + "00", dateformat, CultureInfo.InvariantCulture).ToLocalTime();
                            try
                            {
                                lastReadPivot = Double.Parse(split[1], curCulture);
                            }
                            catch (Exception)
                            {
                                SwapCulture();
                                lastReadPivot = Double.Parse(split[1], curCulture);
                            }

                            GetNextLineDataFormatted(ref  gomData);
                        }

                        else if (split.Length == 4)
                        {
                            gomData.price = Int32.Parse(split[2]) * _tickSize + lastReadPivot;
                            gomData.ticktype = (TickTypeEnum)Enum.Parse(typeof(TickTypeEnum), split[1]);
                            gomData.volume = Int32.Parse(split[3]);

                            lastReadSecond = lastReadMinute.AddSeconds(Int32.Parse(split[0])); ;
                            gomData.time = lastReadSecond;
                        }
                        else if (split.Length == 3)
                        {
                            gomData.price = Int32.Parse(split[1]) * _tickSize + lastReadPivot;
                            gomData.ticktype = (TickTypeEnum)Enum.Parse(typeof(TickTypeEnum), split[0]);
                            gomData.volume = Int32.Parse(split[2]);
                            gomData.time = lastReadSecond;
                        }
                    }

                }
            }
	

			protected override DateTime GetMaxTimeInFile()
			{	
				DateTime retTime=new DateTime(0L);
				string stringRead,lastTimeInFileString="";
				string readSeconds;
				string[] split;

				GomBackwardReader br = new GomBackwardReader(GetFileName());
				
				if (!br.SOF)
				{	
					do						
					{	
						stringRead=br.Readline();
						split=stringRead.Split('\t');
					}
					while ((!br.SOF) && (split.Length!=4));
					
					if (!br.SOF)
						{	
							readSeconds=split[0];
												
							do
							{	
								stringRead=br.Readline();
								split=stringRead.Split('\t');
							}
							while ((!br.SOF) && (split.Length!=2));
							
							if (!br.SOF)
								{
									split=stringRead.Split('\t');
									lastTimeInFileString=split[0]+readSeconds;
									retTime=DateTime.ParseExact(lastTimeInFileString,dateformat,CultureInfo.CurrentCulture).ToLocalTime();	
								}
						}
				}
				br.Dispose();
				
				return retTime;
			}

             public override void Dispose()
             {
                 if ((lastTime != nullDT))
                    this.RecordTick(lastTime.AddMinutes(1),TickTypeEnum.Unknown,0,0);
                base.Dispose();
             }
		}

         class GomFileManagerNinja : GomFileManager //YYMMAAHHMMSS \t TickType \t Price \t Volume
         {
            
             //constructor
             public GomFileManagerNinja(string name, double tickSize, bool isInstrName)
                 : base(name, tickSize, isInstrName)
             {
                 _fileExtensionName = "Ninja";
//                 _writable = false;
             }


             // 3 abstract methods to implement
             public override void RecordTick(DateTime time, TickTypeEnum tickType, double price, int volume)
             {
             }


             public override void GetNextLineDataFormatted(ref GomDataType gomData)
             {

                 string lineread;
                 string[] split;

                 gomData.time=nullDT;

                 if (!sr.EndOfStream)
                 {
                     lineread = sr.ReadLine();
                     if (lineread != null)
                     {
                         split = lineread.Split(';');

                         gomData.time = DateTime.ParseExact(split[0], "yyyyMMdd HHmmss", CultureInfo.InvariantCulture);

                         try
                         {
                             gomData.price = Double.Parse(split[1], curCulture);
                         }
                         catch (Exception)
                         {
                             SwapCulture();
                             gomData.price = Double.Parse(split[1], curCulture);
                         }

                         gomData.volume = Int32.Parse(split[2]);
                         gomData.ticktype = TickTypeEnum.Unknown;
                     }
                 }

             }


             protected override DateTime GetMaxTimeInFile()
             {
                 return nullDT;
             }
         }


         class GomFileManagerIRT : GomFileManager //YYMMAAHHMMSS \t TickType \t Price \t Volume
         {
             private CultureInfo _culture;

             private bool firstline = true;

             //constructor
             public GomFileManagerIRT(string name, double tickSize, bool isInstrName,CultureInfo fileCulture)
                 : base(name, tickSize, isInstrName)
             {
                 _fileExtensionName = "IRT";
//                 _writable = false;
                 _culture = fileCulture;
             }


             // 3 abstract methods to implement
             public override void RecordTick(DateTime time, TickTypeEnum tickType, double price, int volume)
             {
             }


             public override void GetNextLineDataFormatted(ref GomDataType gomData)
             {

                 string lineread;
                 string[] split;

                 gomData.time = nullDT;
                 
                 if (!sr.EndOfStream)
                 {
                     lineread = sr.ReadLine();
                     if ((lineread != null)&& (!firstline))
                     {
                         split = lineread.Split('\t');

                         gomData.time = DateTime.Parse(split[1], _culture);
                         gomData.price = Double.Parse(split[2], _culture);

                         gomData.volume = Int32.Parse(split[3]);
                         gomData.ticktype = GomFileManager.GetIntTickType(Double.Parse(split[4], _culture), Double.Parse(split[5], _culture), gomData.price);
                     }
                     if (firstline)
                     {
                         firstline = false;
                         GetNextLineDataFormatted(ref gomData);
                     }
                 }

             }


             protected override DateTime GetMaxTimeInFile()
             {
                 return nullDT;
             }
         }


         class GomFileManagerCollector : GomFileManager //YYMMAAHHMMSS \t TickType \t Price \t Volume
         {
             private CultureInfo _culture;


             private double bid,ask;

             //constructor
             public GomFileManagerCollector(string name, double tickSize, bool isInstrName, CultureInfo fileCulture)
                 : base(name, tickSize, isInstrName)
             {
                 _fileExtensionName = "Collector";
//                 _writable = false;
                 _culture = fileCulture;
             }


             // 3 abstract methods to implement
             public override void RecordTick(DateTime time, TickTypeEnum tickType, double price, int volume)
             {
             }


             public override void GetNextLineDataFormatted(ref GomDataType gomData)
             {

                 string lineread;
                 string[] split={""};

                 gomData.time = nullDT;

                 if (!sr.EndOfStream)
                     do
                     {
                         lineread = sr.ReadLine();
                         if ((lineread != null))
                         {
                             split = lineread.Split(';');
                             if (split.Length > 1)
                             {

                                 if (split[1] != "")
                                 {
                                     gomData.time = DateTime.Parse(split[0], _culture);
                                     gomData.price = Double.Parse(split[1], _culture);
                                     gomData.volume = Int32.Parse(split[2]);
                                     gomData.ticktype = GomFileManager.GetIntTickType(bid, ask, gomData.price);
                                 }
                                 else
                                 {
                                     if ((split[3] != "") && (split[3] != "0"))
                                         bid = Double.Parse(split[3], _culture);
                                     if ((split[4] != "") && (split[4] != "0"))
                                         ask = Double.Parse(split[4], _culture);

                                     //GetNextLineDataFormatted(ref  gomData);
                                 }
                             }
                         }

                     } while ((split[1] == "") && (lineread!=null));

             }


             protected override DateTime GetMaxTimeInFile()
             {
                 return nullDT;
             }
         }

#endregion



         class GomFileManagerBinary : GomFileManager
         {
             private DateTime lastTime = nullDT;
             private int lastMinute;
             private int lastSecond;
             private double pivotPrice;
             private TickTypeEnum lastTickType = 0;
             private double lastPrice;
             private int accumulatedVolume = 0;
             private bool newminutestarted = true;

             private DateTime lastReadMinute = nullDT;
             private DateTime lastReadSecond;
             private double lastReadPivot;



             //constructor
             public GomFileManagerBinary(string name, double tickSize, bool isInstrName)
                 : base(name, tickSize, isInstrName)
             {
                 _fileExtensionName = "Binary";
 //                _writable = true;
                 _isBinary = true;
             }

             public GomFileManagerBinary(string name, double tickSize, bool isInstrName, bool compressticks,bool writedata)
                : this(name, tickSize, isInstrName)
            {
                _compressticks = compressticks;
				_writable=writedata;
            }



             public void writedata(bool withsecond)
             {
               
                 Byte statbyte;
                 Byte sec;
                 int diff;

                 if (withsecond)
                     statbyte=3<<6;
                 else
                     statbyte=2<<6;

                 statbyte += checked((Byte)((int)lastTickType << 3));

                 diff = Convert.ToInt32(((lastPrice - pivotPrice) / _tickSize));

                 if (diff >= -8 && diff <= +7 && accumulatedVolume <= 15)
                     statbyte += 7;
                 else
                 {
                     if ((diff > SByte.MaxValue) || (diff < SByte.MinValue))
                         statbyte += 1 << 2;

                     if (accumulatedVolume > UInt16.MaxValue)
                         statbyte += 2;
                     else if (accumulatedVolume > Byte.MaxValue)
                         statbyte += 1;
                 }

                 bw.Write(statbyte);

                 if (withsecond)
                 {
                     sec = checked((Byte)lastTime.Second);
                     bw.Write(sec);
                 }

                // bw.Write((byte)0);
                 if (diff >= -8 && diff <= +7 && accumulatedVolume <= 15)
                 {
                     SByte res = checked((SByte)((SByte)(diff << 4) + accumulatedVolume));
                     bw.Write(res);
                 }
                 else
                 {
                     if ((diff > SByte.MaxValue) || (diff < SByte.MinValue))
                     {
                         Int16 res = checked((Int16)diff);
                         bw.Write(res);
                     }
                     else
                     {
                         SByte res = checked((SByte)diff);
                         bw.Write(res);
                     }

                     if (accumulatedVolume > UInt16.MaxValue)
                     {
                         Int32 res = checked((Int32)accumulatedVolume);
                         bw.Write(res);
                     }
                     else if (accumulatedVolume > Byte.MaxValue)
                     {
                         UInt16 res = checked((UInt16)accumulatedVolume);
                         bw.Write(res);
                     }
                     else
                     {
                         Byte res = checked((Byte)accumulatedVolume);
                         bw.Write(res);
                     }
                 }
                 
             }



             // 3 abstract methods to implement
             public override void RecordTick(DateTime time, TickTypeEnum tickType, double price, int volume)
             {
                   
                 int newMinute = 0;
                 bool cancompress = ((time == lastTime) && (tickType == lastTickType) && (price == lastPrice)&& _compressticks );

                 if (cancompress)
                     accumulatedVolume += volume;

              	 else
                 {
                     if ((lastTime.Ticks > 0L) && (lastTime > LastTimeInFile))
                     {

                         if ((lastTime.Second != lastSecond) || newminutestarted)
                         {
                             if (newminutestarted)
                             {
                       
                                 Byte n1 = checked((Byte)1 << 6);
                                 n1 += checked((Byte)(lastMinute % 61));
                                 bw.Write(n1);

                                 UInt32 n2 = checked((UInt32)lastMinute);
                                 bw.Write(n2);
                              
                                 UInt32 n3 = checked( Convert.ToUInt32(pivotPrice / _tickSize));
                                 bw.Write(n3);
 
                                 newminutestarted = false;
                             }

                             writedata(true);

                             lastSecond = lastTime.Second;

                         }
                         else
                         {
                             writedata(false);
                         }

                         newMinute = Int32.Parse(time.ToUniversalTime().ToString("yyMMddHHmm"));
                         if (newMinute != lastMinute)
                         {
                             pivotPrice = price;  // 5 tick offset to avoid - signs				
                             lastMinute = newMinute;
                             newminutestarted = true;
                         }

                         bw.Flush();
                         
                     }

                     lastTime = time;

                     if (lastMinute == 0)
                     {
                         lastMinute = Int32.Parse(lastTime.ToUniversalTime().ToString("yyMMddHHmm"));
                         pivotPrice = price  ;
                         newminutestarted = true;
                         lastSecond = lastTime.Second;
                     }

                     lastTickType = tickType;
                     lastPrice = price;
                     accumulatedVolume = volume;
                 }
             }



             public override void GetNextLineDataFormatted(ref GomDataType gomData)
             {

 
                 gomData.time = nullDT;
                 byte statbyte;


                     try
                     {
                         statbyte = br.ReadByte();
                     }
                     catch (Exception)
                     {
                         return;
                     }


                         if (statbyte >> 6 == 1)
                         {
                             lastReadMinute = DateTime.ParseExact(br.ReadUInt32().ToString("D10") + "00", dateformat, CultureInfo.InvariantCulture).ToLocalTime();
                             lastReadPivot = br.ReadUInt32() * _tickSize;
                             GetNextLineDataFormatted(ref  gomData);
                         }

                         else 
                         {    
                            if (statbyte >> 6 == 3)
                                lastReadSecond = lastReadMinute.AddSeconds(br.ReadByte());

                             gomData.ticktype = (TickTypeEnum)((statbyte & 56 /*00111000*/) >> 3);
                             gomData.time = lastReadSecond;

                             if ((statbyte & 7 /*00000111*/ ) == 7)
                             {
                                 SByte toto = br.ReadSByte();
                                 gomData.volume = toto & 15 /*00001111*/;
                                 gomData.price = lastReadPivot + ((SByte)(toto & 240 /*11110000*/ ) >> 4) * _tickSize;
                             }
                             else
                             {
                                 if ((statbyte & 4 /*00000100*/) > 0)
                                     gomData.price = lastReadPivot + br.ReadInt16() * _tickSize;
                                 else
                                     gomData.price = lastReadPivot + br.ReadSByte() * _tickSize;

                                 if ((statbyte & 3 /*00000011*/) == 0)
                                     gomData.volume = br.ReadByte();
                                 else if ((statbyte & 3 /*00000011*/) == 1)
                                     gomData.volume = br.ReadUInt16();
                                 else if ((statbyte & 3 /*00000011*/) == 2)
                                     gomData.volume = br.ReadInt32();

                             }
                         }                 
             }


             protected override DateTime GetMaxTimeInFile()
             {
                 DateTime retTime=nullDT;
				
                 string readTimeInFile;

                 FileStream fs = new FileStream(GetFileName(), FileMode.Open, FileAccess.Read, FileShare.ReadWrite);
                 fs.Seek(0, SeekOrigin.End);

                 if (fs.Position==0)
                     return retTime;

              
                 byte[] data = new byte[5];

                 bool found = false;
                 do
                 {
                    
                     fs.Seek(-6, SeekOrigin.Current);
                     fs.Read(data, 0, 5);
                    // readByte = checked((byte)fs.ReadByte());

                     if (((data[0] & 192)==64) && ((data[0] & 63)==BitConverter.ToUInt32(data, 1)%61))
						//&& (BitConverter.ToUInt32(data, 1)<1500000000) )
						try
					{	
						DateTime testDate=DateTime.ParseExact(BitConverter.ToUInt32(data, 1).ToString("D10")+"00", dateformat, CultureInfo.CurrentCulture);
						found=true;
					}
					catch(Exception)
					{
					}					
                     
                    // fs.Seek(-1, SeekOrigin.Current);
                 }
                 while ((fs.Position >= 6) && (!found));

                 if (!found)
                     return retTime;

                 readTimeInFile = BitConverter.ToUInt32(data, 1).ToString("D10");

				int enumsize= Enum.GetValues( typeof( TickTypeEnum ) ).Length;
				 
                 found = false;
                 data = new byte[2];
                 fs.Seek(0, SeekOrigin.End);
                 do                 
                 {
                    
                     fs.Seek(-3, SeekOrigin.Current);
                     fs.Read(data, 0, 2);
                    // readByte = checked((byte)fs.ReadByte());

                     if (((data[0]>> 6) == 3) && (((data[0] & 56) >> 3)<= enumsize) && (data[1]<60))
                      found=true;
                     
                    // fs.Seek(-1, SeekOrigin.Current);
                 } while ((fs.Position >= 3) && (!found));

                 if (!found)
                     return retTime;

                 readTimeInFile += data[1].ToString("D2");
                 retTime = DateTime.ParseExact(readTimeInFile, dateformat, CultureInfo.CurrentCulture).ToLocalTime();

                 fs.Close();
                 fs.Dispose();

                 return retTime;
             }

             public override void Dispose()
             {
                 if ((lastTime != nullDT))
                     this.RecordTick(lastTime.AddMinutes(1), TickTypeEnum.Unknown, 0, 0);
                 base.Dispose();
             }
         }
	#endregion
	/// <summary>
	/// Enter the description of your new custom indicator here
	/// </summary>
	[Description("Recorder. Can be instantiated directly or via a derived class")]
	  public class GomRecorderIndicator : Indicator
	{
		#region Variables
		// Wizard generated variables
		// User defined variables (add any user defined variables below)
				
		protected string recordingMessage;
		protected bool recordingOK;
		
		private double ask=Double.MinValue,bid=Double.MaxValue;
		private bool initBidAsk=false;
				
		private bool foundbarinfile=false;
		private bool EOFfound=false;
		private DateTime lastTimeReadInFile;
		
		private int MaxVolume=Int32.MaxValue;
		
		private int MaxTick=Int32.MaxValue;
		
		private DateTime BeginTime=DateTime.MinValue,EndTime=DateTime.MaxValue;
		private bool disableTime=false;
		private bool notimecontrol=false;
		
		//private DateTime lastTimeReadInfile=new DateTime(0L);
		//private string lastTimeReadInFileString="";
		
		TickTypeEnum readTickType=0;
		double readPrice=0.0;
		int readVolume=0;
			
		GomFileManager fm;
		

		protected GomFileFormat recFormat=GomFileFormat.Binary;

        private bool compressTicks = false;
        GomFileManager.GomDataType gomData = new GomFileManager.GomDataType();
		
		private bool writeData;
		private bool initdone;
				
		#endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>		
		
		protected virtual void GomInitialize()
		{
		}
		
		protected override sealed void Initialize()
		{	
			
			GomInitialize();
		}
		
		
		/// <summary>
		/// Called on each bar update event (incoming tick)
		/// </summary>
		protected override sealed void OnBarUpdate()
		{
			if (!initdone)
				{				
					if (fm != null)
						fm.Dispose();
					
					double ts;
					
					ts=Instrument.MasterInstrument.TickSize;
					
					#if NT7			
						if (BarsPeriod.Id==PeriodType.Volume)
							MaxVolume=BarsPeriod.Value;
						
						if (BarsPeriod.Id==PeriodType.Tick)
							MaxTick=BarsPeriod.Value;
						
					#else
						if (Bars.Period.Id==PeriodType.Volume)
							MaxVolume=Bars.Period.Value;	
						
						if (Bars.Period.Id==PeriodType.Tick)
							MaxTick=Bars.Period.Value;	

					#endif
					
					if (recFormat==GomFileFormat.Flat)
						fm=new GomFileManagerFlat(Instrument.FullName,ts,true,compressTicks,writeData);
					else if (recFormat==GomFileFormat.Short)
						fm = new GomFileManagerShort(Instrument.FullName, ts, true, compressTicks,writeData);
					else if (recFormat == GomFileFormat.Binary)
						fm = new GomFileManagerBinary(Instrument.FullName, ts, true,compressTicks,writeData);
	
					
					recordingOK=fm.opendatafile();
					if (writeData)
						recordingMessage="Recording "+fm.FileExtensionName+" "+((recordingOK)?"OK":"KO");
					else
						recordingMessage="Recording OFF";
				
					initdone=true;
				}
			
			
			
			GomOnBarUpdate();
			
			
			int curbarVol=0;
			int curtickcount=0;
			
			gomData.time = GomFileManager.nullDT;
						
			DateTime time0 = Time[0];//.ToUniversalTime();
			
			if (Bars.SessionBreak)
				{
					#if NT7
						Bars.Session.GetNextBeginEnd(time0, out BeginTime, out EndTime);
					#else
						BeginTime=time0.Date.Add(Bars.SessionBegin.TimeOfDay);
						EndTime=time0.Date.Add(Bars.SessionEnd.TimeOfDay);
					#endif
					
					if (EndTime<BeginTime)
						EndTime=EndTime.AddDays(1);	
					
					if (EndTime.TimeOfDay==BeginTime.TimeOfDay)
						notimecontrol=true;
				}
			
			if (!Historical)
			{	
				//if we have pending volume, wait for correct time to send it		
				if (readVolume>0)
				 {		
					if (lastTimeReadInFile<=time0)
					{	
						GomOnMarketDataWithTime(lastTimeReadInFile,readTickType,readPrice,readVolume,false);
						readVolume=0;
					}
				}
			}
			else
			{ 						
				//if we have pending volume, wait for correct time to send it
				if (readVolume>0) 					
				{
					if ((lastTimeReadInFile<time0))
					{
						
						GomOnMarketDataWithTime(lastTimeReadInFile,readTickType,readPrice,readVolume,true);
						curbarVol=readVolume;
						curtickcount=1;
						readVolume=0;
						
					}
					else if ((lastTimeReadInFile==time0))
					{
						GomOnMarketDataWithTime(lastTimeReadInFile,readTickType,readPrice,Math.Min(readVolume,MaxVolume),true);
						curbarVol=Math.Min(readVolume,MaxVolume);
						curtickcount=1;
						readVolume=Math.Max(0,readVolume-MaxVolume);

						if ((curbarVol==MaxVolume)||(curtickcount==MaxTick))
							return;
					}
				}
						
				//while file is not over, let's look for good data in the data file
				if (!EOFfound) 
				{	
					// if we're starting, let's find the first correct bar in the file (corresponding to CurrentBar=0)
					//ie first bar after time[0]
					if (!foundbarinfile) 
					{
						do						
						{
						  fm.GetNextLineDataFormatted(ref gomData);	
						}while ((gomData.time != GomFileManager.nullDT) && (gomData.time <= time0));
						
						//now we're on the first line AFTER Time[0]. this will be open of next bar.

						if (gomData.time != GomFileManager.nullDT)
						{  
							foundbarinfile=true; // we found the the first line
							lastTimeReadInFile=gomData.time;
							readTickType = gomData.ticktype;
							readPrice = gomData.price;
							readVolume = gomData.volume;

						}
						else
							EOFfound=true;
						
					}
					
					//main process : if Time[0]>current time in file, move in the file and build the bar until
					else if (lastTimeReadInFile<=time0)
					{ 
						do
						{
							fm.GetNextLineDataFormatted(ref gomData);
						
							if (gomData.time != GomFileManager.nullDT)
							{
								lastTimeReadInFile = gomData.time;
								readTickType = gomData.ticktype;
								readPrice = gomData.price;
								readVolume = gomData.volume;
									
								if (lastTimeReadInFile<time0)
								{	
									curbarVol+=readVolume;
									curtickcount+=1;
									GomOnMarketDataWithTime(lastTimeReadInFile,readTickType,readPrice,readVolume,false);
									readVolume=0;
								}
									
								else if (lastTimeReadInFile==time0)	
								{	
									if (curtickcount<MaxTick)
									{	
										/* tick splitting method 
										if (curbarVol+readVolume>MaxVolume)
										{
											GomOnMarketDataWithTime(lastTimeReadInFile,readTickType,readPrice,Math.Max(MaxVolume-curbarVol,0),false);
											readVolume-=Math.Max(MaxVolume-curbarVol,0);
											curbarVol=MaxVolume;	
											
											curbarVol=MaxVolume;
										}
										else	
										*/
										{
											curbarVol+=readVolume;
											curtickcount +=1;
											GomOnMarketDataWithTime(lastTimeReadInFile,readTickType,readPrice,readVolume,false);
											readVolume=0;
										}
									}
								}		
									
							}
						} while ((gomData.time != GomFileManager.nullDT) && ((gomData.time < time0)||((gomData.time == time0)&& (curbarVol<MaxVolume)&& (curtickcount<MaxTick))));
						
						
						//we're on the first bar after Time[0], or file is ended.						
						if ((gomData.time == GomFileManager.nullDT))
						{	
							EOFfound=true;
							readVolume=0;
						}												
					}
				}

			}
		}
				
				
		protected override void OnMarketData(MarketDataEventArgs e) 
		{
			TickTypeEnum tickType;
						
			if (!initBidAsk)
				initBidAsk=(ask>bid);
			
			if ((e.MarketDataType==MarketDataType.Last))
			{	
				tickType=GomFileManager.GetIntTickType(bid,ask,e.Price);
				
				if ((recordingOK)&&(initBidAsk)&& (e.Time>fm.LastTimeInFile))
					fm.RecordTick(e.Time,tickType,e.Price,(int)e.Volume);
				GomOnMarketDataWithTime(e.Time,tickType,e.Price,(int)e.Volume,FirstTickOfBar);								
			}
			
			else if (e.MarketDataType == MarketDataType.Ask)  { ask=e.Price; }
				
			else if (e.MarketDataType == MarketDataType.Bid) {	bid=e.Price; }
		}		
		
		protected virtual void GomOnMarketData(TickTypeEnum tickType,double price,int volume)
		{	}
		
		protected virtual void GomOnBarUpdate()
		{	}
		
		protected virtual void GomOnMarketData(TickTypeEnum tickType,double price,int volume,bool firstTickOfBar)
		{
			GomOnMarketData(tickType, price, volume);
		}
		
		protected virtual void GomOnMarketDataWithTime(DateTime tickTime, TickTypeEnum tickType,double price,int volume,bool firstTickOfBar)
		{			
			if (notimecontrol || disableTime||((tickTime>=BeginTime)&&(tickTime<=EndTime)))
				GomOnMarketData( tickType, price, volume, firstTickOfBar);

			
		}

		
		public override void Plot(Graphics graphics, Rectangle bounds, double min, double max)
		
		{	
			base.Plot(	graphics,bounds,min,max);		
			
			Color ColorNeutral=Color.FromArgb(255,255-ChartControl.BackColor.R,255-ChartControl.BackColor.G,255-ChartControl.BackColor.B);

			graphics.DrawString(recordingMessage, ChartControl.Font, new SolidBrush(ColorNeutral), bounds.Left, bounds.Bottom-22,new StringFormat());

	
		}

	#if NT7
		protected override void OnTermination()
		{
			if (fm != null)
			{	
				recordingOK=false;
				fm.Dispose();
			}
		}
	#else	
		public override void Dispose()
		{
			if (fm != null)
			{
				recordingOK=false;
				fm.Dispose();
			}
			
			base.Dispose();
		}
	#endif


			
		#region Properties
		[Description("File Recording Format")]
		[Category("Settings : Recorder")]
		[Gui.Design.DisplayName("File Format")]
		
		 public GomFileFormat RecFormat
		{
			get { return recFormat; }
			set {  recFormat=value;}
		}

        [Description("Compress Ticks on second basis")]
		[Category("Settings : Recorder")]
        [Gui.Design.DisplayName("Compress Ticks")]
        public bool CompressTicks
        {
            get { return compressTicks; }
            set { compressTicks = value; }
        }
		
		[Description("Write Data")]
		[Category("Settings : Recorder")]
        [Gui.Design.DisplayName("Write Data")]
        public bool WriteData
        {
            get { return writeData; }
            set { writeData = value; }
        }
		
		[Description("Disable Tick Time Filtering")]
		[Category("Settings : Recorder")]
        [Gui.Design.DisplayName("Disable Time Filter")]
        public bool DisableTime
        {
            get { return disableTime; }
            set { disableTime = value; }
        }
		#endregion
	}
}

public enum GomFileFormat{ Flat,Short,Binary}

public enum TickTypeEnum
			{ BelowBid,AtBid,BetweenBidAsk,AtAsk,AboveAsk,Unknown}

#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private GomRecorderIndicator[] cacheGomRecorderIndicator = null;

        private static GomRecorderIndicator checkGomRecorderIndicator = new GomRecorderIndicator();

        /// <summary>
        /// Recorder. Can be instantiated directly or via a derived class
        /// </summary>
        /// <returns></returns>
        public GomRecorderIndicator GomRecorderIndicator()
        {
            return GomRecorderIndicator(Input);
        }

        /// <summary>
        /// Recorder. Can be instantiated directly or via a derived class
        /// </summary>
        /// <returns></returns>
        public GomRecorderIndicator GomRecorderIndicator(Data.IDataSeries input)
        {
            if (cacheGomRecorderIndicator != null)
                for (int idx = 0; idx < cacheGomRecorderIndicator.Length; idx++)
                    if (cacheGomRecorderIndicator[idx].EqualsInput(input))
                        return cacheGomRecorderIndicator[idx];

            lock (checkGomRecorderIndicator)
            {
                if (cacheGomRecorderIndicator != null)
                    for (int idx = 0; idx < cacheGomRecorderIndicator.Length; idx++)
                        if (cacheGomRecorderIndicator[idx].EqualsInput(input))
                            return cacheGomRecorderIndicator[idx];

                GomRecorderIndicator indicator = new GomRecorderIndicator();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                Indicators.Add(indicator);
                indicator.SetUp();

                GomRecorderIndicator[] tmp = new GomRecorderIndicator[cacheGomRecorderIndicator == null ? 1 : cacheGomRecorderIndicator.Length + 1];
                if (cacheGomRecorderIndicator != null)
                    cacheGomRecorderIndicator.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheGomRecorderIndicator = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Recorder. Can be instantiated directly or via a derived class
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.GomRecorderIndicator GomRecorderIndicator()
        {
            return _indicator.GomRecorderIndicator(Input);
        }

        /// <summary>
        /// Recorder. Can be instantiated directly or via a derived class
        /// </summary>
        /// <returns></returns>
        public Indicator.GomRecorderIndicator GomRecorderIndicator(Data.IDataSeries input)
        {
            return _indicator.GomRecorderIndicator(input);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Recorder. Can be instantiated directly or via a derived class
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.GomRecorderIndicator GomRecorderIndicator()
        {
            return _indicator.GomRecorderIndicator(Input);
        }

        /// <summary>
        /// Recorder. Can be instantiated directly or via a derived class
        /// </summary>
        /// <returns></returns>
        public Indicator.GomRecorderIndicator GomRecorderIndicator(Data.IDataSeries input)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.GomRecorderIndicator(input);
        }
    }
}
#endregion
