// 
// Copyright (C) 2006, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//

#region Using declarations
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.ComponentModel;
using System.Xml.Serialization;
using NinjaTrader.Data;
using NinjaTrader.Gui.Chart;
#endregion

// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{	
	/// <summary>
	/// Price by Volume.
	/// </summary>
	[Description("Plots a vertical histogram of volume by price.")]
	public class GomVolumeProfile : GomRecorderIndicator
	{
		#region Variables
		internal class VolumeInfoItem
		{
			public double up = 0;
			public double down = 0;
			public double neutral = 0;
		}

		private int							alpha;
		private int							allHours = 0; // 0=unchecked, 1=true, 2=false
        private double                      askPrice = 0;
		private SolidBrush					barBrushDown;
		private SolidBrush					barBrushInactive	= null;
		private SolidBrush					barBrushNeutral;
		private SolidBrush					barBrushUp;
		private int							barSpacing			= 1;
        private double                      bidPrice            = 0;
		private bool						drawLines			= false;
		private Color						lineColor			= Color.Black;
		private Pen							linePen;
		private MarketDataItemEventHandler	marketDataHandler	= null;
		private int							startBarIndex		= 0;
		private DateTime					startTime			= Cbi.Globals.MinDate;
		private int							transparency		= 80;
		private Color						volumeColorDown		= Color.Red;
		private Color						volumeColorNeutral	= Color.Gray;
		private Color						volumeColorUp		= Color.Lime;
		int lastcalcbar=-1;

		private bool reinitSession=true;
		
		private SortedDictionary<double, VolumeInfoItem> 
				volumeInfo = new SortedDictionary<double, VolumeInfoItem>();
		#endregion

		/// <summary>
		/// This method is used to configure the indicator and is called once before any bar data is loaded.
		/// </summary>
		protected override void GomInitialize()
		{
			ChartOnly			= true;
			Overlay				= true;
			PriceTypeSupported	= false;
			CalculateOnBarClose = false;
			ZOrder				= -1;
			startTime			= DateTime.Now;

			RecalculateColors();

			}

		protected override void GomOnBarUpdate()
		{
			if (Bars.SessionBreak && reinitSession)
			{
				volumeInfo.Clear();
			
			}
			
		
		}
		

		protected override void GomOnMarketData(TickTypeEnum tickType,double price,int volume,bool firstTickOfBar)	
		{

			if (!volumeInfo.ContainsKey(price))
				volumeInfo.Add(price, new VolumeInfoItem());

			VolumeInfoItem volumeInfoItem = volumeInfo[price];

			if ((tickType==TickTypeEnum.AboveAsk)||(tickType==TickTypeEnum.AtAsk))
				volumeInfoItem.up += volume;
			else if ((tickType==TickTypeEnum.BelowBid)||(tickType==TickTypeEnum.AtBid))
				volumeInfoItem.down += volume;
			else
				volumeInfoItem.neutral += volume;
			
			lastcalcbar=CurrentBar;
		}

		private int GetYPos(double price, Rectangle bounds, double min, double max)
		{
			#if NT7
				return ChartControl.GetYByValue(this, price);
			#else			
				return (int) ((bounds.Y + bounds.Height) - 
					((price - min) / ChartControl.MaxMinusMin(max, min)) * bounds.Height);
			#endif
		}

		/// <summary>
		/// Called when the indicator is plotted.
		/// </summary>
		public override void Plot(Graphics graphics, Rectangle bounds, double min, double max)
		{
			// Paranoia
			if (Bars == null || Bars.Instrument == null)
				return;

			bool	isInactive	= false;
			double	tickSize	= Bars.Instrument.MasterInstrument.TickSize;

			// Check if we should 'gray out' the bars
			if (ChartControl.LastBarPainted < lastcalcbar)
				isInactive = true;

			double volumeMax = 0;

			// Figure out the max volume
			foreach (KeyValuePair<double, VolumeInfoItem> keyValue in volumeInfo)
			{
				double price = keyValue.Key;

				// Don't watch volume for prices outside the visible chart
				if (price > max  || price < min)
					continue;

				VolumeInfoItem  vii   = keyValue.Value;
				double 			total = vii.up + vii.down + vii.neutral;

				volumeMax = Math.Max(volumeMax, total);
   			}

			if (volumeMax == 0)
				return;

			SolidBrush upBrush		= barBrushUp;
			SolidBrush downBrush	= barBrushDown;
			SolidBrush neutralBrush	= barBrushNeutral;

			if (isInactive)
			{
				if (barBrushInactive == null)
					barBrushInactive = new SolidBrush(Color.FromArgb((int)Math.Min(255, alpha + 50), 
							ChartControl.PriceMarkerInactive.R, 
							ChartControl.PriceMarkerInactive.G, 
							ChartControl.PriceMarkerInactive.B));

				upBrush = downBrush = neutralBrush = barBrushInactive;
			}

			int viiPosition = 0;

			// Plot 'em
			foreach (KeyValuePair<double, VolumeInfoItem> keyValue in volumeInfo)
			{
				viiPosition++;

				VolumeInfoItem vii = keyValue.Value;
				double total = vii.up + vii.down + vii.neutral;

				double tickHeight = 
					(tickSize / ChartControl.MaxMinusMin(max, min)) * bounds.Height;

				double priceLower = keyValue.Key - tickSize / 2;
				int yLower = GetYPos(priceLower, bounds, min, max);
				int yUpper = GetYPos(priceLower + tickSize, bounds, min, max);
				int height = Math.Max(1, Math.Abs(yUpper - yLower) - barSpacing);

 				int barWidthUp		= (int) ((bounds.Width / 2) * (vii.up      / volumeMax));
				int barWidthNeutral = (int) ((bounds.Width / 2) * (vii.neutral / volumeMax));
				int barWidthDown	= (int) ((bounds.Width / 2) * (vii.down    / volumeMax));

				int xpos = bounds.X;

				graphics.FillRectangle(upBrush,      new Rectangle(xpos, yUpper, barWidthUp, height));
				xpos += barWidthUp;				
				graphics.FillRectangle(neutralBrush, new Rectangle(xpos, yUpper, barWidthNeutral, height));
				xpos += barWidthNeutral;
				graphics.FillRectangle(downBrush,    new Rectangle(xpos, yUpper, barWidthDown, height));

				if (drawLines == true) 
				{
					// Lower line
					graphics.DrawLine(linePen, bounds.X, yLower-1, bounds.X + bounds.Width, yLower-1);

					// Upper line (only at the very top)
					if (viiPosition == volumeInfo.Count) 
						graphics.DrawLine(linePen, bounds.X, yUpper-1, bounds.X + bounds.Width, yUpper-1);
				}
			}
		}

		private void RecalculateColors() 
		{
			alpha			 = (int) (255.0 * ((100.0 - transparency) / 100.0));

			barBrushUp       = new SolidBrush(Color.FromArgb(alpha, volumeColorUp.R, volumeColorUp.G, volumeColorUp.B));
			barBrushNeutral  = new SolidBrush(Color.FromArgb(alpha, volumeColorNeutral.R, volumeColorNeutral.G, volumeColorNeutral.B));
			barBrushDown     = new SolidBrush(Color.FromArgb(alpha, volumeColorDown.R, volumeColorDown.G, volumeColorDown.B));
			linePen		     = new Pen(Color.FromArgb(alpha, lineColor.R, lineColor.G, lineColor.B));
		}


		#region Properties
		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Color of volume bars representing sell trades.")]
		[Category("Colors")]
		[Gui.Design.DisplayNameAttribute("Sell color")]
		public Color VolumeColorDown
		{
			get { return volumeColorDown; }
			set { volumeColorDown = value; RecalculateColors(); }
		}

		/// <summary>
		/// </summary>
		[Browsable(false)]
		public string VolumeColorDownSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(volumeColorDown); }
			set { volumeColorDown = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}

		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Color of volume bars representing trades in between the market.")]
		[Category("Colors")]
		[Gui.Design.DisplayNameAttribute("Neutral color")]
		public Color VolumeColorNeutral
		{
			get { return volumeColorNeutral; }
			set { volumeColorNeutral = value; RecalculateColors(); }
		}

		/// <summary>
		/// </summary>
		[Browsable(false)]
		public string VolumeColorNeutralSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(volumeColorNeutral); }
			set { volumeColorNeutral = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}

		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Color of volume bars representing buy trades.")]
		[Category("Colors")]
		[Gui.Design.DisplayNameAttribute("Buy color")]
		public Color VolumeColorUp
		{
			get { return volumeColorUp; }
			set { volumeColorUp = value; RecalculateColors(); }
		}

		/// <summary>
		/// </summary>
		[Browsable(false)]
		public string VolumeColorUpSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(volumeColorUp); }
			set { volumeColorUp = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}

		/// <summary>
		/// </summary>
		[Description("Spacing between the volume bars.")]
		[Category("Parameters")]
		[Gui.Design.DisplayNameAttribute("Bar spacing")]
		public int BarSpacing
		{
			get { return barSpacing; }
			set 
			{ 
				barSpacing = Math.Min(5, Math.Max(0,  value));
			}
		}

		/// <summary>
		/// </summary>
		[Description("Draw horizontal lines between the volume bars.")]
		[Category("Parameters")]
		[Gui.Design.DisplayNameAttribute("Draw lines")]
		public bool DrawLines
		{
			get { return drawLines; }
			set { drawLines = value; }
		}

		/// <summary>
		/// </summary>
		[XmlIgnore()]
		[Description("Color of the line between the volume bars.")]
		[Category("Colors")]
		[Gui.Design.DisplayNameAttribute("Line color")]
		public Color LineColor
		{
			get { return lineColor; }
			set { lineColor = value; RecalculateColors(); }
		}

		/// <summary>
		/// </summary>
		[Browsable(false)]
		public string LineColorSerialize
		{
			get { return NinjaTrader.Gui.Design.SerializableColor.ToString(lineColor); }
			set { lineColor = NinjaTrader.Gui.Design.SerializableColor.FromString(value); }
		}

		/// <summary>
		/// </summary>
		[Description("The percentage of transparency of the volume bars.")]
		[Category("Parameters")]
		public int Transparency
		{
			get { return transparency; }
			set 
			{ 
				transparency = Math.Max(0 , value); 
				transparency = Math.Min(90, value);
			}
		}
		
				/// <summary>
		/// </summary>
		[Description("Reinitialize on session Break")]
		[Category("Parameters")]
		[Gui.Design.DisplayNameAttribute("Reinit Session")]
		public bool ReinitSession
		{
			get { return reinitSession; }
			set {reinitSession=value; }
		}
#endregion
	}
}


#region NinjaScript generated code. Neither change nor remove.
// This namespace holds all indicators and is required. Do not change it.
namespace NinjaTrader.Indicator
{
    public partial class Indicator : IndicatorBase
    {
        private GomVolumeProfile[] cacheGomVolumeProfile = null;

        private static GomVolumeProfile checkGomVolumeProfile = new GomVolumeProfile();

        /// <summary>
        /// Plots a vertical histogram of volume by price.
        /// </summary>
        /// <returns></returns>
        public GomVolumeProfile GomVolumeProfile(int barSpacing, bool drawLines, bool reinitSession, int transparency)
        {
            return GomVolumeProfile(Input, barSpacing, drawLines, reinitSession, transparency);
        }

        /// <summary>
        /// Plots a vertical histogram of volume by price.
        /// </summary>
        /// <returns></returns>
        public GomVolumeProfile GomVolumeProfile(Data.IDataSeries input, int barSpacing, bool drawLines, bool reinitSession, int transparency)
        {
            if (cacheGomVolumeProfile != null)
                for (int idx = 0; idx < cacheGomVolumeProfile.Length; idx++)
                    if (cacheGomVolumeProfile[idx].BarSpacing == barSpacing && cacheGomVolumeProfile[idx].DrawLines == drawLines && cacheGomVolumeProfile[idx].ReinitSession == reinitSession && cacheGomVolumeProfile[idx].Transparency == transparency && cacheGomVolumeProfile[idx].EqualsInput(input))
                        return cacheGomVolumeProfile[idx];

            lock (checkGomVolumeProfile)
            {
                checkGomVolumeProfile.BarSpacing = barSpacing;
                barSpacing = checkGomVolumeProfile.BarSpacing;
                checkGomVolumeProfile.DrawLines = drawLines;
                drawLines = checkGomVolumeProfile.DrawLines;
                checkGomVolumeProfile.ReinitSession = reinitSession;
                reinitSession = checkGomVolumeProfile.ReinitSession;
                checkGomVolumeProfile.Transparency = transparency;
                transparency = checkGomVolumeProfile.Transparency;

                if (cacheGomVolumeProfile != null)
                    for (int idx = 0; idx < cacheGomVolumeProfile.Length; idx++)
                        if (cacheGomVolumeProfile[idx].BarSpacing == barSpacing && cacheGomVolumeProfile[idx].DrawLines == drawLines && cacheGomVolumeProfile[idx].ReinitSession == reinitSession && cacheGomVolumeProfile[idx].Transparency == transparency && cacheGomVolumeProfile[idx].EqualsInput(input))
                            return cacheGomVolumeProfile[idx];

                GomVolumeProfile indicator = new GomVolumeProfile();
                indicator.BarsRequired = BarsRequired;
                indicator.CalculateOnBarClose = CalculateOnBarClose;
#if NT7
                indicator.ForceMaximumBarsLookBack256 = ForceMaximumBarsLookBack256;
                indicator.MaximumBarsLookBack = MaximumBarsLookBack;
#endif
                indicator.Input = input;
                indicator.BarSpacing = barSpacing;
                indicator.DrawLines = drawLines;
                indicator.ReinitSession = reinitSession;
                indicator.Transparency = transparency;
                Indicators.Add(indicator);
                indicator.SetUp();

                GomVolumeProfile[] tmp = new GomVolumeProfile[cacheGomVolumeProfile == null ? 1 : cacheGomVolumeProfile.Length + 1];
                if (cacheGomVolumeProfile != null)
                    cacheGomVolumeProfile.CopyTo(tmp, 0);
                tmp[tmp.Length - 1] = indicator;
                cacheGomVolumeProfile = tmp;
                return indicator;
            }
        }
    }
}

// This namespace holds all market analyzer column definitions and is required. Do not change it.
namespace NinjaTrader.MarketAnalyzer
{
    public partial class Column : ColumnBase
    {
        /// <summary>
        /// Plots a vertical histogram of volume by price.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.GomVolumeProfile GomVolumeProfile(int barSpacing, bool drawLines, bool reinitSession, int transparency)
        {
            return _indicator.GomVolumeProfile(Input, barSpacing, drawLines, reinitSession, transparency);
        }

        /// <summary>
        /// Plots a vertical histogram of volume by price.
        /// </summary>
        /// <returns></returns>
        public Indicator.GomVolumeProfile GomVolumeProfile(Data.IDataSeries input, int barSpacing, bool drawLines, bool reinitSession, int transparency)
        {
            return _indicator.GomVolumeProfile(input, barSpacing, drawLines, reinitSession, transparency);
        }
    }
}

// This namespace holds all strategies and is required. Do not change it.
namespace NinjaTrader.Strategy
{
    public partial class Strategy : StrategyBase
    {
        /// <summary>
        /// Plots a vertical histogram of volume by price.
        /// </summary>
        /// <returns></returns>
        [Gui.Design.WizardCondition("Indicator")]
        public Indicator.GomVolumeProfile GomVolumeProfile(int barSpacing, bool drawLines, bool reinitSession, int transparency)
        {
            return _indicator.GomVolumeProfile(Input, barSpacing, drawLines, reinitSession, transparency);
        }

        /// <summary>
        /// Plots a vertical histogram of volume by price.
        /// </summary>
        /// <returns></returns>
        public Indicator.GomVolumeProfile GomVolumeProfile(Data.IDataSeries input, int barSpacing, bool drawLines, bool reinitSession, int transparency)
        {
            if (InInitialize && input == null)
                throw new ArgumentException("You only can access an indicator with the default input/bar series from within the 'Initialize()' method");

            return _indicator.GomVolumeProfile(input, barSpacing, drawLines, reinitSession, transparency);
        }
    }
}
#endregion
