#region Using declarations

using System;
using System.ComponentModel;
using NinjaTrader.Core;
using NinjaTrader.Core.FloatingPoint;
using NinjaTrader.Gui.Tools;

#endregion

//This namespace holds Add ons in this folder and is required. Do not change it. 

namespace Sim22PriceFormatExtensions
{
    /*
	Notes: Version 1.2.
	Was using a string return but thought an int value would be more efficient.
    Aug 2016 NT8b12 - changed to non-static class
	*/
    public enum DecimalPlacesEnum
    {
        [Description("0")]
        None,
        [Description("1")]
        X,
        [Description("2")]
        XX,
        [Description("3")]
        XXX,
        [Description("4")]
        XXXX,
        [Description("5")]
        XXXXX,
        [Description("6")]
        XXXXXX,
        [Description("7")]
        XXXXXXX,
        [Description("Instrument")]
        SameAsInstrument = -1,
    }

    public static class Sim22PriceFormatExtensions
    {
        

        public static string ToStringDecimalPlaces(this double price, double tickSize, DecimalPlacesEnum dpe)
        {
            switch (dpe)
            {
                case DecimalPlacesEnum.None:
                    return price.ToString("N0");               
                case DecimalPlacesEnum.X:
                    return price.ToString("N1");
                case DecimalPlacesEnum.XX:
                    return price.ToString("N2");
                case DecimalPlacesEnum.XXX:
                    return price.ToString("N3");
                case DecimalPlacesEnum.XXXX:
                    return price.ToString("N4");
                case DecimalPlacesEnum.XXXXX:
                    return price.ToString("N5");
                case DecimalPlacesEnum.XXXXXX:
                    return price.ToString("N6");
                case DecimalPlacesEnum.XXXXXXX:
                    return price.ToString("N7");
                case DecimalPlacesEnum.SameAsInstrument:
                    return price.ToStringSameAsInstrument(tickSize);
                default:
                    return price.ToString("N2");
            }
        }

        public static string ToStringDecimalPlaces(this double price, double tickSize, int decimalPlaces)
        {
            switch (decimalPlaces)
            {
                case 0:
                    return price.ToString("N0");
                case 1:
                    return price.ToString("N1");
                case 2:
                    return price.ToString("N2");
                case 3:
                    return price.ToString("N3");
                case 4:
                    return price.ToString("N4");
                case 5:
                    return price.ToString("N5");
                case 6:
                    return price.ToString("N6");
                case 7:
                    return price.ToString("N7");
                case -1:
                    return price.ToStringSameAsInstrument(tickSize);
                default:
                    return price.ToString("N2");
            }
        }

        public static string ToStringCompacted(this double value, double tickSize, int decPlaces, double fontSize, double maxLength)
        {
            //if (value.ApproxCompare(0) == 0)
            //    return "N/A";

            int count0 = 0;

            double tmpValue = value;
            // used to 'remove' negative sign for counting how many digits are present
            if (value < 0)
            {
                tmpValue = -value;
            }
            // count how many digits are present
            do { count0++; } while ((tmpValue /= 10) >= 1);

            double fontLength = fontSize * 0.8 * count0;

            if (fontLength >= maxLength)
            {
                if (value >= 1000000 || value <= -1000000)
                    return (value / 1000000).ToStringDecimalPlaces(tickSize, decPlaces) + "M";
                if (value >= 1000 || value <= -1000)
                    return (value / 1000).ToStringDecimalPlaces(tickSize, decPlaces) + "K";
                // else
                return value.ToStringDecimalPlaces(tickSize, decPlaces);
            }

            return value.ToStringDecimalPlaces(tickSize, decPlaces);
        }

        public static string ToStringSameAsInstrument(this double value, double tickSize)
        {
            //Thanks to 'pppatil' for this Bond price helper
            //Placed 'trunc' and 'fraction' within the bonds conditions to reduce resource usage for non-bonds.
            var priceMarker = "";

            var isFractionalprice = tickSize.IsFractionalPrice();

            if (isFractionalprice)
            {
                var fraction = 0; // rounding down for ZF and ZT
                var trunc = 0.0;

                if (tickSize.ApproxCompare(0.03125) == 0)
                {
                    trunc = Math.Truncate(value);
                    fraction = Convert.ToInt32(320*Math.Abs(value - trunc) - 0.0001);

                    fraction = fraction/10;

                    if (fraction < 10)
                        priceMarker = trunc + "'0" + fraction;
                    else
                        priceMarker = trunc + "'" + fraction;
                }
                else if (tickSize.ApproxCompare(0.015625) == 0 || tickSize.ApproxCompare(0.0078125) == 0)
                {
                    trunc = Math.Truncate(value);
                    fraction = Convert.ToInt32(320*Math.Abs(value - trunc) - 0.0001);

                    if (fraction < 10)
                        priceMarker = trunc + "'00" + fraction;
                    else if (fraction < 100)
                        priceMarker = trunc + "'0" + fraction;
                    else
                        priceMarker = trunc + "'" + fraction;
                }
            }
            else
                priceMarker = value.ToString(Globals.GetTickFormatString(tickSize));
            return priceMarker;
        }
    }
}