// 
// Copyright (C) 2015, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//
#region Using declarations
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Input;
using System.Windows.Media;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Gui;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Gui.SuperDom;
using NinjaTrader.Data;
using NinjaTrader.NinjaScript;
using NinjaTrader.Core.FloatingPoint;
using NinjaTrader.NinjaScript.DrawingTools;
#endregion

// This namespace holds indicators in this folder and is required. Do not change it.
namespace NinjaTrader.NinjaScript.Indicators.Sim22.Gapless
{
    /// <summary>
    /// The Sim22_MacdGaplessV2 (Moving Average Convergence/Divergence) is a trend following momentum indicator 
    /// that shows the relationship between two moving averages of prices.
    /// @"Intraday Macd that ignores price gaps at the start of the session. Useful only for trend trading from the first bars of the session. Sim22 Dec 2015 NT8b6".
    /// Original concept sourced from the TradeStation forums.
    /// Updated Aug 2016 NT8b12. Tested for tick replay. Now works on EOD charts. Can be used with any input.
    /// </summary>
    public class Sim22_MacdGaplessV2 : Indicator
	{
		private	Series<double>		fastEma;
		private	Series<double>		slowEma;
		private double				constant1;
		private double				constant2;
		private double				constant3;
		private double				constant4;
		private double				constant5;
		private double				constant6;
		private double 				fastEma0;
		private	double 				slowEma0;
		private	double 				macd;
		private	double 				macdAvg;
        private double              invertedTickSize;
	    private string              displayName;
	    private double              prevInput;

		protected override void OnStateChange()
		{
			if (State == State.SetDefaults)
			{
				Description					= @"Macd that ignores price gaps at the start of the session. Sim22 Aug 2016 NT8b12";
				Name						= "Sim22_MACD_GaplessV2";
				Fast						= 12;
				IsSuspendedWhileInactive	= true;
				Slow						= 26;
				Smooth						= 9;
				UseTickSize					= true;
				UseGapless					= true;
				IsOverlay					= false;
				
				AddPlot(Brushes.Green, "MacdGapless");
				AddPlot(Brushes.DarkViolet, "AvgGapless");
				AddPlot(new Stroke(Brushes.Navy, 2), PlotStyle.Bar, "Diff");
				AddLine(Brushes.DarkGray, 0, "ZeroLine");
			}
			else if (State == State.Configure)
			{
				constant1	= 2.0 / (1 + Fast);
				constant2	= (1 - (2.0 / (1 + Fast)));
				constant3	= 2.0 / (1 + Slow);
				constant4	= (1 - (2.0 / (1 + Slow)));
				constant5	= 2.0 / (1 + Smooth);
				constant6	= (1 - (2.0 / (1 + Smooth)));
				
                displayName = "MACDgl " + (UseGapless ? " 'Gapless' " : "") + "(" + Fast + "/" + Slow +")";
            }
            else if (State == State.DataLoaded)
            {
                fastEma = new Series<double>(this);
                slowEma = new Series<double>(this);

                invertedTickSize = UseTickSize ? 1/TickSize : 1;

                if (!(Input is PriceSeries))
                    invertedTickSize = 1;
            }

        }
		
		public override string DisplayName
		{
		    get { return displayName ?? Name; }
		}

		protected override void OnBarUpdate()
		{
		    try
		    {
		        if (Bars == null)
		            return;
		        //for tickreplay. Make sure only updates on each price change
		        if (CurrentBar < 0)
		            return;
		        if (Bars.IsTickReplay && State == State.Historical && !IsFirstTickOfBar && prevInput.CompareTo(Input[0]) == 0)
		            return;

		        prevInput = Input[0];

		        double input0 = Input[0];
				
		        //*************************Added for gapless***************************/

		        double gap = 0d;

		        //Added EOD ability
		        if (CurrentBar > 0 && UseGapless && (!Bars.BarsType.IsIntraday || (Bars.IsFirstBarOfSession && Bars.IsResetOnNewTradingDay)))
		        {
		            //Instrument prices only
		            if (Input is PriceSeries)
		            {
		                gap = Open[0] - Close[1];
		            }
		            else
		            {
		                gap = Input[0] - Input[1];
		            }
		        }
		        //*****************************End added*******************************/

		        if (CurrentBar == 0)
		        {
		            fastEma[0] = input0;
		            slowEma[0] = input0;
		            Value[0] = 0;
		            Avg[0] = 0;
		            Diff[0] = 0;
		            return;
		        }

		        //'gap' is 0 for all other bars other than first bar of session
		        fastEma0 = constant1*input0 + constant2*(fastEma[1] + gap);
		        slowEma0 = constant3*input0 + constant4*(slowEma[1] + gap);
		        macd = (fastEma0 - slowEma0) * invertedTickSize;
		        macdAvg = constant5*macd + constant6*Avg[1];

		        fastEma[0] = fastEma0;
		        slowEma[0] = slowEma0;
		        Value[0] = macd;
		        Avg[0] = macdAvg;
		        Diff[0] = macd - macdAvg;
		    }
		    catch (Exception ex)
		    {
		        Print(Name + " OnBarUpdate: " + ex);
		    }

		}

		#region Properties
		[Browsable(false)]
		[XmlIgnore]
		public Series<double> Avg
		{
			get { return Values[1]; }
		}

		[Browsable(false)]
		[XmlIgnore]
		public Series<double> MACDgl
		{
			get { return Values[0]; }
		}
		
		[Browsable(false)]
		[XmlIgnore]
		public Series<double> Diff
		{
			get { return Values[2]; }
		}

		[Range(1, int.MaxValue), NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Fast", GroupName = "NinjaScriptParameters", Order = 0)]
		public int Fast
		{ get; set; }

		[Range(1, int.MaxValue), NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Slow", GroupName = "NinjaScriptParameters", Order = 1)]
		public int Slow
		{ get; set; }

		[Range(1, int.MaxValue), NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Smooth", GroupName = "NinjaScriptParameters", Order = 2)]
		public int Smooth
		{ get; set; }
		
		[NinjaScriptProperty]
		[Display(Name = "Plot gapless?", GroupName = "NinjaScriptParameters", Order = 3)]
		public bool UseGapless
		{ get; set; }
		
		[NinjaScriptProperty]
		[Display(Name = "Use tick size?", Description = "Divide by 'tick size' to compare with other instruments" ,GroupName = "NinjaScriptParameters", Order = 4)]
		public bool UseTickSize
		{ get; set; }
		
		
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.

namespace NinjaTrader.NinjaScript.Indicators
{
	public partial class Indicator : NinjaTrader.Gui.NinjaScript.IndicatorRenderBase
	{
		private Sim22.Gapless.Sim22_MacdGaplessV2[] cacheSim22_MacdGaplessV2;
		public Sim22.Gapless.Sim22_MacdGaplessV2 Sim22_MacdGaplessV2(int fast, int slow, int smooth, bool useGapless, bool useTickSize)
		{
			return Sim22_MacdGaplessV2(Input, fast, slow, smooth, useGapless, useTickSize);
		}

		public Sim22.Gapless.Sim22_MacdGaplessV2 Sim22_MacdGaplessV2(ISeries<double> input, int fast, int slow, int smooth, bool useGapless, bool useTickSize)
		{
			if (cacheSim22_MacdGaplessV2 != null)
				for (int idx = 0; idx < cacheSim22_MacdGaplessV2.Length; idx++)
					if (cacheSim22_MacdGaplessV2[idx] != null && cacheSim22_MacdGaplessV2[idx].Fast == fast && cacheSim22_MacdGaplessV2[idx].Slow == slow && cacheSim22_MacdGaplessV2[idx].Smooth == smooth && cacheSim22_MacdGaplessV2[idx].UseGapless == useGapless && cacheSim22_MacdGaplessV2[idx].UseTickSize == useTickSize && cacheSim22_MacdGaplessV2[idx].EqualsInput(input))
						return cacheSim22_MacdGaplessV2[idx];
			return CacheIndicator<Sim22.Gapless.Sim22_MacdGaplessV2>(new Sim22.Gapless.Sim22_MacdGaplessV2(){ Fast = fast, Slow = slow, Smooth = smooth, UseGapless = useGapless, UseTickSize = useTickSize }, input, ref cacheSim22_MacdGaplessV2);
		}
	}
}

namespace NinjaTrader.NinjaScript.MarketAnalyzerColumns
{
	public partial class MarketAnalyzerColumn : MarketAnalyzerColumnBase
	{
		public Indicators.Sim22.Gapless.Sim22_MacdGaplessV2 Sim22_MacdGaplessV2(int fast, int slow, int smooth, bool useGapless, bool useTickSize)
		{
			return indicator.Sim22_MacdGaplessV2(Input, fast, slow, smooth, useGapless, useTickSize);
		}

		public Indicators.Sim22.Gapless.Sim22_MacdGaplessV2 Sim22_MacdGaplessV2(ISeries<double> input , int fast, int slow, int smooth, bool useGapless, bool useTickSize)
		{
			return indicator.Sim22_MacdGaplessV2(input, fast, slow, smooth, useGapless, useTickSize);
		}
	}
}

namespace NinjaTrader.NinjaScript.Strategies
{
	public partial class Strategy : NinjaTrader.Gui.NinjaScript.StrategyRenderBase
	{
		public Indicators.Sim22.Gapless.Sim22_MacdGaplessV2 Sim22_MacdGaplessV2(int fast, int slow, int smooth, bool useGapless, bool useTickSize)
		{
			return indicator.Sim22_MacdGaplessV2(Input, fast, slow, smooth, useGapless, useTickSize);
		}

		public Indicators.Sim22.Gapless.Sim22_MacdGaplessV2 Sim22_MacdGaplessV2(ISeries<double> input , int fast, int slow, int smooth, bool useGapless, bool useTickSize)
		{
			return indicator.Sim22_MacdGaplessV2(input, fast, slow, smooth, useGapless, useTickSize);
		}
	}
}

#endregion
