#region Using declarations
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Input;
using System.Windows.Media;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Gui;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Gui.SuperDom;
using NinjaTrader.Data;
using NinjaTrader.NinjaScript;
using NinjaTrader.Core.FloatingPoint;
using NinjaTrader.NinjaScript.DrawingTools;
#endregion

//This namespace holds Indicators in this folder and is required. Do not change it. 
namespace NinjaTrader.NinjaScript.Indicators.Sim22
{
	/// <summary>
	/// AggregateM 2009 by David Varadi - Ported by Sim22 NT8b10 Mar 2016
	/// http://cssanalytics.wordpress.com/2009/11/05/trend-or-mean-reversion-why-make-a-choice-the-simple-aggregate-m-indicator/
	/// 
	/// I have changed David's original terms to avoid confusion:
	/// 
	/// Long and short periods (252 & 10) now are 'Slow Period and Fast period'.
	/// Back and front weighting (0.0-1.0) is now 'CurrentBarWeighting' (0-100%) ie. front weighting only. 'BackBarWeighting' will default to 100 - CurrentBarWeighting.
	/// I have also added trend weighting to increase/decrease trend/correction emphasis. Set to 50% if you do not want any change to the original.
	/// 
	/// Update to V2 NT8.0.2 Dec 2016:
	/// Completely changed the calculation method to dramatically reduce computational load when using OnEachTick or OnPriceChange.
	/// </summary>
	public class Sim22_AggregateMv2 : Indicator
	{
		private Series<double>	m;
	    private bool isInstrumentInput = true; //added Aug 2016

        //TickReplay
        private Calculate tempCalculate = Calculate.OnPriceChange;
	    private double prevInput;

	    private List<double> highSlowList;
	    private List<double> lowSlowList;
	    private List<double> closeSlowList;

	    private List<double> highFastList;
	    private List<double> lowFastList;
	    private List<double> closeFastList;

	    private List<double> hlcSlowList;
	    private List<double> hlcFastList;

        protected override void OnStateChange()
		{
			if (State == State.SetDefaults)
			{
				Description							= @"AggregateM 2009 by David Varadi - Ported by Sim22 NT8b10 Mar 2016";
				Name								= "Sim22_AggregateM++V2";
				Calculate							= Calculate.OnPriceChange;
				IsOverlay							= false;
				DisplayInDataBox					= true;
				DrawOnPricePanel					= true;
				DrawHorizontalGridLines				= true;
				DrawVerticalGridLines				= true;
				PaintPriceMarkers					= true;
				ScaleJustification					= ScaleJustification.Right;
				IsSuspendedWhileInactive			= true;
				
				SlowPeriod							= 252;
				FastPeriod							= 10;
				CurrentBarWeighting					= 60; //emphasis on currentbar
				//Added:
				TrendWeighting						= 50; //trend favourable
				
				AddPlot(new Stroke(Brushes.Black, 2f), PlotStyle.Line, "AggM");
				AddLine(new Stroke(Brushes.DimGray, DashStyleHelper.Dot, 1f), 50, "ZeroLine");
				AddLine(new Stroke(Brushes.Red, DashStyleHelper.Dot, 2f), 5, "Lower");
				AddLine(new Stroke(Brushes.Green, DashStyleHelper.Dot, 2f), 95, "Upper");	
			}
			else if (State == State.Configure)
			{
                //Save default Calculate
                tempCalculate = Calculate;

                highSlowList = new List<double>();
                lowSlowList = new List<double>();
                closeSlowList = new List<double>();

                highFastList = new List<double>();
                lowFastList = new List<double>();
                closeFastList = new List<double>();
                hlcSlowList = new List<double>();
                hlcFastList = new List<double>();
            }
            else if (State == State.DataLoaded)
            {
                m = new Series<double>(this, MaximumBarsLookBack.Infinite);
                //Added Aug 2016. Can be used on any input series
                isInstrumentInput = Input is PriceSeries;
               
            }
            else if (State == State.Historical)
            {
                //Used for tick replay: This indicator will only correctly calculate once per bar historically when using tick replay
                if (Bars != null)
                    if (Bars.IsTickReplay)
                        Calculate = Calculate.OnBarClose;

            }
            else if (State == State.Transition)
            {
                //Return to default Calculate
                Calculate = tempCalculate;
            }
        }
		
		public override string DisplayName
		{
			get { return Name + string.Format("\r\rPeriods \t\t= ({0}/{1})" + "\rTrd Weighting \t= ({2}/{3})" + "\rBar Weighting \t= ({4}/{5})", SlowPeriod, FastPeriod, TrendWeighting, 100 - TrendWeighting, 100 - CurrentBarWeighting, CurrentBarWeighting); }
			
		}
		
		public override string FormatPriceMarker(double price)
		{
		    return price.ToString("N0") + " %";
		}
		
		public override void OnCalculateMinMax()
		{    
			// Maintains a fixed scale to show full range of AggM.
			MinValue = 0.0;
		   	MaxValue = 100.00;
		} 
		
		protected override void OnBarUpdate()
		{
		    try
		    {

		        /*
           Explanation: 

           Find the current rank of the current close versus the combined HLC values of the n period of prior bars.
           In this version I have streamlined the calculations to reduce processing times to 10% of the original on all market events besides the first tick of the bar.

           Reference:

           http://cssanalytics.wordpress.com/2009/11/05/trend-or-mean-reversion-why-make-a-choice-the-simple-aggregate-m-indicator/
           https://support.office.com/en-us/article/PERCENTRANK-function-f1b5836c-9619-4847-9fc9-080ec9024442

           */

		        if (CurrentBar <= 0)
		            return;

		        double close0 = isInstrumentInput ? Close[0] : Input[0];

		        if (IsFirstTickOfBar)
		        {

		            /*Note:
                    I made this change from the original AggMv1 code due to excessive calculations if using OnEachTick or OnPriceChange.
                    Basically the older code would loop through every market update as much as SlowPeriod*3 number of values, not to mention the Fast values as well.
                    I have therefore saved the required past values into several Lists at the first tick of a new bar.
                    For all other market events it is only thus required to compare the current HLC values to the lists values.
                    Please see V1 for the concept.
                
                */
		            double close1;
		            // Add prior bar HL values
		            if (isInstrumentInput)
		            {
		                double high1 = High[1];
		                double low1 = Low[1];
		                close1 = Close[1];

		                highSlowList.Add(high1);
		                lowSlowList.Add(low1);

		                highFastList.Add(high1);
		                lowFastList.Add(low1);
		            }
		            else
		            {
		                close1 = Input[1];
		            }
		            // Add prior bar Cl values
		            closeSlowList.Add(close1);
		            closeFastList.Add(close1);

		            //Remove the first value on the lists. Much like the SMA calculation.
		            if (CurrentBar >= SlowPeriod)
		            {
		                if (isInstrumentInput)
		                {
		                    highSlowList.RemoveAt(0);
		                    lowSlowList.RemoveAt(0);
		                }
		                closeSlowList.RemoveAt(0);
		            }

		            if (CurrentBar >= FastPeriod)
		            {
		                if (isInstrumentInput)
		                {
		                    highFastList.RemoveAt(0);
		                    lowFastList.RemoveAt(0);
		                }
		                closeFastList.RemoveAt(0);
		            }

		            //Combine all HLC values to a hlcList for sorting.
		            hlcSlowList = closeSlowList.ToList();
		            hlcFastList = closeFastList.ToList();

		            if (isInstrumentInput)
		            {
		                hlcSlowList.AddRange(highSlowList);
		                hlcSlowList.AddRange(lowSlowList);

		                hlcFastList.AddRange(highFastList);
		                hlcFastList.AddRange(lowFastList);
		            }
                    // not needed to sort?
		            //hlcSlowList.Sort();
		            //hlcFastList.Sort();
		        }


		        //for tickreplay. Make sure historical values only update on each price change
		        if (Bars.IsTickReplay && State == State.Historical && !IsFirstTickOfBar && prevInput.CompareTo(Input[0]) == 0)
		            return;

		        prevInput = Input[0];

		        //On every market event count how many values are below the current Cl.
                double slowCount = hlcSlowList.AsParallel().Count(x => close0 > x);

                //The current close will never be greater than the current high so we ignore that.
                //However if the current close is greater than the current low then add to the rank count.
                if (isInstrumentInput && close0 > Low[0])
		            slowCount++;

		        double hlc1 = 100*slowCount/(hlcSlowList.Count + (isInstrumentInput ? 2 : 0)); //+2 is to include current high and low

		        //On every market event count how many values are below the current Cl.
                double fastCount = hlcFastList.AsParallel().Count(x => close0 > x);

                //The current close will never be greater than the current high so we ignore that.
                //However if the current close is greater than the current low then add to the rank count.
                if (isInstrumentInput && close0 > Low[0])
		            fastCount++;

		        double hlc2 = 100*fastCount/(hlcFastList.Count + (isInstrumentInput ? 2 : 0)); //+2 is to include current high and low

		        //Original is balance between the two
		        //m[0]	= (hlc1 + hlc2) * 0.5; 

		        //Sim22
		        m[0] = (hlc1*TrendWeighting*0.01 + hlc2*(100 - TrendWeighting)*0.01);

		        double agm = ((100 - CurrentBarWeighting)*0.01*m[1]) + (CurrentBarWeighting*0.01*m[0]);

		        Value[0] = agm;


		    }
            catch (Exception ex)
            {
                Print(Name + " OnBarUpdate: " + ex);
            }
        }
		
		#region Properties
		
		[Browsable(false)]
		[XmlIgnore()]
		public Series<double> AggM
		{
			get { return Value; }
		}
		
		[Range(2, int.MaxValue), NinjaScriptProperty]
		[Display(Description="Select larger number of bars (Default = 252)", Name = "Slow period", GroupName = "NinjaScriptParameters", Order = 0)]
		public int SlowPeriod
		{ get; set; }
		
		[Range(2, int.MaxValue), NinjaScriptProperty]
		[Display(Description="Select smaller number of bars (Default = 10)", Name = "Fast period", GroupName = "NinjaScriptParameters", Order = 1)]
		public int FastPeriod
		{ get; set; }
		
		[Range(0, 100), NinjaScriptProperty]
		[Display(Description="Smoothing between current bar and last bar", Name = "Current bar weighting % (0-100)", GroupName = "NinjaScriptParameters", Order = 2)]
		public int CurrentBarWeighting
		{ get; set; }
		
		[Range(0, 100), NinjaScriptProperty]
		[Display(Description="New: Increase for more trend based smoothing (0-100)", Name = "New: Trend weighting % (0-100)", GroupName = "NinjaScriptParameters", Order = 3)]
		public int TrendWeighting
		{ get; set; }
		

		#endregion
	}
	
}

#region NinjaScript generated code. Neither change nor remove.

namespace NinjaTrader.NinjaScript.Indicators
{
	public partial class Indicator : NinjaTrader.Gui.NinjaScript.IndicatorRenderBase
	{
		private Sim22.Sim22_AggregateMv2[] cacheSim22_AggregateMv2;
		public Sim22.Sim22_AggregateMv2 Sim22_AggregateMv2(int slowPeriod, int fastPeriod, int currentBarWeighting, int trendWeighting)
		{
			return Sim22_AggregateMv2(Input, slowPeriod, fastPeriod, currentBarWeighting, trendWeighting);
		}

		public Sim22.Sim22_AggregateMv2 Sim22_AggregateMv2(ISeries<double> input, int slowPeriod, int fastPeriod, int currentBarWeighting, int trendWeighting)
		{
			if (cacheSim22_AggregateMv2 != null)
				for (int idx = 0; idx < cacheSim22_AggregateMv2.Length; idx++)
					if (cacheSim22_AggregateMv2[idx] != null && cacheSim22_AggregateMv2[idx].SlowPeriod == slowPeriod && cacheSim22_AggregateMv2[idx].FastPeriod == fastPeriod && cacheSim22_AggregateMv2[idx].CurrentBarWeighting == currentBarWeighting && cacheSim22_AggregateMv2[idx].TrendWeighting == trendWeighting && cacheSim22_AggregateMv2[idx].EqualsInput(input))
						return cacheSim22_AggregateMv2[idx];
			return CacheIndicator<Sim22.Sim22_AggregateMv2>(new Sim22.Sim22_AggregateMv2(){ SlowPeriod = slowPeriod, FastPeriod = fastPeriod, CurrentBarWeighting = currentBarWeighting, TrendWeighting = trendWeighting }, input, ref cacheSim22_AggregateMv2);
		}
	}
}

namespace NinjaTrader.NinjaScript.MarketAnalyzerColumns
{
	public partial class MarketAnalyzerColumn : MarketAnalyzerColumnBase
	{
		public Indicators.Sim22.Sim22_AggregateMv2 Sim22_AggregateMv2(int slowPeriod, int fastPeriod, int currentBarWeighting, int trendWeighting)
		{
			return indicator.Sim22_AggregateMv2(Input, slowPeriod, fastPeriod, currentBarWeighting, trendWeighting);
		}

		public Indicators.Sim22.Sim22_AggregateMv2 Sim22_AggregateMv2(ISeries<double> input , int slowPeriod, int fastPeriod, int currentBarWeighting, int trendWeighting)
		{
			return indicator.Sim22_AggregateMv2(input, slowPeriod, fastPeriod, currentBarWeighting, trendWeighting);
		}
	}
}

namespace NinjaTrader.NinjaScript.Strategies
{
	public partial class Strategy : NinjaTrader.Gui.NinjaScript.StrategyRenderBase
	{
		public Indicators.Sim22.Sim22_AggregateMv2 Sim22_AggregateMv2(int slowPeriod, int fastPeriod, int currentBarWeighting, int trendWeighting)
		{
			return indicator.Sim22_AggregateMv2(Input, slowPeriod, fastPeriod, currentBarWeighting, trendWeighting);
		}

		public Indicators.Sim22.Sim22_AggregateMv2 Sim22_AggregateMv2(ISeries<double> input , int slowPeriod, int fastPeriod, int currentBarWeighting, int trendWeighting)
		{
			return indicator.Sim22_AggregateMv2(input, slowPeriod, fastPeriod, currentBarWeighting, trendWeighting);
		}
	}
}

#endregion
