#region Using declarations

using System;
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using System.IO;
using System.Windows.Media;
using System.Xml.Serialization;
using NinjaTrader.Core;
using NinjaTrader.Gui;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Gui.NinjaScript;
using NinjaTrader.Gui.Tools;
using NinjaTrader.NinjaScript.AddOns;
using NinjaTrader.NinjaScript.DrawingTools;
using NinjaTrader.NinjaScript.Indicators.Sim22;
using NinjaTrader.NinjaScript.Indicators.Sim22.Gapless;
using SharpDX;
using SharpDX.Direct2D1;
using Sim22PriceFormatExtensions;
using Brush = System.Windows.Media.Brush;
using Color = System.Windows.Media.Color;
using Ellipse = SharpDX.Direct2D1.Ellipse;
using GradientStop = SharpDX.Direct2D1.GradientStop;
using GradientStopCollection = SharpDX.Direct2D1.GradientStopCollection;
using LinearGradientBrush = System.Windows.Media.LinearGradientBrush;
using Point = System.Windows.Point;
using RadialGradientBrush = SharpDX.Direct2D1.RadialGradientBrush;
using SolidColorBrush = System.Windows.Media.SolidColorBrush;

#endregion


//This namespace holds Indicators in this folder and is required. Do not change it. 
public enum MacdBbv4SignalTypeEnum
{
    [Description("Bollinger cross")]
    BollingerCross,
    [Description("Average cross")]
    AverageCross,
    [Description("Zeroline cross")]
    ZeroLineCross
}

public enum MacdBbv4BandDevTypeEnum
{
    [Description("Standard Deviation")]
    StandardDeviation,
    [Description("Mean Absolute Deviation (MAD)")]
    MeanAbsoluteDeviation
}
public enum MacdBbV4DisplayTypeEnum
{
    Standard,
    Pullback,
    Gradient
}

namespace NinjaTrader.NinjaScript.Indicators.Sim22
{
    /// <summary>
    /// MacdBB. Sim22 NT8b6
    /// Updated Aug 2016 NT8b12.
    /// v2: Changed the gradient code to use the AggregateM mean reversion indicator instead of a stochastic. Now works
    /// with tick replay whilst using the gradient. Use of brush addon.
    /// 
    /// V3: Using OnRender to allow more flexibility with the plots. Added radial gradiant brush.
    /// Updated Feb 2017 NT8.0.4
    /// V4: Added 'Pullback' setting to smooth out signals:
    /// eg. if the dots are up but inside the Bolls BUT below the average (possible pullback), they will color the down inside color instead of up inside color.
    /// </summary>
    public class Sim22_MacdBBGaplessV4 : Indicator
    {
        //Brushes
        private readonly Brush[] _macdBackgroundBrushes = new Brush[4];
        private readonly Brush[] _macdBarBrushes = new Brush[4];
        private readonly Brush[] _macdBrushes = new Brush[4];
        private readonly Brush[] _macdOutlineBrushes = new Brush[4];

        private Series<int> _signalIntSeries;
        private LinearGradientBrush _linearGradientBrush;
        private Point _startPoint;
        private Point _endPoint;
        private Color _color;

        private StdDev _stdDev;
        private Sim22_MeanAbsDev _sim22MeanAbsDev;
        private Sim22_AggregateMv2 _sim22AggregateMv2;
        private Sim22_MacdGaplessV2 _sim22MacdGaplessV2;
        private Series<double> _macdAggMGradValue;

        private string _displayName;
        private double _prevInput;
        
        //Alerts
        private bool _soundUpPlayed;
        private string _longSoundPath;
        private bool _longSoundFileExists;
        private string _shortSoundPath;
        private bool _shortSoundFileExists;
        private bool _soundDownPlayed;
      
        //TickReplay
        private Calculate _tempCalculate = Calculate.OnPriceChange;

        private bool _isUpEntry;
        private bool _isDownEntry;


        protected override void OnStateChange()
        {
            switch (State)
            {
                case State.SetDefaults:

                    Description = @"MacdBB with session open gap-ignoring option. Sim22 NT8.0.4 Feb 2017.";
                    Name = "Sim22_MacdBBGaplessV4";
                    Calculate = Calculate.OnPriceChange;
                    IsOverlay = false;
                    DisplayInDataBox = true;
                    DrawOnPricePanel = false;
                    PaintPriceMarkers = true;
                    DrawHorizontalGridLines = false;
                    DrawVerticalGridLines = false;
                    ScaleJustification = ScaleJustification.Right;
                    IsSuspendedWhileInactive = false;

                    FastMaPeriod = 12;
                    SlowMaPeriod = 26;
                    AverageBollPeriod = 10;
                    NumberDeviations = 1.0;

                    BackGroundOpacity = 40;
                    BandOpacity = 20;
                    BarOpacity = 80;
                    OutlineTonePercent = 120;
                    UseTickSize = true;
                    UseGapless = true;
                    ShowHistogram = false;
                    PaintRadialGradient = true;
                    PlotPaintBars = true;
                    PaintOutline = true;
                    PaintBackground = true;
                    PaintBackgroundAll = false;
                    //PaintConservative = true;
                    PlayCrossSound = true;
                    _soundUpPlayed = false;
                    _soundDownPlayed = false;
                    LongWavFileNameStr = "MacdUp.wav";
                    ShortWavFileNameStr = "MacdDown.wav";
                    VersionStr = "V4.1.0 | Feb 2017";
                    //Added Feb 2017
                    DrawEntrySignals = true;
                    DrawExitSignals = true;
                    MacdBbv4SignalType = MacdBbv4SignalTypeEnum.BollingerCross;

                    MacdBbDisplayType = MacdBbV4DisplayTypeEnum.Standard;
                    FormatNumberDecimalPlaces = DecimalPlacesEnum.SameAsInstrument;
                
                    //AggM for gradient
                    FastPeriodAggM = 10;
                    SlowPeriodAggM = 252;
                    CurrentBarWeightingAggM = 60;
                    TrendWeightingAggM = 75;

                    UpBrushOut = Brushes.Lime;
                    UpBrushIn = new SolidColorBrush(Color.FromRgb(0, 140, 0));
                    DnBrushOut = Brushes.Red;
                    DnBrushIn = new SolidColorBrush(Color.FromRgb(150, 0, 0));
                    UpBrushGrad = Brushes.Lime;
                    DnBrushGrad = Brushes.Blue;
                    BandBrush = Brushes.Gray;

                    UpBrushOut.Freeze();
                    UpBrushIn.Freeze();
                    DnBrushOut.Freeze();
                    DnBrushIn.Freeze();
                    UpBrushGrad.Freeze();
                    DnBrushGrad.Freeze();
                    BandBrush.Freeze();

                    AddPlot(new Stroke(Brushes.Black, 1), PlotStyle.Dot, "MacdGL"); // not of any visual value
                    AddPlot(new Stroke(Brushes.Black, 10), PlotStyle.Square, "ZeroLine");
                    AddPlot(new Stroke(Brushes.Orange, 4), PlotStyle.Bar, "Histogram");
                    AddPlot(new Stroke(Brushes.Gray, 1), PlotStyle.Line, "UpperBollinger");
                    AddPlot(new Stroke(Brushes.Gray, 1), PlotStyle.Line, "LowerBollinger");
                    AddPlot(new Stroke(Brushes.Black, 1), PlotStyle.Line, "Average"); //Bollinger center
                    AddPlot(new Stroke(Brushes.Black, 1), PlotStyle.Line, "MacdLine");
                    AddPlot(new Stroke(Brushes.Yellow, 10), PlotStyle.Dot, "LineSignalPlot");
                    AddPlot(new Stroke(Brushes.Black, 4), PlotStyle.Dot, "MacdDotFrame");
                    AddPlot(new Stroke(Brushes.LightGray, 3), PlotStyle.Dot, "MacdDot");
                    break;

                case State.Configure:

                    try
                    {
                        //Save default Calculate
                        _tempCalculate = Calculate;

                        // set all brushes to UI values
                        _macdBrushes[0] = UpBrushOut;
                        _macdBrushes[1] = UpBrushIn;
                        _macdBrushes[2] = DnBrushOut;
                        _macdBrushes[3] = DnBrushIn;

                        foreach (Brush brush in _macdBrushes)
                        {
                            brush.Freeze();
                        }

                        for (int i = 0; i < _macdBarBrushes.Length; i++)
                        {
                            _macdBarBrushes[i] = _macdBrushes[i].Clone();
                            _macdBarBrushes[i].Opacity = BarOpacity * 0.01f;
                            _macdBarBrushes[i].Freeze();
                        }

                        for (int i = 0; i < _macdOutlineBrushes.Length; i++)
                        {
                            _macdOutlineBrushes[i] = Sim22_BrushUtilitiesStat.ToneChangeBrush(_macdBrushes[i],
                                OutlineTonePercent);
                            _macdOutlineBrushes[i].Freeze();
                        }

                        for (int i = 0; i < _macdBackgroundBrushes.Length; i++)
                        {
                            _macdBackgroundBrushes[i] = _macdBrushes[i].Clone();
                            _macdBackgroundBrushes[i].Opacity = BackGroundOpacity * 0.01f;
                            _macdBackgroundBrushes[i].Freeze();
                        }

                        Plots[8].Width = Plots[9].Width + 1.0f; //create constant frame around 'MacdDot'

                        //lineargradient call once at startup
                        if (MacdBbDisplayType == MacdBbV4DisplayTypeEnum.Gradient)
                        {
                            _startPoint = new Point(0.0, 0d);
                            _endPoint = new Point(0.0, 100d);

                            _linearGradientBrush = Sim22_BrushUtilitiesStat.MakeLinearGradientBrush(_endPoint, _startPoint, DnBrushGrad, 0.0, UpBrushGrad, 1.0);
                            _linearGradientBrush.Freeze();
                        }

                        try
                        {
                            //Alerts
                            if (PlayCrossSound)
                            {
                                _longSoundPath = Globals.InstallDir + @"sounds\" + LongWavFileNameStr;
                                _shortSoundPath = Globals.InstallDir + @"sounds\" + ShortWavFileNameStr;

                                if (File.Exists(_longSoundPath))
                                    _longSoundFileExists = true;
                           
                                if (File.Exists(_shortSoundPath))
                                    _shortSoundFileExists = true;     
                            }
                        }
                        catch (Exception ex)
                        {
                            Print(Name + " Alert Sound File: " + ex);
                        }
                    }
                    catch (Exception ex)
                    {
                        Print(Name + " Configure: " + ex);
                    }

                    break;

                case State.Historical:

                    //Used for tick replay: This indicator (using the gradient version) will only correctly calculate once per bar historically when using tick replay
                    if (Bars != null)
                        if (Bars.IsTickReplay)
                            Calculate = Calculate.OnBarClose;
                    break;

                case State.DataLoaded:

                    //Note: These indicators and data series have been added here due to best practices changes from NT8.0.4 here: 
                    // https://ninjatrader.com/support/helpGuides/nt8/en-us/?ninjascript_best_practices.htm

                    _sim22MacdGaplessV2 = Sim22_MacdGaplessV2(Input, FastMaPeriod, SlowMaPeriod, AverageBollPeriod, UseGapless, UseTickSize);

                    if (MacdBbv4BandDevType == MacdBbv4BandDevTypeEnum.StandardDeviation)
                        _stdDev = StdDev(_sim22MacdGaplessV2, AverageBollPeriod);
                    else
                        _sim22MeanAbsDev = Sim22_MeanAbsDev(_sim22MacdGaplessV2, AverageBollPeriod);

                    _signalIntSeries = new Series<int>(this, MaximumBarsLookBack.TwoHundredFiftySix);
                    //lineargradient call once at startup
                    if (MacdBbDisplayType == MacdBbV4DisplayTypeEnum.Gradient)
                    {
                        //gradient
                        _macdAggMGradValue = new Series<double>(this, MaximumBarsLookBack.Infinite);
                        _sim22AggregateMv2 = Sim22_AggregateMv2(_macdAggMGradValue, SlowPeriodAggM, FastPeriodAggM, CurrentBarWeightingAggM, TrendWeightingAggM);

                    }
                    break;

                case State.Transition:

                    //Return to default Calculate
                    Calculate = _tempCalculate;

                    break;
            }
        }

        public override string DisplayName
        {
            get
            {
                string signalTypeStr = string.Empty;
                string devTypeStr = string.Empty;

                switch (MacdBbv4SignalType)
                {
                    case MacdBbv4SignalTypeEnum.BollingerCross:
                        signalTypeStr = " BollX ";
                        break;
                    case MacdBbv4SignalTypeEnum.AverageCross:
                        signalTypeStr = " AvgX ";
                        break;
                    case MacdBbv4SignalTypeEnum.ZeroLineCross:
                        signalTypeStr = " ZeroX ";
                        break;
                }

                switch (MacdBbv4BandDevType)
                {
                    case MacdBbv4BandDevTypeEnum.StandardDeviation:
                        devTypeStr = " StdDev";
                        break;
                    case MacdBbv4BandDevTypeEnum.MeanAbsoluteDeviation:
                        devTypeStr = " MAD";
                        break;
                    
                }
                _displayName = "MACDBBv4gl" + (UseGapless ? " Gapless " : "") + "(" + FastMaPeriod + "/" + SlowMaPeriod + "/" + AverageBollPeriod + ")" + devTypeStr + signalTypeStr;
                // If any are null then use the string below.
                return _displayName ?? (Name ?? "MACDBBv4gl");
            }
        }

        public override string FormatPriceMarker(double value)
        {
            /* using Sim22PriceFormatExtensions in the Addons folder. 
            When set to 'SameAsInstrument' I chose to not round to the nearest ticksize.
            You can use this instead if you chose to:
            return  Instrument.MasterInstrument.RoundDownToTickSize(value);
            */

            return FormatNumberDecimalPlaces == Sim22PriceFormatExtensions.DecimalPlacesEnum.SameAsInstrument ? value.ToStringSameAsInstrument(TickSize) : value.ToStringDecimalPlaces(TickSize, FormatNumberDecimalPlaces);
        }

        protected override void OnBarUpdate() 
        {
            try
            {
                if (Bars == null)
                    return;
                if (CurrentBar < 0)
                    return;
                //for tickreplay. Make sure historical values only update on each price change
                if (Bars.IsTickReplay && State == State.Historical && !IsFirstTickOfBar && _prevInput.CompareTo(Input[0]) == 0)
                    return;

                _prevInput = Input[0];

                if (PlayCrossSound)
                {
                    if (!_longSoundFileExists)
                        Draw.TextFixed(this, "LongSoundFileNotExists0", "Your sound file " + LongWavFileNameStr + " does not exist in the folder \r" + _longSoundPath.Replace(LongWavFileNameStr, "") + ". \rPlease disable your alerts or replace the file.", TextPosition.TopLeft, Brushes.Black, ChartControl.Properties.LabelFont, Brushes.Gold, Brushes.Gold, 70);

                    if (!_shortSoundFileExists)
                        Draw.TextFixed(this, "ShortSoundFileNotExists0", "Your sound file " + ShortWavFileNameStr + " does not exist in the folder \r" + _shortSoundPath.Replace(ShortWavFileNameStr, "") + ". \rPlease disable your alerts or replace the file.", TextPosition.BottomLeft, Brushes.Black, ChartControl.Properties.LabelFont, Brushes.Gold, Brushes.Gold, 70);
                }

                //Macd
                double macdValue = _sim22MacdGaplessV2[0];
                MacdGl[0] = macdValue;
                MacdLine[0] = macdValue;
                MacdDotFrame[0] = macdValue;
                MacdDot[0] = macdValue;

                //Average
                double avg = _sim22MacdGaplessV2.Avg[0];
                Average[0] = avg;
                ZeroLine[0] = 0.0;

                if (ShowHistogram)
                    Histogram[0] = macdValue - avg;

                //Bollingers
                double devValue = MacdBbv4BandDevType == MacdBbv4BandDevTypeEnum.StandardDeviation ? _stdDev[0] : _sim22MeanAbsDev[0];
                
                UpperBollinger[0] = avg + NumberDeviations*devValue;
                LowerBollinger[0] = avg - NumberDeviations*devValue;

                Draw.Region(this, "macdbbBand", CurrentBar, 0, UpperBollinger, LowerBollinger, null, BandBrush, BandOpacity, Displacement);

                //gradient normalizing
                if (MacdBbDisplayType == MacdBbV4DisplayTypeEnum.Gradient)
                {
                    try
                    {
                        _macdAggMGradValue[0] = macdValue;

                        var macdGradientValue = _sim22AggregateMv2[0];

                        var macdPoint = new Point(0.0, macdGradientValue);

                        // Find the color of the macd value within the Lgb.
                        _color = Sim22_BrushUtilitiesStat.GetColorAtPoint(_linearGradientBrush, _startPoint, _endPoint, macdPoint);
                        PlotBrushes[9][0] = new SolidColorBrush(_color);
                        PlotBrushes[1][0] = new SolidColorBrush(_color);

                        var backOpacity = (byte) Math.Round(BackGroundOpacity*2.55, 0); // 0-255

                        if (PlotPaintBars) BarBrush = BarOpacity == 100 ? new SolidColorBrush(_color) : new SolidColorBrush(Color.FromScRgb(BarOpacity*0.01f, _color.ScR, _color.ScG, _color.ScB));

                        if (PaintOutline) CandleOutlineBrush = OutlineTonePercent == 100 ? new SolidColorBrush(_color) : Sim22_BrushUtilitiesStat.ToneChangeColorToBrush(_color, OutlineTonePercent);

                        if (PaintBackground)
                        {
                            BackBrush = new SolidColorBrush(Color.FromArgb(backOpacity, _color.R, _color.G, _color.B));
                        }

                        if (PaintBackgroundAll)
                        {
                            BackBrushAll = new SolidColorBrush(Color.FromArgb(backOpacity, _color.R, _color.G, _color.B));
                        }
                    }
                    catch (Exception ex)
                    {
                        Print(Name + " OnBarUpdate: " + ex);
                    }
                }
                else
                {
                    if (IsRising(MacdGl))
                    {
                        if (macdValue <= UpperBollinger[0])
                        {
                            if (MacdBbDisplayType == MacdBbV4DisplayTypeEnum.Pullback && macdValue < avg)
                            {
                                PlotBrushes[9][0] = DnBrushIn;
                                if (PlotPaintBars) BarBrush = _macdBarBrushes[3];
                                if (PaintOutline) CandleOutlineBrush = _macdOutlineBrushes[3];
                            }
                            else
                            {
                                PlotBrushes[9][0] = UpBrushIn; //dots
                                if (PlotPaintBars) BarBrush = _macdBarBrushes[1];
                                if (PaintOutline) CandleOutlineBrush = _macdOutlineBrushes[1];
                            }
                        }
                        else
                        {
                            PlotBrushes[9][0] = UpBrushOut;
                            if (PlotPaintBars) BarBrush = _macdBarBrushes[0];
                            if (PaintOutline) CandleOutlineBrush = _macdOutlineBrushes[0];
                        }
                    }
                    else
                    {
                        if (macdValue >= LowerBollinger[0])
                        {
                            if (MacdBbDisplayType == MacdBbV4DisplayTypeEnum.Pullback && macdValue > avg)
                            {
                                PlotBrushes[9][0] = UpBrushIn;
                                if (PlotPaintBars) BarBrush = _macdBarBrushes[1];
                                if (PaintOutline) CandleOutlineBrush = _macdOutlineBrushes[1];
                            }
                            else
                            {
                                PlotBrushes[9][0] = DnBrushIn;
                                if (PlotPaintBars) BarBrush = _macdBarBrushes[3];
                                if (PaintOutline) CandleOutlineBrush = _macdOutlineBrushes[3];
                            }
                        }
                        else
                        {
                            PlotBrushes[9][0] = DnBrushOut;
                            if (PlotPaintBars) BarBrush = _macdBarBrushes[2];
                            if (PaintOutline) CandleOutlineBrush = _macdOutlineBrushes[2];
                        }
                    }

                    try
                    {
                        switch (MacdBbv4SignalType)
                        {
                            case MacdBbv4SignalTypeEnum.AverageCross:

                                if (PaintBackground) BackBrush = macdValue > avg ? _macdBackgroundBrushes[0] : _macdBackgroundBrushes[2];
                                if (PaintBackgroundAll) BackBrushAll = macdValue > avg ? _macdBackgroundBrushes[0] : _macdBackgroundBrushes[2];
                                break;

                            case MacdBbv4SignalTypeEnum.BollingerCross:

                                if (PaintBackground)
                                {
                                    if (macdValue > UpperBollinger[0])
                                        BackBrush = _macdBackgroundBrushes[0];
                                    else if (macdValue < LowerBollinger[0])
                                        BackBrush = _macdBackgroundBrushes[2];
                                }
                                if (PaintBackgroundAll)
                                {
                                    if (macdValue > UpperBollinger[0])
                                        BackBrushAll = _macdBackgroundBrushes[0];
                                    else if (macdValue < LowerBollinger[0])
                                        BackBrushAll = _macdBackgroundBrushes[2];
                                }
                                break;

                            case MacdBbv4SignalTypeEnum.ZeroLineCross:

                                if (PaintBackground) BackBrush = macdValue > 0.0 ? _macdBackgroundBrushes[0] : _macdBackgroundBrushes[2];
                                if (PaintBackgroundAll) BackBrushAll = macdValue > 0.0 ? _macdBackgroundBrushes[0] : _macdBackgroundBrushes[2];
                                break;
                        }
                        
                        PlotBrushes[5][0] = macdValue > avg ? UpBrushOut : DnBrushOut;
                        PlotBrushes[1][0] = macdValue > 0.0 ? UpBrushOut : DnBrushOut;
                    }
                    catch (Exception ex)
                    {
                        Print(Name + " OnBarUpdate: " + ex);
                    }
                }

                bool playSound = State == State.Realtime && PlayCrossSound;

                switch (MacdBbv4SignalType)
                {
                    case MacdBbv4SignalTypeEnum.BollingerCross:

                        if (CrossAbove(MacdGl, UpperBollinger, 1))
                        {
                            LineSignalPlot[0] = macdValue;
                            _signalIntSeries[0] = 1;

                            if (playSound && _longSoundFileExists && !_soundUpPlayed)
                            {
                                PlaySound(_longSoundPath);

                                _soundUpPlayed = true;
                                _soundDownPlayed = false;
                            }
                            if (DrawEntrySignals)
                            {
                                Draw.TriangleUp(this, "UpEntry" + CurrentBar, true, 0, Low[0] - TickSize,
                                    PlotBrushes[9][0],
                                    true);
                                _isUpEntry = true;
                            }
                        }
                        else if (CrossBelow(MacdGl, LowerBollinger, 1))
                        {
                            LineSignalPlot[0] = macdValue;
                            _signalIntSeries[0] = -1;

                            if (playSound && _shortSoundFileExists && !_soundDownPlayed)
                            {
                                PlaySound(_shortSoundPath);

                                _soundDownPlayed = true;
                                _soundUpPlayed = false;
                            }

                            if (DrawEntrySignals)
                            {
                                Draw.TriangleDown(this, "DownEntry" + CurrentBar, true, 0, High[0] + TickSize,
                                    PlotBrushes[9][0],
                                    true);
                                _isDownEntry = true;
                            }
                        }
                        else
                        {
                            LineSignalPlot.Reset();
                            _signalIntSeries[0] = 0;
                        }
                        break;

                    case MacdBbv4SignalTypeEnum.AverageCross:

                        if (CrossAbove(MacdGl, Average, 1))
                        {
                            LineSignalPlot[0] = macdValue;
                            _signalIntSeries[0] = 1;

                            if (playSound && _longSoundFileExists && !_soundUpPlayed)
                            {
                                PlaySound(_longSoundPath);

                                _soundUpPlayed = true;
                                _soundDownPlayed = false;
                            }
                            if (DrawEntrySignals)
                                Draw.TriangleUp(this, "UpEntry" + CurrentBar, true, 0, Low[0] - TickSize, PlotBrushes[9][0], true);
                        }
                        else if (CrossBelow(MacdGl, Average, 1))
                        {
                            LineSignalPlot[0] = macdValue;
                            _signalIntSeries[0] = -1;

                            if (playSound && _shortSoundFileExists && !_soundDownPlayed)
                            {
                                PlaySound(_shortSoundPath);

                                _soundDownPlayed = true;
                                _soundUpPlayed = false;
                            }
                            if (DrawEntrySignals)
                                Draw.TriangleDown(this, "DownEntry" + CurrentBar, true, 0, High[0] + TickSize, PlotBrushes[9][0], true);

                        }
                        else
                        {
                            LineSignalPlot.Reset();
                            _signalIntSeries[0] = 0;
                        }
                        break;

                    case MacdBbv4SignalTypeEnum.ZeroLineCross:

                        if (CrossAbove(MacdGl, 0.0, 1))
                        {
                            LineSignalPlot[0] = macdValue;
                            _signalIntSeries[0] = 1;

                            if (playSound && _longSoundFileExists && !_soundUpPlayed)
                            {
                                PlaySound(_longSoundPath);

                                _soundUpPlayed = true;
                                _soundDownPlayed = false;
                            }
                            if (DrawEntrySignals)
                                Draw.TriangleUp(this, "UpEntry" + CurrentBar, true, 0, Low[0] - TickSize,PlotBrushes[9][0], true);

                        }
                        else if (CrossBelow(MacdGl, 0.0, 1))
                        {
                            LineSignalPlot[0] = macdValue;
                            _signalIntSeries[0] = -1;

                            if (playSound && _shortSoundFileExists && !_soundDownPlayed)
                            {
                                PlaySound(_shortSoundPath);

                                _soundDownPlayed = true;
                                _soundUpPlayed = false;
                            }
                            if (DrawEntrySignals)
                                Draw.TriangleDown(this, "DownEntry" + CurrentBar, true, 0, High[0] + TickSize, PlotBrushes[9][0], true);

                        }
                        else
                        {
                            LineSignalPlot.Reset();
                            _signalIntSeries[0] = 0;
                        }
                        break;
                }

                if (DrawExitSignals)
                {
                    if (!DrawEntrySignals)
                    {
                        _isUpEntry = true;
                        _isDownEntry = true;
                    }

                    if (MacdBbv4SignalType == MacdBbv4SignalTypeEnum.BollingerCross)
                    {
                        if (CrossBelow(MacdGl, UpperBollinger, 1) && _isUpEntry)
                        {
                            Draw.Dot(this, "UpExit" + CurrentBar, true, 0, High[0] + TickSize, PlotBrushes[9][0], true);
                            _isUpEntry = false;
                        }

                        if (CrossAbove(MacdGl, LowerBollinger, 1) && _isDownEntry)
                        {
                            Draw.Dot(this, "DownExit" + CurrentBar, true, 0, Low[0] - TickSize, PlotBrushes[9][0], true);
                            _isDownEntry = false;
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Print(Name + " OnBarUpdate: " + ex);
            }
        }

        protected override void OnRender(ChartControl chartControl, ChartScale chartScale)
        {
            try
            {
                if (Bars == null || ChartControl == null || Bars.Instrument == null || !IsVisible)
                {
                    return;
                }
                base.OnRender(chartControl, chartScale);

                //Radial Gradient:

                if (!PaintRadialGradient)
                    return;

                int cbti = Calculate == Calculate.OnBarClose ? ChartBars.ToIndex - 1 : ChartBars.ToIndex;
                int cbfi = ChartBars.FromIndex;

                //To prevent unecessary radial gradient calculations when the plots are too small/too many bars
                float barDistance = chartControl.Properties.BarDistance;
                if (barDistance < 5f)
                    return;

                try
                {
                    for (int idx = cbfi; idx <= cbti; idx++)
                    {
                        double macdValue = Values[0].GetValueAt(idx);
                        float macdY = chartScale.GetYByValue(macdValue);

                        float x = chartControl.GetXByBarIndex(ChartBars, idx);

                        Point dotCenterPoint = new Point(x, macdY);
                        Vector2 dotCenterVector2 = dotCenterPoint.ToVector2();

                        float dotRadius = Plots[9].Width + 1f;

                        if (MacdBbDisplayType == MacdBbV4DisplayTypeEnum.Gradient)
                        {
                            var tempDotBrushDx = PlotBrushes[9].Get(idx).ToDxBrush(RenderTarget); //ToDx was causing grief

                            if (tempDotBrushDx == null)
                                return;

                            // disposes automatically after usage
                            using (RadialGradientBrush tempRadialGradientBrushDx = new RadialGradientBrush(RenderTarget,
                                new RadialGradientBrushProperties
                                {
                                    Center = new Vector2((float) dotCenterPoint.X + dotRadius*0.4f, (float) dotCenterPoint.Y - dotRadius*0.4f),
                                    RadiusX = dotRadius,
                                    RadiusY = dotRadius,
                                    GradientOriginOffset = new Vector2(0f, 0f)
                                }
                                ,
                                new GradientStopCollection(RenderTarget, new[]
                                {
                                    new GradientStop
                                    {
                                        Color = Sim22_BrushUtilitiesStat.LerpSharpDxBrushToColor(tempDotBrushDx, true, 0.7f),
                                        Position = 0
                                    },
                                    new GradientStop
                                    {
                                        Color = Sim22_BrushUtilitiesStat.LerpSharpDxBrushToColor(tempDotBrushDx, true, 0.0f),
                                        Position = 0.4f
                                    },
                                    new GradientStop
                                    {
                                        Color = Sim22_BrushUtilitiesStat.LerpSharpDxBrushToColor(tempDotBrushDx, false, 0.65f),
                                        Position = 1
                                    }
                                })))
                            {
                                RenderTarget.AntialiasMode = AntialiasMode.PerPrimitive;
                                Ellipse elip = new Ellipse(dotCenterVector2, dotRadius, dotRadius);

                                RenderTarget.FillEllipse(elip, tempRadialGradientBrushDx);
                                RenderTarget.AntialiasMode = AntialiasMode.Aliased;
                            }
                            
                            tempDotBrushDx.Dispose();
                        }
                        else
                        {
                            int i = 0;
                            ////foreach loop calls once as opposed to 'for' loop.
                            foreach (var brushes in _macdBrushes)
                            {
                                // find the right brush for the bar
                                if (!Equals(PlotBrushes[9].Get(idx), brushes))
                                {
                                    i++;
                                    // skip to the next brush
                                    continue;
                                }

                                var tempDotBrushDx = _macdBrushes[i].ToDxBrush(RenderTarget);

                                if (tempDotBrushDx == null)
                                    return;

                                // disposes automatically after usage
                                using (RadialGradientBrush tempRadialGradientBrushDx = new RadialGradientBrush(RenderTarget,
                                    new RadialGradientBrushProperties
                                    {
                                        Center = new Vector2((float) dotCenterPoint.X + dotRadius*0.3f, (float) dotCenterPoint.Y - dotRadius*0.3f),
                                        RadiusX = dotRadius,
                                        RadiusY = dotRadius,
                                        GradientOriginOffset = new Vector2(0f, 0f)
                                    }
                                    ,
                                    new GradientStopCollection(RenderTarget, new[]
                                    {
                                        new GradientStop
                                        {
                                            Color = Sim22_BrushUtilitiesStat.LerpSharpDxBrushToColor(tempDotBrushDx, true, 0.7f),
                                            Position = 0
                                        },
                                        new GradientStop
                                        {
                                            Color = Sim22_BrushUtilitiesStat.LerpSharpDxBrushToColor(tempDotBrushDx, true, 0.0f),
                                            Position = 0.4f
                                        },
                                        new GradientStop
                                        {
                                            Color = Sim22_BrushUtilitiesStat.LerpSharpDxBrushToColor(tempDotBrushDx, false, 0.65f),
                                            Position = 1
                                        }
                                    })))
                                {
                                    RenderTarget.AntialiasMode = AntialiasMode.PerPrimitive;
                                    Ellipse elip = new Ellipse(dotCenterVector2, dotRadius, dotRadius);

                                    RenderTarget.FillEllipse(elip, tempRadialGradientBrushDx);
                                    RenderTarget.AntialiasMode = AntialiasMode.Aliased;
                                }

                                tempDotBrushDx.Dispose();
                            }
                        }
                    }
                }
                catch (Exception ex)
                {
                    Print(Name + " OnRender: " + ex);
                }
            }
            catch (Exception ex)
            {
                Print(Name + " OnRender: " + ex);
            }
        }

        #region Properties

        [
            Browsable(false)]
        [XmlIgnore]
        public Series<double> MacdGl //of no visual value
        {
            get { return Values[0]; }
        }

        [Browsable(false)]
        [XmlIgnore]
        public Series<double> ZeroLine
        {
            get { return Values[1]; }
        }

        [Browsable(false)]
        [XmlIgnore]
        public Series<double> Histogram
        {
            get { return Values[2]; }
        }

        [Browsable(false)]
        [XmlIgnore]
        public Series<double> UpperBollinger
        {
            get { return Values[3]; }
        }

        [Browsable(false)]
        [XmlIgnore]
        public Series<double> LowerBollinger
        {
            get { return Values[4]; }
        }

        [Browsable(false)]
        [XmlIgnore]
        public Series<double> Average
        {
            get { return Values[5]; }
        }

        [Browsable(false)]
        [XmlIgnore]
        public Series<double> MacdLine
        {
            get { return Values[6]; }
        }

        [Browsable(false)]
        [XmlIgnore]
        public Series<double> LineSignalPlot
        {
            get { return Values[7]; }
        }

        [Browsable(false)]
        [XmlIgnore]
        public Series<double> MacdDotFrame
        {
            get { return Values[8]; }
        }

        [Browsable(false)]
        [XmlIgnore]
        public Series<double> MacdDot
        {
            get { return Values[9]; }
        }

        [Browsable(false)]
        [XmlIgnore]
        public Series<int> SignalIntSeries
        {
            // Use this to get up/down signals for your external indicators.
            get { return _signalIntSeries; }
        }

        [NinjaScriptProperty]
        [TypeConverter(typeof(Sim22EnumUtilities.EnumDescriptionConverter))]
        [PropertyEditor("NinjaTrader.Gui.Tools.StringStandardValuesEditorKey")]
        [Display(Name = "Decimal Places (Price.xx)", Description = "", Order = 0, GroupName = "Set up")]
        public Sim22PriceFormatExtensions.DecimalPlacesEnum FormatNumberDecimalPlaces { get; set; }

        [ReadOnly(true)]
        [NinjaScriptProperty]
        [Display(Name = "Version", GroupName = "1. Parameters", Order = 0)]
        public string VersionStr { get; set; }

        [Range(1, int.MaxValue), NinjaScriptProperty]
        [Display(Name = "Fast period", GroupName = "1. Parameters", Order = 10)]
        public int FastMaPeriod { get; set; }

        [Range(1, int.MaxValue), NinjaScriptProperty]
        [Display(Name = "Slow period", GroupName = "1. Parameters", Order = 11)]
        public int SlowMaPeriod { get; set; }

        [Range(1, int.MaxValue), NinjaScriptProperty]
        [Display(Name = "Average period", GroupName = "1. Parameters", Order = 12)]
        public int AverageBollPeriod { get; set; }

        [NinjaScriptProperty]
        [TypeConverter(typeof(Sim22EnumUtilities.EnumDescriptionConverter))]
        [PropertyEditor("NinjaTrader.Gui.Tools.StringStandardValuesEditorKey")]
        [Display(Name = "MacdBB band deviation type", Description = "Use either standard deviations or MAD.", GroupName = "1. Parameters", Order = 13)]
        public MacdBbv4BandDevTypeEnum MacdBbv4BandDevType { get; set; }

        [Range(0.0, double.MaxValue)]
        [NinjaScriptProperty]
        [Display(Name = "Number of deviations", Description = "Deviations for bollinger bands > 0.00", GroupName = "1. Parameters", Order = 14)]
        public double NumberDeviations { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "Plot gapless?", Description = "Intraday = Ignores session open 'gap'.  EOD = Ignores bar open 'gap'.", GroupName = "1. Parameters", Order = 15)]
        public bool UseGapless { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "Display in ticks?", Description = "Show in tick value to compare with other instruments.", GroupName = "1. Parameters", Order = 16)]
        public bool UseTickSize { get; set; }

        [NinjaScriptProperty]
        [TypeConverter(typeof(Sim22EnumUtilities.EnumDescriptionConverter))]
        [PropertyEditor("NinjaTrader.Gui.Tools.StringStandardValuesEditorKey")]
        [Display(Name = "MacdBB display type", Description = "Set display type.", GroupName = "2. Plots", Order = 0)]
        public MacdBbV4DisplayTypeEnum MacdBbDisplayType { get; set; }

        [NinjaScriptProperty]
        [TypeConverter(typeof(Sim22EnumUtilities.EnumDescriptionConverter))]
        [PropertyEditor("NinjaTrader.Gui.Tools.StringStandardValuesEditorKey")]
        [Display(Name = "Coloring & signal calculation", Description = "Select how you want the coloring/signal to plot on the chart.", GroupName = "2. Plots", Order = 1)]
        public MacdBbv4SignalTypeEnum MacdBbv4SignalType { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "Paint radial gradient?", Description = "Add a bit of bling to your macdbb", GroupName = "2. Plots", Order = 2)]
        public bool PaintRadialGradient { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "Paint bars?", Description = "Colors bars based on the MACD value.", GroupName = "2. Plots", Order = 6)]
        public bool PlotPaintBars { get; set; }

        [Range(0, 100), NinjaScriptProperty]
        [Display(Name = "Bar opacity (%)", Description = "Set between 0-100", GroupName = "2. Plots", Order = 7)]
        public int BarOpacity { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "Paint candle outline?", Description = "Colors candle outline based on MACD value.", GroupName = "2. Plots", Order = 8)]
        public bool PaintOutline { get; set; }

        [Range(0, 200), NinjaScriptProperty]
        [Display(Name = "Outline tone change (%)", Description = "Set (0-100) for darker and (100-200) for lighter.", GroupName = "2. Plots", Order = 9)]
        public int OutlineTonePercent { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "Paint panel background?", Description = "Paints background in panel only.", GroupName = "2. Plots", Order = 10)]
        public bool PaintBackground { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "Paint chart background?", Description = "Paints chart background", GroupName = "2. Plots", Order = 11)]
        public bool PaintBackgroundAll { get; set; }

        [Range(0, 100), NinjaScriptProperty]
        [Display(Name = "Background opacity (%)", Description = "Set between 0-100", GroupName = "2. Plots", Order = 12)]
        public int BackGroundOpacity { get; set; }


        [NinjaScriptProperty]
        [Display(Name = "Show histogram?", Description = "Plot histogram of Macd - Average.", GroupName = "2. Plots", Order = 13)]
        public bool ShowHistogram { get; set; }

        /// Brushes
        [XmlIgnore]
        [Display(Name = "Up brush outside", Description = "Outside upper bollinger bands.", Order = 0, GroupName = "3. Colors")]
        public Brush UpBrushOut
        {
            get { return _macdBrushes[0]; }
            set { _macdBrushes[0] = value; }
        }

        [Browsable(false)]
        public string UpBrushOutSerializable
        {
            get { return Serialize.BrushToString(UpBrushOut); }
            set { UpBrushOut = Serialize.StringToBrush(value); }
        }

        [XmlIgnore]
        [Display(Name = "Up brush inside", Description = "Inside bollinger bands.", Order = 1, GroupName = "3. Colors")]
        public Brush UpBrushIn
        {
            get { return _macdBrushes[1]; }
            set { _macdBrushes[1] = value; }
        }

        [Browsable(false)]
        public string UpBrushInSerializable
        {
            get { return Serialize.BrushToString(UpBrushIn); }
            set { UpBrushIn = Serialize.StringToBrush(value); }
        }

        [XmlIgnore]
        [Display(Name = "Down brush outside", Description = "Outside bollinger bands.", Order = 2, GroupName = "3. Colors")]
        public Brush DnBrushOut
        {
            get { return _macdBrushes[2]; }
            set { _macdBrushes[2] = value; }
        }

        [Browsable(false)]
        public string DnBrushOutSerializable
        {
            get { return Serialize.BrushToString(DnBrushOut); }
            set { DnBrushOut = Serialize.StringToBrush(value); }
        }

        [XmlIgnore]
        [Display(Name = "Down brush inside", Description = "Inside bollinger bands.", Order = 3, GroupName = "3. Colors")]
        public Brush DnBrushIn
        {
            get { return _macdBrushes[3]; }
            set { _macdBrushes[3] = value; }
        }

        [Browsable(false)]
        public string DnBrushInSerializable
        {
            get { return Serialize.BrushToString(DnBrushIn); }
            set { DnBrushIn = Serialize.StringToBrush(value); }
        }

        //Gradient brushes 
        [XmlIgnore]
        [Display(Name = "Up brush gradient", Description = "Up gradient color", Order = 4, GroupName = "4. Gradient Colors")]
        public Brush UpBrushGrad { get; set; }

        [Browsable(false)]
        public string UpBrushGradSerializable
        {
            get { return Serialize.BrushToString(UpBrushGrad); }
            set { UpBrushGrad = Serialize.StringToBrush(value); }
        }

        [XmlIgnore]
        [Display(Name = "Down brush gradient", Description = "Down gradient color", Order = 5, GroupName = "4. Gradient Colors")]
        public Brush DnBrushGrad { get; set; }

        [Browsable(false)]
        public string DnBrushGradSerializable
        {
            get { return Serialize.BrushToString(DnBrushGrad); }
            set { DnBrushGrad = Serialize.StringToBrush(value); }
        }

        [Range(2, int.MaxValue), NinjaScriptProperty]
        [Display(Description = "Select larger number of bars (Default = 252)", Name = "Slow period", GroupName = "5. Gradient Parameters (AggM)", Order = 1)]
        public int SlowPeriodAggM { get; set; }

        [Range(2, int.MaxValue), NinjaScriptProperty]
        [Display(Description = "Select smaller number of bars (Default = 10)", Name = "Fast period", GroupName = "5. Gradient Parameters (AggM)", Order = 2)]
        public int FastPeriodAggM { get; set; }

        [Range(0, 100), NinjaScriptProperty]
        [Display(Description = "Smoothing between current bar and last bar", Name = "Current bar weighting % (0-100)", GroupName = "5. Gradient Parameters (AggM)", Order = 3)]
        public int CurrentBarWeightingAggM { get; set; }

        [Range(0, 100), NinjaScriptProperty]
        [Display(Description = "New: Increase for more trend based smoothing (0-100)", Name = "New: Trend weighting % (0-100)", GroupName = "5. Gradient Parameters (AggM)", Order = 4)]
        public int TrendWeightingAggM { get; set; }

        [XmlIgnore]
        [Display(Name = "Band brush", Description = "Band color", Order = 0, GroupName = "6. Bands")]
        public Brush BandBrush { get; set; }

        [Browsable(false)]
        public string BandBrushInSerializable
        {
            get { return Serialize.BrushToString(BandBrush); }
            set { BandBrush = Serialize.StringToBrush(value); }
        }

        [Range(0, 100), NinjaScriptProperty]
        [Display(Name = "Band opacity", Description = "Set between 0-100", GroupName = "6. Bands", Order = 1)]
        public int BandOpacity { get; set; }

        //Alerts

        [NinjaScriptProperty]
        [Display(Name = "Cross alert", Description = "Play a sound when the signal is given.", GroupName = "7. Alerts", Order = 0)]
        public bool PlayCrossSound { get; set; }

        [Display(Name = "Long sound", Description = @"Set the sound to be played when a long signal occurs. Place any new sounds in your NT8\sounds directory.", GroupName = "7. Alerts", Order = 1)]
        [PropertyEditor("NinjaTrader.Gui.Tools.FilePathPicker", Filter = "Wav Files (*.wav)|*.wav")]
        public string LongWavFileNameStr { get; set; }

        [Display(Name = "Short sound", Description = @"Set the sound to be played when a short signal occurs. Place any new sounds in your NT8\sounds directory.", GroupName = "7. Alerts", Order = 2)]
        [PropertyEditor("NinjaTrader.Gui.Tools.FilePathPicker", Filter = "Wav Files (*.wav)|*.wav")]
        public string ShortWavFileNameStr { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "Draw entry signals?", Description = "Draw an entry signal.", GroupName = "8. Signals", Order = 1)]
        public bool DrawEntrySignals { get; set; }

        [NinjaScriptProperty]
        [Display(Name = "Draw exit signals?", Description = "Draw an exit signal.", GroupName = "8. Signals", Order = 2)]
        public bool DrawExitSignals { get; set; }

        #endregion
    }
}

#region NinjaScript generated code. Neither change nor remove.

namespace NinjaTrader.NinjaScript.Indicators
{
	public partial class Indicator : NinjaTrader.Gui.NinjaScript.IndicatorRenderBase
	{
		private Sim22.Sim22_MacdBBGaplessV4[] cacheSim22_MacdBBGaplessV4;
		public Sim22.Sim22_MacdBBGaplessV4 Sim22_MacdBBGaplessV4(Sim22PriceFormatExtensions.DecimalPlacesEnum formatNumberDecimalPlaces, string versionStr, int fastMaPeriod, int slowMaPeriod, int averageBollPeriod, MacdBbv4BandDevTypeEnum macdBbv4BandDevType, double numberDeviations, bool useGapless, bool useTickSize, MacdBbV4DisplayTypeEnum macdBbDisplayType, MacdBbv4SignalTypeEnum macdBbv4SignalType, bool paintRadialGradient, bool plotPaintBars, int barOpacity, bool paintOutline, int outlineTonePercent, bool paintBackground, bool paintBackgroundAll, int backGroundOpacity, bool showHistogram, int slowPeriodAggM, int fastPeriodAggM, int currentBarWeightingAggM, int trendWeightingAggM, int bandOpacity, bool playCrossSound, bool drawEntrySignals, bool drawExitSignals)
		{
			return Sim22_MacdBBGaplessV4(Input, formatNumberDecimalPlaces, versionStr, fastMaPeriod, slowMaPeriod, averageBollPeriod, macdBbv4BandDevType, numberDeviations, useGapless, useTickSize, macdBbDisplayType, macdBbv4SignalType, paintRadialGradient, plotPaintBars, barOpacity, paintOutline, outlineTonePercent, paintBackground, paintBackgroundAll, backGroundOpacity, showHistogram, slowPeriodAggM, fastPeriodAggM, currentBarWeightingAggM, trendWeightingAggM, bandOpacity, playCrossSound, drawEntrySignals, drawExitSignals);
		}

		public Sim22.Sim22_MacdBBGaplessV4 Sim22_MacdBBGaplessV4(ISeries<double> input, Sim22PriceFormatExtensions.DecimalPlacesEnum formatNumberDecimalPlaces, string versionStr, int fastMaPeriod, int slowMaPeriod, int averageBollPeriod, MacdBbv4BandDevTypeEnum macdBbv4BandDevType, double numberDeviations, bool useGapless, bool useTickSize, MacdBbV4DisplayTypeEnum macdBbDisplayType, MacdBbv4SignalTypeEnum macdBbv4SignalType, bool paintRadialGradient, bool plotPaintBars, int barOpacity, bool paintOutline, int outlineTonePercent, bool paintBackground, bool paintBackgroundAll, int backGroundOpacity, bool showHistogram, int slowPeriodAggM, int fastPeriodAggM, int currentBarWeightingAggM, int trendWeightingAggM, int bandOpacity, bool playCrossSound, bool drawEntrySignals, bool drawExitSignals)
		{
			if (cacheSim22_MacdBBGaplessV4 != null)
				for (int idx = 0; idx < cacheSim22_MacdBBGaplessV4.Length; idx++)
					if (cacheSim22_MacdBBGaplessV4[idx] != null && cacheSim22_MacdBBGaplessV4[idx].FormatNumberDecimalPlaces == formatNumberDecimalPlaces && cacheSim22_MacdBBGaplessV4[idx].VersionStr == versionStr && cacheSim22_MacdBBGaplessV4[idx].FastMaPeriod == fastMaPeriod && cacheSim22_MacdBBGaplessV4[idx].SlowMaPeriod == slowMaPeriod && cacheSim22_MacdBBGaplessV4[idx].AverageBollPeriod == averageBollPeriod && cacheSim22_MacdBBGaplessV4[idx].MacdBbv4BandDevType == macdBbv4BandDevType && cacheSim22_MacdBBGaplessV4[idx].NumberDeviations == numberDeviations && cacheSim22_MacdBBGaplessV4[idx].UseGapless == useGapless && cacheSim22_MacdBBGaplessV4[idx].UseTickSize == useTickSize && cacheSim22_MacdBBGaplessV4[idx].MacdBbDisplayType == macdBbDisplayType && cacheSim22_MacdBBGaplessV4[idx].MacdBbv4SignalType == macdBbv4SignalType && cacheSim22_MacdBBGaplessV4[idx].PaintRadialGradient == paintRadialGradient && cacheSim22_MacdBBGaplessV4[idx].PlotPaintBars == plotPaintBars && cacheSim22_MacdBBGaplessV4[idx].BarOpacity == barOpacity && cacheSim22_MacdBBGaplessV4[idx].PaintOutline == paintOutline && cacheSim22_MacdBBGaplessV4[idx].OutlineTonePercent == outlineTonePercent && cacheSim22_MacdBBGaplessV4[idx].PaintBackground == paintBackground && cacheSim22_MacdBBGaplessV4[idx].PaintBackgroundAll == paintBackgroundAll && cacheSim22_MacdBBGaplessV4[idx].BackGroundOpacity == backGroundOpacity && cacheSim22_MacdBBGaplessV4[idx].ShowHistogram == showHistogram && cacheSim22_MacdBBGaplessV4[idx].SlowPeriodAggM == slowPeriodAggM && cacheSim22_MacdBBGaplessV4[idx].FastPeriodAggM == fastPeriodAggM && cacheSim22_MacdBBGaplessV4[idx].CurrentBarWeightingAggM == currentBarWeightingAggM && cacheSim22_MacdBBGaplessV4[idx].TrendWeightingAggM == trendWeightingAggM && cacheSim22_MacdBBGaplessV4[idx].BandOpacity == bandOpacity && cacheSim22_MacdBBGaplessV4[idx].PlayCrossSound == playCrossSound && cacheSim22_MacdBBGaplessV4[idx].DrawEntrySignals == drawEntrySignals && cacheSim22_MacdBBGaplessV4[idx].DrawExitSignals == drawExitSignals && cacheSim22_MacdBBGaplessV4[idx].EqualsInput(input))
						return cacheSim22_MacdBBGaplessV4[idx];
			return CacheIndicator<Sim22.Sim22_MacdBBGaplessV4>(new Sim22.Sim22_MacdBBGaplessV4(){ FormatNumberDecimalPlaces = formatNumberDecimalPlaces, VersionStr = versionStr, FastMaPeriod = fastMaPeriod, SlowMaPeriod = slowMaPeriod, AverageBollPeriod = averageBollPeriod, MacdBbv4BandDevType = macdBbv4BandDevType, NumberDeviations = numberDeviations, UseGapless = useGapless, UseTickSize = useTickSize, MacdBbDisplayType = macdBbDisplayType, MacdBbv4SignalType = macdBbv4SignalType, PaintRadialGradient = paintRadialGradient, PlotPaintBars = plotPaintBars, BarOpacity = barOpacity, PaintOutline = paintOutline, OutlineTonePercent = outlineTonePercent, PaintBackground = paintBackground, PaintBackgroundAll = paintBackgroundAll, BackGroundOpacity = backGroundOpacity, ShowHistogram = showHistogram, SlowPeriodAggM = slowPeriodAggM, FastPeriodAggM = fastPeriodAggM, CurrentBarWeightingAggM = currentBarWeightingAggM, TrendWeightingAggM = trendWeightingAggM, BandOpacity = bandOpacity, PlayCrossSound = playCrossSound, DrawEntrySignals = drawEntrySignals, DrawExitSignals = drawExitSignals }, input, ref cacheSim22_MacdBBGaplessV4);
		}
	}
}

namespace NinjaTrader.NinjaScript.MarketAnalyzerColumns
{
	public partial class MarketAnalyzerColumn : MarketAnalyzerColumnBase
	{
		public Indicators.Sim22.Sim22_MacdBBGaplessV4 Sim22_MacdBBGaplessV4(Sim22PriceFormatExtensions.DecimalPlacesEnum formatNumberDecimalPlaces, string versionStr, int fastMaPeriod, int slowMaPeriod, int averageBollPeriod, MacdBbv4BandDevTypeEnum macdBbv4BandDevType, double numberDeviations, bool useGapless, bool useTickSize, MacdBbV4DisplayTypeEnum macdBbDisplayType, MacdBbv4SignalTypeEnum macdBbv4SignalType, bool paintRadialGradient, bool plotPaintBars, int barOpacity, bool paintOutline, int outlineTonePercent, bool paintBackground, bool paintBackgroundAll, int backGroundOpacity, bool showHistogram, int slowPeriodAggM, int fastPeriodAggM, int currentBarWeightingAggM, int trendWeightingAggM, int bandOpacity, bool playCrossSound, bool drawEntrySignals, bool drawExitSignals)
		{
			return indicator.Sim22_MacdBBGaplessV4(Input, formatNumberDecimalPlaces, versionStr, fastMaPeriod, slowMaPeriod, averageBollPeriod, macdBbv4BandDevType, numberDeviations, useGapless, useTickSize, macdBbDisplayType, macdBbv4SignalType, paintRadialGradient, plotPaintBars, barOpacity, paintOutline, outlineTonePercent, paintBackground, paintBackgroundAll, backGroundOpacity, showHistogram, slowPeriodAggM, fastPeriodAggM, currentBarWeightingAggM, trendWeightingAggM, bandOpacity, playCrossSound, drawEntrySignals, drawExitSignals);
		}

		public Indicators.Sim22.Sim22_MacdBBGaplessV4 Sim22_MacdBBGaplessV4(ISeries<double> input , Sim22PriceFormatExtensions.DecimalPlacesEnum formatNumberDecimalPlaces, string versionStr, int fastMaPeriod, int slowMaPeriod, int averageBollPeriod, MacdBbv4BandDevTypeEnum macdBbv4BandDevType, double numberDeviations, bool useGapless, bool useTickSize, MacdBbV4DisplayTypeEnum macdBbDisplayType, MacdBbv4SignalTypeEnum macdBbv4SignalType, bool paintRadialGradient, bool plotPaintBars, int barOpacity, bool paintOutline, int outlineTonePercent, bool paintBackground, bool paintBackgroundAll, int backGroundOpacity, bool showHistogram, int slowPeriodAggM, int fastPeriodAggM, int currentBarWeightingAggM, int trendWeightingAggM, int bandOpacity, bool playCrossSound, bool drawEntrySignals, bool drawExitSignals)
		{
			return indicator.Sim22_MacdBBGaplessV4(input, formatNumberDecimalPlaces, versionStr, fastMaPeriod, slowMaPeriod, averageBollPeriod, macdBbv4BandDevType, numberDeviations, useGapless, useTickSize, macdBbDisplayType, macdBbv4SignalType, paintRadialGradient, plotPaintBars, barOpacity, paintOutline, outlineTonePercent, paintBackground, paintBackgroundAll, backGroundOpacity, showHistogram, slowPeriodAggM, fastPeriodAggM, currentBarWeightingAggM, trendWeightingAggM, bandOpacity, playCrossSound, drawEntrySignals, drawExitSignals);
		}
	}
}

namespace NinjaTrader.NinjaScript.Strategies
{
	public partial class Strategy : NinjaTrader.Gui.NinjaScript.StrategyRenderBase
	{
		public Indicators.Sim22.Sim22_MacdBBGaplessV4 Sim22_MacdBBGaplessV4(Sim22PriceFormatExtensions.DecimalPlacesEnum formatNumberDecimalPlaces, string versionStr, int fastMaPeriod, int slowMaPeriod, int averageBollPeriod, MacdBbv4BandDevTypeEnum macdBbv4BandDevType, double numberDeviations, bool useGapless, bool useTickSize, MacdBbV4DisplayTypeEnum macdBbDisplayType, MacdBbv4SignalTypeEnum macdBbv4SignalType, bool paintRadialGradient, bool plotPaintBars, int barOpacity, bool paintOutline, int outlineTonePercent, bool paintBackground, bool paintBackgroundAll, int backGroundOpacity, bool showHistogram, int slowPeriodAggM, int fastPeriodAggM, int currentBarWeightingAggM, int trendWeightingAggM, int bandOpacity, bool playCrossSound, bool drawEntrySignals, bool drawExitSignals)
		{
			return indicator.Sim22_MacdBBGaplessV4(Input, formatNumberDecimalPlaces, versionStr, fastMaPeriod, slowMaPeriod, averageBollPeriod, macdBbv4BandDevType, numberDeviations, useGapless, useTickSize, macdBbDisplayType, macdBbv4SignalType, paintRadialGradient, plotPaintBars, barOpacity, paintOutline, outlineTonePercent, paintBackground, paintBackgroundAll, backGroundOpacity, showHistogram, slowPeriodAggM, fastPeriodAggM, currentBarWeightingAggM, trendWeightingAggM, bandOpacity, playCrossSound, drawEntrySignals, drawExitSignals);
		}

		public Indicators.Sim22.Sim22_MacdBBGaplessV4 Sim22_MacdBBGaplessV4(ISeries<double> input , Sim22PriceFormatExtensions.DecimalPlacesEnum formatNumberDecimalPlaces, string versionStr, int fastMaPeriod, int slowMaPeriod, int averageBollPeriod, MacdBbv4BandDevTypeEnum macdBbv4BandDevType, double numberDeviations, bool useGapless, bool useTickSize, MacdBbV4DisplayTypeEnum macdBbDisplayType, MacdBbv4SignalTypeEnum macdBbv4SignalType, bool paintRadialGradient, bool plotPaintBars, int barOpacity, bool paintOutline, int outlineTonePercent, bool paintBackground, bool paintBackgroundAll, int backGroundOpacity, bool showHistogram, int slowPeriodAggM, int fastPeriodAggM, int currentBarWeightingAggM, int trendWeightingAggM, int bandOpacity, bool playCrossSound, bool drawEntrySignals, bool drawExitSignals)
		{
			return indicator.Sim22_MacdBBGaplessV4(input, formatNumberDecimalPlaces, versionStr, fastMaPeriod, slowMaPeriod, averageBollPeriod, macdBbv4BandDevType, numberDeviations, useGapless, useTickSize, macdBbDisplayType, macdBbv4SignalType, paintRadialGradient, plotPaintBars, barOpacity, paintOutline, outlineTonePercent, paintBackground, paintBackgroundAll, backGroundOpacity, showHistogram, slowPeriodAggM, fastPeriodAggM, currentBarWeightingAggM, trendWeightingAggM, bandOpacity, playCrossSound, drawEntrySignals, drawExitSignals);
		}
	}
}

#endregion
