// 
// Copyright (C) 2016, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//
#region Using declarations
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Input;
using System.Windows.Media;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Gui;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Gui.SuperDom;
using NinjaTrader.Data;
using NinjaTrader.NinjaScript;
using NinjaTrader.Core.FloatingPoint;
using NinjaTrader.NinjaScript.DrawingTools;
#endregion

// This namespace holds indicators in this folder and is required. Do not change it.
namespace NinjaTrader.NinjaScript.Indicators.Sim22
{
    /// <summary>
    /// Mean Absolute Deviation is a statistical measure of volatility. 
    /// Mean Absolute Deviation is typically used as a component of other indicators, 
    /// rather than as a stand-alone indicator.
    /// Idea from Lawrence Chan here: http://www.daytradingbias.com
    /// Sim22 NT8.0.2 Jan 2017
    /// </summary>
    public class Sim22_MeanAbsDev : Indicator
	{
        private SMA sma;

		protected override void OnStateChange()
		{
			if (State == State.SetDefaults)
			{
				Description					= "Mean Absolute Deviation is a statistical measure of volatility, similar to standard deviation. Sim22 Jan 2017 NT8.0.2";
				Name						= "Sim22_MeanAbsDev";
				IsOverlay					= false;
				IsSuspendedWhileInactive	= true;
				Period						= 14;

				AddPlot(Brushes.Green, "Mad");
			}
			else if (State == State.Configure)
			{
			    sma = SMA(Input, Period);
			}
		}

		protected override void OnBarUpdate()
		{
			if (CurrentBar < 1)
			{
				Value[0] = 0;
			}
			else
			{
                double sum = 0;

                for (int barsBack = Math.Min(CurrentBar, Period - 1); barsBack >= 0; barsBack--)
                    sum += Math.Abs(Input[barsBack] - sma[0]);

                Value[0] = sum / Math.Min(CurrentBar + 1, Period);
            }
        }

		#region Properties
		[Range(1, int.MaxValue), NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Period", GroupName = "NinjaScriptParameters", Order = 0)]
		public int Period
		{ get; set; }
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.

namespace NinjaTrader.NinjaScript.Indicators
{
	public partial class Indicator : NinjaTrader.Gui.NinjaScript.IndicatorRenderBase
	{
		private Sim22.Sim22_MeanAbsDev[] cacheSim22_MeanAbsDev;
		public Sim22.Sim22_MeanAbsDev Sim22_MeanAbsDev(int period)
		{
			return Sim22_MeanAbsDev(Input, period);
		}

		public Sim22.Sim22_MeanAbsDev Sim22_MeanAbsDev(ISeries<double> input, int period)
		{
			if (cacheSim22_MeanAbsDev != null)
				for (int idx = 0; idx < cacheSim22_MeanAbsDev.Length; idx++)
					if (cacheSim22_MeanAbsDev[idx] != null && cacheSim22_MeanAbsDev[idx].Period == period && cacheSim22_MeanAbsDev[idx].EqualsInput(input))
						return cacheSim22_MeanAbsDev[idx];
			return CacheIndicator<Sim22.Sim22_MeanAbsDev>(new Sim22.Sim22_MeanAbsDev(){ Period = period }, input, ref cacheSim22_MeanAbsDev);
		}
	}
}

namespace NinjaTrader.NinjaScript.MarketAnalyzerColumns
{
	public partial class MarketAnalyzerColumn : MarketAnalyzerColumnBase
	{
		public Indicators.Sim22.Sim22_MeanAbsDev Sim22_MeanAbsDev(int period)
		{
			return indicator.Sim22_MeanAbsDev(Input, period);
		}

		public Indicators.Sim22.Sim22_MeanAbsDev Sim22_MeanAbsDev(ISeries<double> input , int period)
		{
			return indicator.Sim22_MeanAbsDev(input, period);
		}
	}
}

namespace NinjaTrader.NinjaScript.Strategies
{
	public partial class Strategy : NinjaTrader.Gui.NinjaScript.StrategyRenderBase
	{
		public Indicators.Sim22.Sim22_MeanAbsDev Sim22_MeanAbsDev(int period)
		{
			return indicator.Sim22_MeanAbsDev(Input, period);
		}

		public Indicators.Sim22.Sim22_MeanAbsDev Sim22_MeanAbsDev(ISeries<double> input , int period)
		{
			return indicator.Sim22_MeanAbsDev(input, period);
		}
	}
}

#endregion
