#region Using declarations
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Input;
using System.Windows.Media;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Gui;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Gui.SuperDom;
using NinjaTrader.Data;
using NinjaTrader.NinjaScript;
using NinjaTrader.Core.FloatingPoint;
using NinjaTrader.NinjaScript.DrawingTools;
#endregion

//This namespace holds Indicators in this folder and is required. Do not change it. 
namespace NinjaTrader.NinjaScript.Indicators
{
	
/*
Converted to Ninjatrader 8 beta release 3,  7/30/2015 - Ninjascript team
	
*original comments from Nt7 version follow*	
	
Modified: Oct 14 2010 
Modification: added logic to reset the Plots when the CalculateOnBarClose=false;
              previously the bars would not paint correctly.
              enjoy


Barry Taylor's Better Volume Indicator as described at
http://emini-watch.com/free-stuff/volume-indicator/
Coded for NinjaTrader by Alex Matulich, 2009-10-08, Unicorn Research Corporation.
This indicator optionally uses True Range rather than Range, it plots an exponential
moving average of the volume instead of simple moving average for efficiency,
and it optionally paints the price bars when BetterVolume2 indicates something
other than the normal color.

Watch for these at different stages of the market:

                  | Bottom | Start |Continue|  Top   |  Start  |Continue
                  |        |uptrend|uptrend |        |downtrend|downtrend
------------------+--------+-------+--------+--------+---------+---------
Volume Climax Up  |        |  Yes  |        |  Yes   |         |   Yes     red         (NT8 - Crimson)
Volume Climax Down|  Yes   |       |  Yes   |        |  Yes    |           white/black (NT8- DimGray)
High Volume Churn |  Yes   |       |        |  Yes   |         |           green/blue  (NT8- DarkGreen)
Climax + Hi Churn |  Yes   |       |        |  Yes   |         |           magenta     (NT8 - Magenta)
Low Volume        |  Yes   |       |  Yes   |  Yes   |         |   Yes     yellow      (NT8 - Goldenrod)
-------------------------------------------------------------------------

This indicator works the same on daily or intraday bars. The original Tradestaion
code makes use of up volume and down volume data for intraday bars, and for daily
bars it calculates up volume and down volume from the relative positions of open,
high, low, and close. We use the latter method here, because NinjaTrader does not
provide up volume and down volume data.
*/	
	
	public class BetterVolume3 : Indicator
	{
		private int barwidth		= 	2;
		private double vwt;
		private Series <double> [] value = new Series <double>[23];
		private bool eod 			=	true;
		
		protected override void OnStateChange()
		{
			if (State == State.SetDefaults)
			{
				Description					= @"Better Volume indicator from emini-watch.com, using true range instead of range in calculations.";
				Name						= "BetterVolume3";
				Calculate					= Calculate.OnEachTick;
				IsOverlay					= false;
				DisplayInDataBox			= true;
				DrawOnPricePanel			= true;
				DrawHorizontalGridLines		= false;
				DrawVerticalGridLines		= false;
				PaintPriceMarkers			= true;
				ScaleJustification			= NinjaTrader.Gui.Chart.ScaleJustification.Right;
				IsSuspendedWhileInactive	= false;
				
				LookBack					= 20;
				PaintBars					= true;
				Use2Bars					= true;
				UseTrueRange				= true;	
				
				AddPlot(new Stroke(Brushes.Silver, 3), PlotStyle.Bar, "VBar");	
				AddPlot(new Stroke(Brushes.Goldenrod, 3), PlotStyle.Bar, "LowVol");	
				AddPlot(new Stroke(Brushes.Crimson, 3), PlotStyle.Bar, "ClimaxUp");	
				AddPlot(new Stroke(Brushes.Black, 3), PlotStyle.Bar, "ClimaxDown");	
				AddPlot(new Stroke(Brushes.DarkGreen, 3), PlotStyle.Bar, "HiChurn");				
				AddPlot(new Stroke(Brushes.Fuchsia, 3), PlotStyle.Bar, "ClimaxChurn");
				AddPlot(new Stroke(Brushes.CornflowerBlue, 2), PlotStyle.Line, "VolEMA");
			}
			else if (State == State.Configure)
			{
				for (int i = 1; i <= 22; ++i) 
				{
					value[i] = new Series <double> (this, MaximumBarsLookBack.TwoHundredFiftySix);  // value1 to value22
				}
				
				vwt = 2.0 / (5.0 * LookBack + 1.0); // bar average period is 5*LookBack						
			}
		}

		protected override void OnBarUpdate()
		{
		
			
			double range, netvol;
			bool cond1, cond2, cond3, cond4, cond5, cond6, cond7, cond8, cond9, cond10,
				cond11=false, cond12=false, cond13=false, cond14=false, cond15=false, 
				cond16=false, cond17=false, cond18=false, cond19=false, cond20=false,
				hiclose, loclose;
			
			// reset the current bars plots at the top of OnBarUpdate
			// NT7 will otherwise give a multi-colored bars since the
			//  values of the plots will be different depending on the volume 
			VBar.Reset();   		// vbar 0 			Oct 14 2010 
			LowVol.Reset();			// lowvolume 1		Oct 14 2010 
			ClimaxUp.Reset();		// climaxup 2		Oct 14 2010 
			ClimaxDown.Reset();		// climaxdown 3		Oct 14 2010 
			HiChurn.Reset();		// hi churn 4		Oct 14 2010 
			ClimaxChurn.Reset();	// climaxchurn 5	Oct 14 2010 
			
			
			if (CurrentBar <= 1) 
			{
				range = High[0] - Low[0];
				VolEMA[0] = Volume[0];
				
				eod = (BarsPeriod.BarsPeriodType == BarsPeriodType.Day 
					|| BarsPeriod.BarsPeriodType == BarsPeriodType.Week 
					|| BarsPeriod.BarsPeriodType == BarsPeriodType.Month 
					|| BarsPeriod.BarsPeriodType == BarsPeriodType.Year);
				
				return;
			}
					
			// calculate up volume and down volume

			range = UseTrueRange ?
				Math.Max(High[0], Close[1]) - Math.Min(Low[0], Close[1]) // use true range
				: High[0] - Low[0]; // use normal range	
			
			if (range == 0.0)
			{
				value[1][0] = value[2][0] = 0.5 * Volume[0];
			}
			else 
			{
				if (Close[0] > Open[0])
				{
					value[1][0] = range / (2.0 * range + Open[0] - Close[0]) * Volume[0];

				}
				else if (Close[0] < Open[0])
				{
					value[1][0] = (range + Close[0] - Open[0]) / (2.0 * range + Close[0] - Open[0]) * Volume[0];
				}
				else 
				{
					value[1][0] = 0.5 * Volume[0];
				}
				
				value[2][0] = Volume[0] - value[1][0];
			}
			
			netvol = value[1][0] - value[2][0];
			value[3][0] = Volume[0]; //Math.Abs(value[1][0]+value[2][0]);
			value[4][0] = value[1][0] * range;
			value[5][0] = netvol * range;
			value[6][0] = value[2][0] * range;
			value[7][0] = -netvol * range;
			
			if (range != 0.0) 
			{
				value[8][0] = value[1][0] / range;
				value[9][0] = netvol / range;
				value[10][0] = value[2][0] / range;
				value[11][0] = -netvol / range;
				value[12][0] = value[3][0]  /range;
			}
			
			if (Use2Bars) 
			{
				double hh = MAX(High, 2)[0], ll = MIN(Low, 2)[0];
				range = UseTrueRange ? Math.Max(hh, Close[2]) - Math.Min(ll, Close[2]) // use true range
					: hh - ll; // use normal range
				
				value[13][0] = value[3][0] + value[3][1];
				value[14][0] = (value[1][0] + value[1][1]) * range;
				netvol = value[1][0] + value[1][1] - value[2][0] - value[2][1];
				value[15][0] = netvol * range;
				value[16][0] = (value[2][0] + value[2][1]) * range;
				value[17][0] = -netvol * range;
				
				if (range != 0.0) 
					{
						value[18][0] = (value[1][0] + value[1][1]) / range;
						value[19][0] = netvol/range;
						value[20][0] = (value[2][0] + value[2][1]) / range;
						value[21][0] = -netvol / range;
						value[22][0] = value[13][0] / range;
					} 
				else 
					{ // set to previous values
						for (int i = 18; i <= 22; ++i) value[i][0] = value[i][1];
					}
			}

			hiclose = (Close[0] > Open[0]);
			loclose = (Close[0] < Open[0]);
			cond1 = (value[3][0] == MIN(value[3], LookBack)[0]);
			cond2 = (value[4][0] == MAX(value[4], LookBack)[0] && hiclose);
			cond3 = (value[5][0] == MAX(value[5], LookBack)[0] && hiclose);
			cond4 = (value[6][0] == MAX(value[6], LookBack)[0] && loclose);
			cond5 = (value[7][0] == MAX(value[7], LookBack)[0] && loclose);
			cond6 = (value[8][0] == MIN(value[8], LookBack)[0] && loclose);
			cond7 = (value[9][0] == MIN(value[9], LookBack)[0] && loclose);
			cond8 = (value[10][0] == MIN(value[10], LookBack)[0] && hiclose);
			cond9 = (value[11][0] == MIN(value[11], LookBack)[0] && hiclose);
			cond10 = (value[12][0] == MAX(value[12], LookBack)[0]);
			
			if (Use2Bars) 
			{
				hiclose = (Close[0] > Open[0] && Close[1] > Open[1]);
				loclose = (Close[0] < Open[0] && Close[1] < Open[1]);
				cond11 = (value[13][0] == MIN(value[13], LookBack)[0]);
				cond12 = (value[14][0] == MAX(value[14], LookBack)[0] && hiclose);
				cond13 = (value[15][0] == MAX(value[15], LookBack)[0] && hiclose);
				cond14 = (value[16][0] == MAX(value[16], LookBack)[0] && loclose);
				cond15 = (value[17][0] == MAX(value[17], LookBack)[0] && loclose);
				cond16 = (value[18][0] == MIN(value[18], LookBack)[0] && loclose);
				cond17 = (value[19][0] == MIN(value[19], LookBack)[0] && loclose);
				cond18 = (value[20][0] == MIN(value[20], LookBack)[0] && hiclose);
				cond19 = (value[21][0] == MIN(value[21], LookBack)[0] && hiclose);
				cond20 = (value[22][0] == MAX(value[22], LookBack)[0]);
			}

			// bc = bar color = index number of plot created in Initialize()

			bool sameday = eod ? true : (Time[0].Day == Time[1].Day);
			
			int bc = 0; // default 0 (normal volume bar)
			
			if (cond1 || (cond11 && sameday)) 
			{
				bc = 1; //LowVol
			}
			
			if (cond2 || cond3 || cond8 || cond9 || ((cond12 || cond13 || cond18 || cond19) && sameday)) 
			{
				bc = 2; //ClimaxUp
			}
			
			if (cond4 || cond5 || cond6 || cond7 || ((cond14 || cond15 || cond16 || cond17) && sameday)) 
			{
				bc = 3; //ClimaxDown
			}
			
			if (cond10 || (cond20 && sameday)) 
			{
				bc = 4; //Churn
			}
			
			if (bc == 4 && (cond2 || cond3 || cond4 || cond5 || cond6 || cond7 || cond8 || cond9 || (cond12 || cond13 
				|| cond14 || cond15 || cond16 || cond17 || cond18 || cond19) && sameday))
			{
				bc = 5; //ClimaxChurn
			}

			// plot stuff
			Values[bc][0] = Volume[0]; //draw the volume bar we found
			
			if (PaintBars && bc > 0) 
			{
				BarBrush = Plots[bc].Brush; // paint the price bar
			}
			
			VolEMA[0] = (vwt * (Volume[0] - VolEMA[1]) + VolEMA[1]); // plot average; faster than SMA
					
		} // End OnBarUpdate()
		
		
		#region Properties
		[Range(1, int.MaxValue)]
		[NinjaScriptProperty]
		[Display(Name="LookBack", Description="Number of bars in LookBack period", Order=1, GroupName="Parameters")]
		public int LookBack
		{ get; set; }

		[NinjaScriptProperty]
		[Display(Name="PaintBars", Description="Paint the price bars?", Order=2, GroupName="Parameters")]
		public bool PaintBars
		{ get; set; }

		[NinjaScriptProperty]
		[Display(Name="Use2Bars", Description="Use last 2 bars for calculations? False = currentbar only", Order=3, GroupName="Parameters")]
		public bool Use2Bars
		{ get; set; }

		[NinjaScriptProperty]
		[Display(Name="UseTrueRange", Description="Use true range in calculations, false = use High-low range", Order=4, GroupName="Parameters")]
		public bool UseTrueRange
		{ get; set; }

		[Browsable(false)]
		[XmlIgnore]
		public Series<double> VBar
		{
			get { return Values[0]; }
		}

		[Browsable(false)]
		[XmlIgnore]
		public Series<double> LowVol
		{
			get { return Values[1]; }
		}

		[Browsable(false)]
		[XmlIgnore]
		public Series<double> ClimaxUp
		{
			get { return Values[2]; }
		}

		[Browsable(false)]
		[XmlIgnore]
		public Series<double> ClimaxDown
		{
			get { return Values[3]; }
		}

		[Browsable(false)]
		[XmlIgnore]
		public Series<double> HiChurn
		{
			get { return Values[4]; }
		}

		[Browsable(false)]
		[XmlIgnore]
		public Series<double> ClimaxChurn
		{
			get { return Values[5]; }
		}

		[Browsable(false)]
		[XmlIgnore]
		public Series<double> VolEMA
		{
			get { return Values[6]; }
		}
		#endregion

	}
}

#region NinjaScript generated code. Neither change nor remove.

namespace NinjaTrader.NinjaScript.Indicators
{
	public partial class Indicator : NinjaTrader.Gui.NinjaScript.IndicatorRenderBase
	{
		private BetterVolume3[] cacheBetterVolume3;
		public BetterVolume3 BetterVolume3(int lookBack, bool paintBars, bool use2Bars, bool useTrueRange)
		{
			return BetterVolume3(Input, lookBack, paintBars, use2Bars, useTrueRange);
		}

		public BetterVolume3 BetterVolume3(ISeries<double> input, int lookBack, bool paintBars, bool use2Bars, bool useTrueRange)
		{
			if (cacheBetterVolume3 != null)
				for (int idx = 0; idx < cacheBetterVolume3.Length; idx++)
					if (cacheBetterVolume3[idx] != null && cacheBetterVolume3[idx].LookBack == lookBack && cacheBetterVolume3[idx].PaintBars == paintBars && cacheBetterVolume3[idx].Use2Bars == use2Bars && cacheBetterVolume3[idx].UseTrueRange == useTrueRange && cacheBetterVolume3[idx].EqualsInput(input))
						return cacheBetterVolume3[idx];
			return CacheIndicator<BetterVolume3>(new BetterVolume3(){ LookBack = lookBack, PaintBars = paintBars, Use2Bars = use2Bars, UseTrueRange = useTrueRange }, input, ref cacheBetterVolume3);
		}
	}
}

namespace NinjaTrader.NinjaScript.MarketAnalyzerColumns
{
	public partial class MarketAnalyzerColumn : MarketAnalyzerColumnBase
	{
		public Indicators.BetterVolume3 BetterVolume3(int lookBack, bool paintBars, bool use2Bars, bool useTrueRange)
		{
			return indicator.BetterVolume3(Input, lookBack, paintBars, use2Bars, useTrueRange);
		}

		public Indicators.BetterVolume3 BetterVolume3(ISeries<double> input , int lookBack, bool paintBars, bool use2Bars, bool useTrueRange)
		{
			return indicator.BetterVolume3(input, lookBack, paintBars, use2Bars, useTrueRange);
		}
	}
}

namespace NinjaTrader.NinjaScript.Strategies
{
	public partial class Strategy : NinjaTrader.Gui.NinjaScript.StrategyRenderBase
	{
		public Indicators.BetterVolume3 BetterVolume3(int lookBack, bool paintBars, bool use2Bars, bool useTrueRange)
		{
			return indicator.BetterVolume3(Input, lookBack, paintBars, use2Bars, useTrueRange);
		}

		public Indicators.BetterVolume3 BetterVolume3(ISeries<double> input , int lookBack, bool paintBars, bool use2Bars, bool useTrueRange)
		{
			return indicator.BetterVolume3(input, lookBack, paintBars, use2Bars, useTrueRange);
		}
	}
}

#endregion
