//
// Copyright (C) 2016, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//
#region Using declarations
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Input;
using System.Windows.Media;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Core;
using NinjaTrader.Gui;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Gui.SuperDom;
using NinjaTrader.Data;
using NinjaTrader.NinjaScript;
using NinjaTrader.Core.FloatingPoint;
using NinjaTrader.NinjaScript.DrawingTools;
#endregion

//CamsVpR2 version 1.0 written by Camdo 6/19/2017
//Volume Profile indicator
//Tick Replay not necessary but for accuracy
//if Tick Replay not enabled then:
//	bid/ask derived from granularity up/down bar, and CalculateOnBarClose
//if Tick Replay is enabled then:
//	bid/ask derived from NinjaTtrader last bar, and CalculateOnEachTick

//R1: initial release 6/19/2017
//R2: fixed granular bar offset, added Day and Volume granularity, improved VA calculation, added VA smoothing, fixed right orientation for region bar to bar, 07/07/2017

//This namespace holds Indicators in this folder and is required. Do not change it.
namespace NinjaTrader.NinjaScript.Indicators.Cam
{
	public class CamsVpR2 : Indicator
	{
		#region Properties
		internal class VolItem
		{
			public long up = 0;
			public long down = 0;
			public long total = 0;
		}
		private VolItem volItem;
		private List<VolItem> listVolItem = new List<VolItem>();
		private List<int> listPtr = new List<int>();
		private bool transitionComplete;
		private long buys = 0;
		private long sells = 0;
		private long totVol = 0;

		private double barSpacing = 0;
		private float xpos = 0;
		private float x = 0;
		private float yUpper = 0;
		private int	barWidth = 0;
		private float barHeight = 0;
		private List<int> regionIdxStart = new List<int>();
		private List<int> regionIdxEnd = new List<int>();
		private List<double> barWidthFacList = new List<double>();
		private List<int> xposList = new List<int>();
		private List<int> sessionStartList = new List<int>();
		private List<int> sessionStartsInWindowList = new List<int>();

		private List<Dictionary<double, VolItem>> dictionaryList = new List<Dictionary<double, VolItem>>();
		private Dictionary<double, VolItem> priceDictionary = new Dictionary<double, VolItem>();

		private double windowFac;
		private long volMax = 0;
		private double poc = 0;
		private double valueAreaHigh = 0;
		private double valueAreaLow = 0;
		private double valueAreaFac = 0;
		private int color = 0;

		private double price;
		private long volume;
		private long vol;
		private int regionStartBar0 = 0;
		private int regionStartBar1 = 0;
		private int boundaryFromX = 0;
		private int boundaryToX = 0;
		private int regionEndBar0 = 0;
		private int regionEndBar1 = 0;
		private DateTime dateF = DateTime.MinValue;
		private DateTime dateT = DateTime.MinValue;
		private	DateTime lastChartBarTime = DateTime.MinValue;
		private	DateTime firstChartBarTime = DateTime.MinValue;
		private int vpStartBar = 0;
		private int vpEndBar = 0;
		private int regionStartSlot = 0;
		private int regionEndSlot = 0;
		private int rgnStartSlot = 0;
		private int rgnEndSlot = 0;
		private double rgnStartX = 0;
		private double rgnEndX = 0;
		private double regionStartX = 0;
		private double regionEndX = 0;
		private double regionWidth = 0;
		private double barWidthP = 0;
		private SharpDX.RectangleF rectText;
		private DateTime startTime = DateTime.MinValue;
		private int a = 0;
		private int at = 0;
		private int firstWindowBar = 0;
		private int lastWindowBar = 0;

		private int cycle = 0;
		private double strtSlotIdx = 0;
		private double endSlotIdx = 0;

		private int lastBar0 = -1000;
		private int lastBar1 = -1000;
		private int calcMode = 0;

		#endregion

		protected override void OnStateChange()
		{
			if (State == State.SetDefaults)
			{
				Description				= @"Volume Profile: Plots a horizontal histogram of volume by price.";
				Name					= "CamsVpR2";
				Calculate				= Calculate.OnEachTick;
				BarsRequiredToPlot		= 0;
				IsChartOnly				= true;
				IsOverlay				= true;
				DrawOnPricePanel		= false;

				scaleType				= CamsVpR2ScaleType.MaxOfLadder;
				windowPercent			= 25;
				fixedScalingNumber		= .01;
				minSize					= 25;
				deltaScale				= .25;
				barSpacingPercent		= 0;
				opacity					= 100;
				zOrder					= -1;

				orientation				= CamsVpR2OrientationType.Left;
				margin					= 0;

				showVolumeProfile		= true;
				showAskBidProfile		= false;
				showDeltaProfile		= false;
				showScale				= CamsVpR2ShowScaleType.None;
				showPoc					= true;
				showValueArea			= true;
				valueAreaPercent		= 68.27;
				smoothVA				= 1;
				showRegionBoundaries	= false;

				showPerformance			= false;

				granularity				= CamsVpR2GranularityType.Volume;
				granularityPeriod		= 1;

				regionFrom				= CamsVpR2RegionFromType.Daily;
				regionTo				= CamsVpR2RegionToType.Daily;
				dateFrom				= DateTime.Parse("3/7/2017 4:00:00 PM");
				dateTo					= DateTime.Parse("3/7/2017 9:30:00 PM");

				VolFillBrush			= Brushes.Blue;
				askVolumeBrush			= Brushes.MediumAquamarine;
				bidVolumeBrush			= Brushes.Crimson;
				deltaPositiveBrush		= Brushes.Green;
				deltaNegativeBrush		= Brushes.DarkOrange;
				TextPosColorBrush		= Brushes.Snow;
				TextNegColorBrush		= Brushes.Maroon;
				PocColorBrush			= Brushes.Red;
				ValueAreaColorBrush		= Brushes.Brown;
			}
			else if (State == State.Configure)
			{
				windowFac = windowPercent / 100;
				barSpacing = barSpacingPercent / 100;
				valueAreaFac = valueAreaPercent / 100;
				transitionComplete = false;
				ClearOutputWindow();

				if (regionFrom == CamsVpR2RegionFromType.Bar || regionTo == CamsVpR2RegionToType.Bar)
				{
					regionFrom = CamsVpR2RegionFromType.Bar;
					regionTo = CamsVpR2RegionToType.Bar;
					showRegionBoundaries = false;
				}

				if (regionFrom == CamsVpR2RegionFromType.Daily || regionTo == CamsVpR2RegionToType.Daily)
				{
					regionFrom = CamsVpR2RegionFromType.Daily;
					regionTo = CamsVpR2RegionToType.Daily;
					showRegionBoundaries = false;
				}

				if (regionFrom == CamsVpR2RegionFromType.None || regionTo == CamsVpR2RegionToType.None)
				{
					regionFrom = CamsVpR2RegionFromType.None;
					regionTo = CamsVpR2RegionToType.None;
					showRegionBoundaries = false;
				}

				switch (granularity)
				{
					case CamsVpR2GranularityType.Minute:
						AddDataSeries(BarsPeriodType.Minute, granularityPeriod);
						break;
					case CamsVpR2GranularityType.Tick:
						AddDataSeries(BarsPeriodType.Tick, granularityPeriod);
						break;
					case CamsVpR2GranularityType.Day:
						AddDataSeries(BarsPeriodType.Day, granularityPeriod);
						break;
					case CamsVpR2GranularityType.Volume:
						AddDataSeries(BarsPeriodType.Volume, granularityPeriod);
						break;
					default:
						AddDataSeries(BarsPeriodType.Minute, granularityPeriod);
						break;
				}

				if (IsTickReplays[0] == true)
				{
					Calculate = Calculate.OnEachTick;
					calcMode = 0;
					listVolItem.Add(new VolItem()); //add 1st item
					sessionStartList.Add(0);
				}
				else
				{
					Calculate = Calculate.OnBarClose;
					calcMode = 1;
					listVolItem.Add(new VolItem()); //add 1st item
					listVolItem.Add(new VolItem()); //add 2nd item
				}
			}
			else if (State == State.DataLoaded)
			{
			}
			else if (State == State.Historical)
			{
				 SetZOrder(zOrder);
			}
			else if (State == State.Transition)
			{
				transitionComplete = true;
			}
		}

		protected override void OnBarUpdate()
		{
			if (calcMode == 0) //Tick Replay enabled
			{
				if (BarsInProgress == 1)
				{
					if (CurrentBars[1] != lastBar1)
					{
						listVolItem[listVolItem.Count - 1].down = sells;
						listVolItem[listVolItem.Count - 1].up = buys;
						listVolItem[listVolItem.Count - 1].total = buys + sells;

						buys = 0;
						sells = 0;
						listVolItem.Add(new VolItem());

						lastBar1 = CurrentBars[1];
					}

					listVolItem[listVolItem.Count - 1].down = sells;
					listVolItem[listVolItem.Count - 1].up = buys;
					listVolItem[listVolItem.Count - 1].total = buys + sells;
				}
				else //BIP == 0
				{
					if (CurrentBars[0] != lastBar0 && CurrentBars[0] > 0)
					{
						listPtr.Add(CurrentBars[1]); //list index = CurrentBars[0], value = CurrentBars[1]

						if (regionFrom == CamsVpR2RegionFromType.Daily)
						{
							if (Bars.IsFirstBarOfSession)
							{
								sessionStartList.Add(CurrentBars[0]);
							}
						}

						lastBar0 = CurrentBars[0];

						if (transitionComplete)
						{
							updateStraddledRegions();
						}
					}
				}
			}
			else if (calcMode == 1) //up/dowm COBC mode
			{
				if (BarsInProgress == 1)
				{
					listVolItem[listVolItem.Count - 1].down = Closes[1][0] < Opens[1][0] ? (long)Volumes[1][0] : 0;
					listVolItem[listVolItem.Count - 1].up = Closes[1][0] > Opens[1][0] ? (long)Volumes[1][0] : 0;
					listVolItem[listVolItem.Count - 1].total = (long)Volumes[1][0];

					listVolItem.Add(new VolItem());
				}
				else //BIP == 0
				{
					listPtr.Add(CurrentBars[1]); //list index = CurrentBars[0], value = CurrentBars[1]

					if (regionFrom == CamsVpR2RegionFromType.Daily)
					{
						if (Bars.IsFirstBarOfSession)
						{
							sessionStartList.Add(CurrentBars[0]);
						}
					}

					if (transitionComplete)
					{
						updateStraddledRegions();
					}
				}
			}
		}

		protected override void OnMarketData(MarketDataEventArgs e)
		{
			if (BarsInProgress == 0 || calcMode != 0)
				return;

			if (e.MarketDataType == MarketDataType.Last)
			{
				if (e.Price >= e.Ask)
				{
					buys += e.Volume;
				}
				else if (e.Price <= e.Bid)
				{
					sells += e.Volume;
				}
			}
		}

		//if straddled, keep drawing tool anchors syncronized with working bar.
		//A NT8 indicator cannot move a manually drawn achor. However, anchors placed in empty slots can be moved.
		//If the anchor is at least 2 bars ahead of the working bar, then this method will work, although unsupported by NT.

		void updateStraddledRegions()
		{
			double startSlot = 0;
			double endSlot = 0;

			foreach (DrawingTool dt in DrawObjects)
			{
				if (dt.Name == "CamsVp Region")
				{
					foreach (ChartAnchor anchor in dt.Anchors)
					{
						if (anchor.DisplayName == "Start")
						{
							startSlot = anchor.SlotIndex;
							if (startSlot >= CurrentBars[0] - 1)
							{
								anchor.SlotIndex = startSlot + 1;
							}
						}
						else if (anchor.DisplayName == "End")
						{
							endSlot = anchor.SlotIndex;
							if (endSlot >= CurrentBars[0] - 1)
							{
								anchor.SlotIndex = endSlot + 1;
							}
						}
					}
				}
			}
		}

		void makeVPDic(ChartControl chartControl, ChartScale chartScale, ChartPanel chartPanel)
		{
			getRegion(chartControl, chartScale, chartPanel);
			dictionaryList.Clear();

			if (regionIdxStart.Count !=0)
			{
				for (int k = 0; k <= regionIdxStart.Count - 1; k++)
				{
					priceDictionary = new Dictionary<double, VolItem>();
					dictionaryList.Add(priceDictionary);

					int startIdx = regionIdxStart[k];
					int endIdx = regionIdxEnd[k];
					for (int i = startIdx; i <= endIdx; i++)
					{
						price = BarsArray[1].GetClose(i - 1); //note listVolItem[i] is matched with Close[i - 1]
						if (!dictionaryList[k].ContainsKey(price))
						{
							VolItem v = new VolItem();
							dictionaryList[k].Add(price, v); //new dictionary element
							dictionaryList[k][price].up = listVolItem[i].up;
							dictionaryList[k][price].down = listVolItem[i].down;
							dictionaryList[k][price].total = listVolItem[i].total;
						}
						else
						{
							dictionaryList[k][price].up += listVolItem[i].up; //update existing dictionary element
							dictionaryList[k][price].down += listVolItem[i].down;
							dictionaryList[k][price].total += listVolItem[i].total;
						}
					}
				}
			}
		}

		void getRegion(ChartControl chartControl, ChartScale chartScale, ChartPanel chartPanel)
		{
			if (ChartBars != null)
			{
				regionIdxStart.Clear();
				regionIdxEnd.Clear();
				barWidthFacList.Clear();
				xposList.Clear();

				if (regionFrom == CamsVpR2RegionFromType.All)
				{
					regionStartBar0 = 0;
					regionStartBar1 = regionStartBar0 > 0 ? listPtr[regionStartBar0 - 1] + 2 : 1; //fine data composes larger bars
					boundaryFromX = chartControl.GetXByBarIndex(ChartBars, regionStartBar0);
					regionIdxStart.Add(regionStartBar1);
				}
				else if (regionFrom == CamsVpR2RegionFromType.Window)
				{
					regionStartBar0 = ChartBars.FromIndex; //first bar on chart has index = 0 and recorded in listPtr[1] with last granular bar idx
					regionStartBar1 = regionStartBar0 > 0 ? listPtr[regionStartBar0 - 1] + 2 : 1; //fine data composes larger bars
					boundaryFromX = chartControl.GetXByBarIndex(ChartBars, regionStartBar0);
					regionIdxStart.Add(regionStartBar1);
				}
				else if (regionFrom == CamsVpR2RegionFromType.Date)
				{
					dateF = dateRange(chartControl, dateFrom);
					regionStartBar0 = ChartBars.GetBarIdxByTime(chartControl, dateF);
					regionStartBar1 = regionStartBar0 > 0 ? listPtr[regionStartBar0 - 1] + 2 : 1; //fine data composes larger bars
					boundaryFromX = chartControl.GetXByBarIndex(ChartBars, regionStartBar0);
					regionIdxStart.Add(regionStartBar1);
				}
				else if (regionFrom == CamsVpR2RegionFromType.Bar)
				{
					//no action, pick up in regionTo
					//configure sets to and from as bar if either is bar
				}
				else if (regionFrom == CamsVpR2RegionFromType.Daily)
				{
					//no action, pick up in regionTo
					//configure sets to and from as Daily if either is Daily
				}
				else if (regionFrom == CamsVpR2RegionFromType.None)
				{
					//no action, DrawingTool code is at bottom of this method
					//configure sets to and from as none if either is none
				}

				if (regionTo == CamsVpR2RegionToType.Current)
				{
					regionEndBar0 = BarsArray[0].Count - 1;
					regionEndBar1 = listVolItem.Count - 2; //extra volItems added in configure
					boundaryToX = chartControl.GetXByBarIndex(ChartBars, regionEndBar0);
					regionIdxEnd.Add(regionEndBar1);
					windowBarWidthFacList(chartPanel);
					windowXpos(chartControl);
				}
				else if (regionTo == CamsVpR2RegionToType.Window)
				{
					regionEndBar0 = ChartBars.ToIndex;
					loadRegionEndBar1();
					boundaryToX = chartControl.GetXByBarIndex(ChartBars, regionEndBar0);
					regionIdxEnd.Add(regionEndBar1);
					windowBarWidthFacList(chartPanel);
					windowXpos(chartControl);
				}
				else if (regionTo == CamsVpR2RegionToType.Date)
				{
					dateT = dateRange(chartControl, dateTo);
					regionEndBar0 = ChartBars.GetBarIdxByTime(chartControl, dateT);
					loadRegionEndBar1();
					boundaryToX = chartControl.GetXByBarIndex(ChartBars, regionEndBar0);
					regionIdxEnd.Add(regionEndBar1);
					windowBarWidthFacList(chartPanel);
					windowXpos(chartControl);
				}
				else if (regionTo == CamsVpR2RegionToType.Bar)
				{
					firstWindowBar = ChartBars.FromIndex;
					lastWindowBar = ChartBars.ToIndex;

					for (int i = firstWindowBar; i <= lastWindowBar; i++)
					{
						regionStartBar0 = i;
						regionStartBar1 = regionStartBar0 > 0 ? listPtr[regionStartBar0 - 1] + 2 : 1; //fine data composes larger bars
						regionIdxStart.Add(regionStartBar1);
						regionEndBar0 = i;
						loadRegionEndBar1();
						regionIdxEnd.Add(regionEndBar1);

						barWidthP = chartControl.BarWidth + 5; //+ some small margin, note that .BarWidth is distance from wick to body edge
						regionStartX = chartControl.GetXByBarIndex(ChartBars, regionStartBar0);

						if (regionStartBar0 < BarsArray[0].Count - 1)
						{
							regionWidth = Math.Max(.80 * ((chartControl.GetXByBarIndex(ChartBars, regionStartBar0 + 1) - regionStartX) - (2 * barWidthP)), 0);
						}
						else
						{
							regionWidth = Math.Max(.80 * ((regionStartX - chartControl.GetXByBarIndex(ChartBars, regionStartBar0 - 1)) - (2 * barWidthP)), 0);
						}
						regionStartX = regionStartX + (barWidthP);
						regionEndX = regionStartX + regionWidth;

						barBarWidthFacList();
						barXpos();
					}
				}
				else if (regionTo == CamsVpR2RegionToType.Daily)
				{
					firstWindowBar = ChartBars.FromIndex;
					lastWindowBar = ChartBars.ToIndex;
					fillSessionStartsInWindowList(); //build list of FirstBarOfSession bars in window

					for (int i = 0; i <= sessionStartsInWindowList.Count - 1; i++)
					{
						regionStartBar0 = sessionStartsInWindowList[i];
						regionStartBar1 = regionStartBar0 > 0 ? listPtr[regionStartBar0 - 1] + 2 : 1; //fine data composes larger bars
						regionIdxStart.Add(regionStartBar1);

						if (i + 1 <= sessionStartsInWindowList.Count - 1)
						{
							if (sessionStartsInWindowList[i + 1] <= lastWindowBar)
							{
								regionEndBar0 = sessionStartsInWindowList[i + 1] - 1;
							}
							else
							{
								regionEndBar0 = lastWindowBar;
							}
						}
						else
						{
							if (lastWindowBar >= BarsArray[0].Count - 1)
							{
								regionEndBar0 = BarsArray[0].Count - 1;
							}
							else
							{
								regionEndBar0 = lastWindowBar;
							}
						}
						loadRegionEndBar1();
						regionEndBar1 = regionEndBar1 - 1;
						regionIdxEnd.Add(regionEndBar1);

						barWidthP = chartControl.BarWidth; //BarWidth is distance from wick to body edge
						regionStartX = chartControl.GetXByBarIndex(ChartBars, regionStartBar0);
						regionWidth = Math.Max(.80 * ((chartControl.GetXByBarIndex(ChartBars, regionEndBar0) - regionStartX) - (2 * barWidthP)), 0);
						regionStartX = regionStartX + (barWidthP);
						regionEndX = regionStartX + regionWidth;

						barBarWidthFacList();
						barXpos();
					}
				}
				else if (regionTo == CamsVpR2RegionToType.None)
				{
					//no action
					//configure sets to and from none if either is none
				}
			}
			else
			{
				//no chart bars
				Print("ChartBars == null, Indictor was not loaded from a chart, exiting strategy...");
				return;
			}

			foreach (DrawingTool dt in DrawObjects)
			{
				if (dt.Name == "CamsVp Region")
				{
					foreach (ChartAnchor anchor in dt.Anchors)
					{
						if (anchor.DisplayName == "Start")
						{
							rgnStartSlot = (int)anchor.SlotIndex; //start on wick end prior to next wick, Bars[0][0] = slot0, Bars[0][1] = slot1
							rgnStartX = anchor.GetPoint(chartControl, chartPanel, chartScale).X; //these are chart pixels

						}
						else if (anchor.DisplayName == "End")
						{
							rgnEndSlot = (int)anchor.SlotIndex; //start on wick end prior to next wick, Bars[0][0] = slot0, Bars[0][1] = slot1
							rgnEndX = anchor.GetPoint(chartControl, chartPanel, chartScale).X; //these are chart pixels
						}
					}

					if (rgnStartSlot >= 0 && rgnEndSlot >= 0) //1st cycle abnormalties when dt region is off window
					{
						regionStartSlot = Math.Min(rgnStartSlot, rgnEndSlot);
						regionEndSlot = Math.Max(rgnStartSlot, rgnEndSlot);
						regionStartX = Math.Min(rgnStartX, rgnEndX);
						regionEndX = Math.Max(rgnStartX, rgnEndX);

						if (regionEndX < chartControl.GetXByBarIndex(ChartBars, regionEndSlot))
						{
							if (regionEndSlot != regionStartSlot) regionEndSlot = regionEndSlot - 1;
						}

						if (regionStartSlot >= BarsArray[0].Count - 1)
						{
							regionStartBar0 = BarsArray[0].Count - 1;
						}
						else
						{
							regionStartBar0 = Math.Min(regionStartSlot, listPtr.Count - 1); //don't go out of range of listPtr[] scrolling bars to extreme left (one bar showing)
						}
						regionStartBar1 = regionStartBar0 > 0 ? listPtr[regionStartBar0 - 1] + 2 : 1; //fine data composes larger bars
						regionIdxStart.Add(regionStartBar1);

						if (regionEndSlot >= BarsArray[0].Count - 1)
						{
							regionEndBar0 = BarsArray[0].Count - 1; //note regionEndBar0 is meaningless since region is somewhere beyond the current bar
						}
						else
						{
							regionEndBar0 = regionEndSlot;
						}

						loadRegionEndBar1();
						regionIdxEnd.Add(regionEndBar1);

						DrawingToolBarWidthFacList();
						DrawingToolXpos();
					}
				}
			}
		}

		void windowBarWidthFacList(ChartPanel chartPanel)
		{
			if (scaleType == CamsVpR2ScaleType.MaxOfLadder)
			{
				barWidthFacList.Add(windowFac * chartPanel.W); //order matters in mixed (integer & double) math. Decimals less than zero will be zero
			}
			else
			{
				barWidthFacList.Add(fixedScalingNumber);
			}
		}

		void DrawingToolBarWidthFacList()
		{
			if (scaleType == CamsVpR2ScaleType.MaxOfLadder)
			{
				barWidthFacList.Add(Math.Max((regionEndX - regionStartX), minSize));
			}
			else
			{
				barWidthFacList.Add(fixedScalingNumber);
			}
		}

		void barBarWidthFacList()
		{
			if (scaleType == CamsVpR2ScaleType.MaxOfLadder)
			{
				barWidthFacList.Add(regionWidth);
			}
			else
			{
				barWidthFacList.Add(fixedScalingNumber);
			}
		}

		void windowXpos(ChartControl chartControl)
		{
			if (orientation == CamsVpR2OrientationType.Left)
			{
				xposList.Add(chartControl.CanvasLeft + margin);
			}
			else
			{
				xposList.Add(chartControl.CanvasRight - margin);
			}
		}

		void DrawingToolXpos()
		{
			if (orientation == CamsVpR2OrientationType.Left)
			{
				xposList.Add((int) regionStartX);
			}
			else
			{
				xposList.Add((int) regionEndX);
			}
		}

		void barXpos()
		{
			if (orientation == CamsVpR2OrientationType.Left)
			{
				xposList.Add((int)regionStartX);
			}
			else
			{
				xposList.Add((int)(regionStartX - (2 * barWidthP)));
			}
		}

		DateTime dateRange(ChartControl chartControl, DateTime date)
		{
			lastChartBarTime = ChartBars.GetTimeByBarIdx(chartControl, (BarsArray[0].Count - 1));
			firstChartBarTime = ChartBars.GetTimeByBarIdx(chartControl, 0);

			if (date > lastChartBarTime)
			{
				date = lastChartBarTime;
			}
			else if (date < firstChartBarTime)
			{
				date = firstChartBarTime;
			}
			return date;
		}

		void loadRegionEndBar1()
		{
			if (regionEndBar0 == (BarsArray[0].Count - 1))
			{
				regionEndBar1 = listVolItem.Count - 2; //extra volItems added in configure
			}
			else //not the working bar
			{
				if ((BarsArray[1].Count - 1) >= listPtr[regionEndBar0] + 1) //if it exists
				{
					regionEndBar1 = listPtr[regionEndBar0] + 1;
				}
				else
				{
					regionEndBar1 = listPtr[regionEndBar0];
				}
			}
		}

		void fillSessionStartsInWindowList()
		{
			int start = 0;

			sessionStartsInWindowList.Clear();

			for (int i = sessionStartList.Count - 1; i >= 0; i--)
			{
				if (sessionStartList[i] <= firstWindowBar)
				{
					start = i;
					break;
				}
			}

			for (int i = start; i <= sessionStartList.Count - 1; i++)
			{
				if (sessionStartList[i] >= firstWindowBar && sessionStartList[i] <= lastWindowBar)
				{
					sessionStartsInWindowList.Add(sessionStartList[i]);
				}

				if (sessionStartList[i] >= lastWindowBar)
				{
					break;
				}
			}
		}

		protected override void OnRender(ChartControl chartControl, ChartScale chartScale)
		{
			if (Bars == null || Bars.Instrument == null || IsInHitTest || !transitionComplete)
				return;

			if (showPerformance)
			{
				++cycle;
				startTime = NinjaTrader.Core.Globals.Now;
			}

			double	tickSize = Bars.Instrument.MasterInstrument.TickSize;
			ChartPanel chartPanel = ChartPanel;
			SharpDX.Direct2D1.Brush brushDX = VolFillBrush.ToDxBrush(RenderTarget); //just to declare

			SharpDX.Direct2D1.Brush volFillBrushDX	= VolFillBrush.ToDxBrush(RenderTarget);
			volFillBrushDX.Opacity = (float)(opacity / 100.0);

			SharpDX.Direct2D1.Brush askVolumeBrushDX	= askVolumeBrush.ToDxBrush(RenderTarget);
			askVolumeBrushDX.Opacity = (float)(opacity / 100.0);

			SharpDX.Direct2D1.Brush bidVolumeBrushDX	= bidVolumeBrush.ToDxBrush(RenderTarget);
			bidVolumeBrushDX.Opacity = (float)(opacity / 100.0);

			SharpDX.Direct2D1.Brush deltaPositiveBrushDX	= deltaPositiveBrush.ToDxBrush(RenderTarget);
			deltaPositiveBrushDX.Opacity = (float)(opacity / 100.0);

			SharpDX.Direct2D1.Brush deltaNegativeBrushDX	= deltaNegativeBrush.ToDxBrush(RenderTarget);
			deltaNegativeBrushDX.Opacity = (float)(opacity / 100.0);

			SharpDX.Direct2D1.Brush textPosBrushDX = TextPosColorBrush.ToDxBrush(RenderTarget);
			textPosBrushDX.Opacity = (float)(opacity / 100.0);

			SharpDX.Direct2D1.Brush textNegBrushDX = TextNegColorBrush.ToDxBrush(RenderTarget);
			textNegBrushDX.Opacity = (float)(opacity / 100.0);

			SharpDX.Direct2D1.Brush pocBrushDX = PocColorBrush.ToDxBrush(RenderTarget);
			pocBrushDX.Opacity = (float)(opacity / 100.0);

			SharpDX.Direct2D1.Brush valueAreaBrushDX = ValueAreaColorBrush.ToDxBrush(RenderTarget);
			valueAreaBrushDX.Opacity = (float)(opacity / 100.0);

			CamsVpR2BarType barType;

			makeVPDic(chartControl, chartScale, chartPanel);

			if (showRegionBoundaries)
			{
				barWidth = (int) chartControl.GetBarPaintWidth(chartControl.BarsArray[0]); //pixels
				barHeight = barWidth;
				yUpper = chartScale.GetYByValue(BarsArray[0].GetHigh(regionStartBar0) + (16 * tickSize)) + barWidth; //square in pixels
				xpos = boundaryFromX - (barWidth / 2);
				brushDX = deltaPositiveBrushDX;
				RenderTarget.FillRectangle(new SharpDX.RectangleF(xpos, yUpper, barWidth, barHeight), brushDX);
				yUpper = chartScale.GetYByValue(BarsArray[0].GetHigh(regionEndBar0) + (18 * TickSize)) + barWidth; //square in pixels
				xpos = boundaryToX - (barWidth / 2);
				brushDX = deltaNegativeBrushDX;
				RenderTarget.FillRectangle(new SharpDX.RectangleF(xpos, yUpper, barWidth, barHeight), brushDX);
			}

			double space = Math.Max(0, tickSize * barSpacing / 2); //$
			double yUpperAdder = (tickSize / 2) - space;
				barHeight = Math.Max(1, chartScale.GetPixelsForDistance(tickSize - (tickSize * barSpacing))); //pixels
			float textHeight = Math.Max(1, chartScale.GetPixelsForDistance(tickSize - (tickSize * .1))); //pixels
			double textYUpperAddr = tickSize * .45;

			if (dictionaryList.Count != 0)
			{
				for (int i = 0; i <= dictionaryList.Count - 1; i++)
				{
					if (dictionaryList[i].Count > 0)
					{
						poc = 0;
						volMax = 0;

						foreach (KeyValuePair<double, VolItem> element in dictionaryList[i])
						{
							if (element.Value.total > volMax)
							{
								volMax = element.Value.total;
								poc = element.Key; //price at max volume is poc
							}
						}

						if (showValueArea)
						{
							GetValueArea(i, tickSize);
						}

						double barWidthFac;

						if (scaleType == CamsVpR2ScaleType.MaxOfLadder)
						{
							barWidthFac = barWidthFacList[i] / volMax; //order matters in mixed (integer & double) math. Decimals less than zero will be zero
						}
						else
						{
							barWidthFac = barWidthFacList[i];
						}

						if (orientation == CamsVpR2OrientationType.Left)
						{
							xpos = xposList[i];
						}
						else
						{
							x = xposList[i];
						}

						if (showVolumeProfile == true)
						{
							barType = CamsVpR2BarType.Total;
							PaintBars(chartScale, yUpperAdder, barWidthFac, volFillBrushDX, barHeight, barType, i);
						}

						if (showAskBidProfile == true)
						{
							if (orientation == CamsVpR2OrientationType.Left)
							{
								barType = CamsVpR2BarType.Total;
								PaintBars(chartScale, yUpperAdder, barWidthFac, volFillBrushDX, barHeight, barType, i);

								barType = CamsVpR2BarType.AskBid;
								PaintBars(chartScale, yUpperAdder, barWidthFac, bidVolumeBrushDX, barHeight, barType, i);

								barType = CamsVpR2BarType.Ask;
								PaintBars(chartScale, yUpperAdder, barWidthFac, askVolumeBrushDX, barHeight, barType, i);
							}

							if (orientation == CamsVpR2OrientationType.Right)
							{
								barType = CamsVpR2BarType.Total;
								PaintBars(chartScale, yUpperAdder, barWidthFac, volFillBrushDX, barHeight, barType, i);

								barType = CamsVpR2BarType.AskBid;
								PaintBars(chartScale, yUpperAdder, barWidthFac, askVolumeBrushDX, barHeight, barType, i);

								barType = CamsVpR2BarType.Bid;
								PaintBars(chartScale, yUpperAdder, barWidthFac, bidVolumeBrushDX, barHeight, barType, i);
							}
						}

						if (showDeltaProfile == true)
						{
							double delta = 0;
							double deltaMax = 0;

							foreach (KeyValuePair<double, VolItem> element in dictionaryList[i])
							{
								delta = element.Value.up - element.Value.down;

								if (Math.Abs(delta) > deltaMax)
								{
									deltaMax = Math.Abs(delta); //deltaMax is an absolute value
								}
							}

							double deltaBarWidthFac = barWidthFac * volMax / deltaMax * deltaScale; //delta relative to volMax

							foreach (KeyValuePair<double, VolItem> element in dictionaryList[i])
							{
								yUpper = chartScale.GetYByValue(element.Key + yUpperAdder); //pixels
								delta = element.Value.up - element.Value.down;
								barWidth = (int)(Math.Abs(delta) * deltaBarWidthFac);

								if (delta >= 0)
								{
									brushDX = deltaPositiveBrushDX;
								}
								else
								{
									brushDX = deltaNegativeBrushDX;
								}

								if (orientation == CamsVpR2OrientationType.Right)
								{
									xpos = x - barWidth;
								}

								RenderTarget.FillRectangle(new SharpDX.RectangleF(xpos, yUpper, barWidth, barHeight), brushDX);
							}
						}

						if (showPoc && poc != 0)
						{
							yUpper = chartScale.GetYByValue(poc + yUpperAdder); //pixels
							barWidth = (int) (dictionaryList[i][poc].total * barWidthFac);

							if (orientation == CamsVpR2OrientationType.Right)
							{
								xpos = x - barWidth;
							}
							brushDX = pocBrushDX;

							RenderTarget.FillRectangle(new SharpDX.RectangleF(xpos, yUpper, barWidth, barHeight), brushDX);
						}

						if (showValueArea && valueAreaHigh != 0)
						{
							yUpper = chartScale.GetYByValue(valueAreaHigh + yUpperAdder); //pixels
							barWidth = (int) (dictionaryList[i][valueAreaHigh].total * barWidthFac);

							if (orientation == CamsVpR2OrientationType.Right)
							{
								xpos = x - barWidth;
							}
							brushDX = valueAreaBrushDX;

							RenderTarget.FillRectangle(new SharpDX.RectangleF(xpos, yUpper, barWidth, barHeight), brushDX);

							yUpper = chartScale.GetYByValue(valueAreaLow + yUpperAdder); //pixels
							barWidth = (int) (dictionaryList[i][valueAreaLow].total * barWidthFac);

							if (orientation == CamsVpR2OrientationType.Right)
							{
								xpos = x - barWidth;
							}

							RenderTarget.FillRectangle(new SharpDX.RectangleF(xpos, yUpper, barWidth, barHeight), brushDX);
						}

						if (showScale != CamsVpR2ShowScaleType.None)
						{
							string rowVolumeStr = "";

							if (textHeight >= 0) //8
							{
								foreach (KeyValuePair<double, VolItem> element in dictionaryList[i])
								{
									yUpper = chartScale.GetYByValue(element.Key + yUpperAdder); //pixels

									if (showScale == CamsVpR2ShowScaleType.Volume)
									{
										rowVolumeStr = RowVolumeString(element.Value.total); //format number
										color = 0;
									}
									else if (showScale == CamsVpR2ShowScaleType.AskBid)
									{
										rowVolumeStr = RowVolumeString(element.Value.up) + " x " + RowVolumeString(element.Value.down);
										color = 0;
									}
									else if (showScale == CamsVpR2ShowScaleType.Delta)
									{
						 				long delta = element.Value.up - element.Value.down;
										rowVolumeStr = RowVolumeString(delta);

										if (delta >= 0)
										{
											color = 0;
										}
										else
										{
											color = 1;
										}
									}

									if (orientation == CamsVpR2OrientationType.Right)
									{
										xpos = x - barWidth;
									}

									int y = chartScale.GetYByValue(element.Key + textYUpperAddr);

									if (orientation == CamsVpR2OrientationType.Left)
									{
										SharpDX.DirectWrite.TextFormat textFormat = new SharpDX.DirectWrite.TextFormat(NinjaTrader.Core.Globals.DirectWriteFactory, "Arial", textHeight){TextAlignment = SharpDX.DirectWrite.TextAlignment.Leading};
										rectText = new SharpDX.RectangleF(xpos + 2, y, 150, textHeight);

										if (color == 0)
										{
											RenderTarget.DrawText(rowVolumeStr, textFormat, rectText, textPosBrushDX);
										}
										else
										{
											RenderTarget.DrawText(rowVolumeStr, textFormat, rectText, textNegBrushDX);
										}
										textFormat.Dispose();
									}
									else
									{
										SharpDX.DirectWrite.TextFormat textFormat = new SharpDX.DirectWrite.TextFormat(NinjaTrader.Core.Globals.DirectWriteFactory, "Arial", textHeight){TextAlignment = SharpDX.DirectWrite.TextAlignment.Trailing};
										rectText = new SharpDX.RectangleF(x - 152, y, 150, textHeight);

										if (color == 0)
										{
											RenderTarget.DrawText(rowVolumeStr, textFormat, rectText, textPosBrushDX);
										}
										else
										{
											RenderTarget.DrawText(rowVolumeStr, textFormat, rectText, textNegBrushDX);
										}
										textFormat.Dispose();
									}
								}
							}
						}
					}
				}
			}

			brushDX.Dispose();
			volFillBrushDX.Dispose();
			askVolumeBrushDX.Dispose();
			bidVolumeBrushDX.Dispose();
			deltaPositiveBrushDX.Dispose();
			deltaNegativeBrushDX.Dispose();
			textPosBrushDX.Dispose();
			textNegBrushDX.Dispose();
			pocBrushDX.Dispose();
			valueAreaBrushDX.Dispose();

			if (showPerformance)
			{
				double secBarCount = BarsArray[1].Count - 1;
				double actualElapsedTime = NinjaTrader.Core.Globals.Now.Subtract(startTime).TotalSeconds;
				Print("cycle = " + cycle
				//	+ "    startTime = " + startTime.ToString("HH:mm:ss:ffffff")
				//	+ "    endTime = " + NinjaTrader.Core.Globals.Now.ToString("HH:mm:ss:ffffff")
					+ "    paintTime = " + actualElapsedTime.ToString("N6") + " Sec"
					+ "    granular Bars = " + secBarCount + "    " + (secBarCount / int.MaxValue * 100).ToString("N6") + " % capacity"
				//	+ "    max granularBars = " + int.MaxValue
					+ " "
					);
			}
		}

		void PaintBars(ChartScale chartScale, double yUpperAdder, double barWidthFac, SharpDX.Direct2D1.Brush brushDX, float barHeight, CamsVpR2BarType barType, int i)
		{
			foreach (KeyValuePair<double, VolItem> element in dictionaryList[i])
			{
				yUpper = chartScale.GetYByValue(element.Key + yUpperAdder); //pixels

				if (barType == CamsVpR2BarType.Total)
				{
					barWidth = (int)(element.Value.total * barWidthFac);
				}
				else if (barType == CamsVpR2BarType.AskBid)
				{
					barWidth = (int)((element.Value.up + element.Value.down) * barWidthFac);
				}
				else if (barType == CamsVpR2BarType.Ask)
				{
					barWidth = (int)(element.Value.up * barWidthFac);
				}
				else if (barType == CamsVpR2BarType.Bid)
				{
					barWidth = (int)(element.Value.down * barWidthFac);
				}
				else if (barType == CamsVpR2BarType.Delta)
				{
					barWidth = (int)Math.Abs((element.Value.up - element.Value.down) * barWidthFac);
				}

				if (orientation == CamsVpR2OrientationType.Right)
				{
					xpos = x - barWidth;
				}

				RenderTarget.FillRectangle(new SharpDX.RectangleF(xpos, yUpper, barWidth, barHeight), brushDX);
			}
		}

		void GetValueArea(int i, double tickSize)
		{
			if (dictionaryList[i].Count > 0 && poc != 0)
			{
				int highIdx = 0;
				int lowIdx = 0;
				long upTestV = 0;
				long dnTestV = 0;
				long valueVol = 0;
				long areaVol = 0;
				valueAreaHigh = 0;
				valueAreaLow = 0;

				List<double> priceList = new List<double>();

				foreach (KeyValuePair<double, VolItem> element in dictionaryList[i])
				{
					valueVol += element.Value.total;
					priceList.Add(element.Key);
				}
				valueVol = (long)(valueVol * valueAreaFac);
				priceList.Sort();

				for (int n = 0; n <= priceList.Count - 1; ++n)
				{
					if (priceList[n] == poc)
					{
						highIdx = n;
						lowIdx = highIdx;
						break;
					}
				}
				areaVol = dictionaryList[i][poc].total;

				while (areaVol < valueVol)
				{
					if (upTestV == 0)
					{
						for (int n = 1; (n <= smoothVA && highIdx < priceList.Count - 1); ++n)
						{
							upTestV += dictionaryList[i][priceList[highIdx + 1]].total;
							++highIdx;
						}
					}

					if (dnTestV == 0)
					{
						for (int n = 1; (n <= smoothVA && lowIdx > 0); ++n)
						{
							dnTestV += dictionaryList[i][priceList[lowIdx - 1]].total;
							--lowIdx;
						}
					}

					if (upTestV > dnTestV)
					{
						areaVol += upTestV;
						upTestV = 0;
					}
					else
					{
						areaVol += dnTestV;
						dnTestV = 0;
					}
				}
				valueAreaHigh = priceList[highIdx];
				valueAreaLow = priceList[lowIdx];
			}
		}

		private string RowVolumeString(long volume)
		{
			if (volume >= 100000)
				return (volume / 1000).ToString() + "K";
			else
				return volume.ToString();
		}

		#region Properties

		[NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Scaling Scheme", Description = "", Order = 0, GroupName = "NinjaScriptParameters")]
		public CamsVpR2ScaleType scaleType
		{ get; set; }

		[Range(1, 100)]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Window Percent", Description = "Percent of window the largest volume bar will occupy. For MaxOfLadder scaling scheme", Order = 1, GroupName = "NinjaScriptParameters")]
		public double windowPercent
		{ get; set; }

		[Range(0,1), NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Scaling Number", Description = "For fixed scaling scheme", Order = 2, GroupName = "NinjaScriptParameters")]
		public double fixedScalingNumber
		{ get; set; }

		[Range(0,int.MaxValue), NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Minimum Size", Description = "For drawing tool regions, pixels", Order = 3, GroupName = "NinjaScriptParameters")]
		public int minSize
		{ get; set; }

		[Range(0,1), NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Delta Scale", Description = "Delta bar length relative to max volume", Order = 4, GroupName = "NinjaScriptParameters")]
		public double deltaScale
		{ get; set; }

		[Range(0, 99)]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Bar Spacing", Description = "Percent of tick size (0 - 99%), 0 = solid 99 = thin bars", Order = 5, GroupName = "NinjaScriptParameters")]
		public double barSpacingPercent
		{ get; set; }

		[Range(0, double.MaxValue)]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Opacity", Order = 6, GroupName = "NinjaScriptParameters")]
		public double opacity
		{ get; set; }

		[Range(-1,int.MaxValue), NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Z-Order", Description = "-1 background, 999999... foreground", Order = 7, GroupName = "NinjaScriptParameters")]
		public int zOrder
		{ get; set; }

		[NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Orientation", Description="", Order = 8, GroupName = "NinjaScriptParameters")]
		public CamsVpR2OrientationType orientation
		{get; set;}

		[Range(0, int.MaxValue)]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Margin", Description = "Pixels", Order = 9, GroupName = "NinjaScriptParameters")]
		public int margin
		{ get; set; }

		[Display(ResourceType = typeof(Custom.Resource), Name = "Show Volume Profile", Description="", Order = 10, GroupName = "NinjaScriptParameters")]
		public bool showVolumeProfile
		{ get; set; }

		[Display(ResourceType = typeof(Custom.Resource), Name = "Show Ask-Bid Profile", Description = "", Order = 11, GroupName = "NinjaScriptParameters")]
		public bool showAskBidProfile
		{ get; set; }

		[Display(ResourceType = typeof(Custom.Resource), Name = "Show Delta Profile", Description = "", Order = 12, GroupName = "NinjaScriptParameters")]
		public bool showDeltaProfile
		{ get; set; }

		[NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Show Scale", Description = "", Order = 13, GroupName = "NinjaScriptParameters")]
		public CamsVpR2ShowScaleType showScale
		{get; set;}

		[Display(ResourceType = typeof(Custom.Resource), Name = "Show POC", Description = "", Order = 14, GroupName = "NinjaScriptParameters")]
		public bool showPoc
		{ get; set; }

		[Display(ResourceType = typeof(Custom.Resource), Name = "Show Value Area", Description = "", Order = 15, GroupName = "VA")]
		public bool showValueArea
		{ get; set; }

		[Range(0, 99.99)]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Value Area Percent", Description = "Defines value area as percent of total, usually 70%", Order = 16, GroupName = "VA")]
		public double valueAreaPercent
		{ get; set; }

		[Range(1, int.MaxValue)]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Smooth VA", Description = "Group n histogram levels to smooth, usually 2", Order = 17, GroupName = "VA")]
		public int smoothVA
		{ get; set; }

		[Display(ResourceType = typeof(Custom.Resource), Name = "Show Region Boundarys", Description = "", Order = 18, GroupName = "NinjaScriptParameters")]
		public bool showRegionBoundaries
		{ get; set; }

		[Display(ResourceType = typeof(Custom.Resource), Name = "Show Performance", Description = "Elapsed time to generate the volume profile and repaint the screen", Order = 19, GroupName = "NinjaScriptParameters")]
		public bool showPerformance
		{ get; set; }

		[NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Granularity", Description = "", Order = 20, GroupName = "NinjaScriptParameters")]
		public CamsVpR2GranularityType granularity
		{get; set;}

		[Range(0, int.MaxValue)]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Granularity Period", Description = "", Order = 21, GroupName = "NinjaScriptParameters")]
		public int granularityPeriod
		{ get; set; }

		[NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Region From", Description = "Profile data begins", Order = 22, GroupName = "NinjaScriptParameters")]
		public CamsVpR2RegionFromType regionFrom
		{get; set;}

		[NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Region To", Description = "Profile data ends", Order = 23, GroupName = "NinjaScriptParameters")]
		public CamsVpR2RegionToType regionTo
		{get; set;}

		[NinjaScriptProperty]
		[PropertyEditor("NinjaTrader.Gui.Tools.ChartAnchorTimeEditor")]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Date From", Description = "Used for region by date", Order = 24, GroupName = "NinjaScriptParameters")]
		public DateTime dateFrom
		{get; set;}

		[NinjaScriptProperty]
		[PropertyEditor("NinjaTrader.Gui.Tools.ChartAnchorTimeEditor")]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Date To", Description = "Used for region by date", Order = 25, GroupName = "NinjaScriptParameters")]
		public DateTime dateTo
		{get; set;}

		[XmlIgnore]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Volume Fill Color", Order = 26, GroupName = "NinjaScriptParameters")]
		public Brush VolFillBrush { get; set; }

		[Browsable(false)]
		public string VolFillBrushSerialize
		{
			get { return Serialize.BrushToString(VolFillBrush); }
			set { VolFillBrush = Serialize.StringToBrush(value); }
		}

		[XmlIgnore]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Ask Volume Color", Order = 27, GroupName = "NinjaScriptParameters")]
		public Brush askVolumeBrush { get; set; }

		[Browsable(false)]
		public string askVolumeBrushSerialize
		{
			get { return Serialize.BrushToString(askVolumeBrush); }
			set { askVolumeBrush = Serialize.StringToBrush(value); }
		}

		[XmlIgnore]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Bid Volume Color", Order = 28, GroupName = "NinjaScriptParameters")]
		public Brush bidVolumeBrush { get; set; }

		[Browsable(false)]
		public string bidVolumeBrushSerialize
		{
			get { return Serialize.BrushToString(bidVolumeBrush); }
			set { bidVolumeBrush = Serialize.StringToBrush(value); }
		}

		[XmlIgnore]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Delta Positive Color", Order = 29, GroupName = "NinjaScriptParameters")]
		public Brush deltaPositiveBrush { get; set; }

		[Browsable(false)]
		public string deltaPositiveBrushSerialize
		{
			get { return Serialize.BrushToString(deltaPositiveBrush); }
			set { deltaPositiveBrush = Serialize.StringToBrush(value); }
		}

		[XmlIgnore]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Delta Negative Color", Order = 30, GroupName = "NinjaScriptParameters")]
		public Brush deltaNegativeBrush { get; set; }

		[Browsable(false)]
		public string deltaNegativeBrushSerialize
		{
			get { return Serialize.BrushToString(deltaNegativeBrush); }
			set { deltaNegativeBrush = Serialize.StringToBrush(value); }
		}

		[XmlIgnore]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Text Positive Color", Order = 31, GroupName = "NinjaScriptParameters")]
		public Brush TextPosColorBrush { get; set; }

		[Browsable(false)]
		public string TextPosColorBrushSerialize
		{
			get { return Serialize.BrushToString(TextPosColorBrush); }
			set { TextPosColorBrush = Serialize.StringToBrush(value); }
		}

		[XmlIgnore]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Text Negative Color", Order = 32, GroupName = "NinjaScriptParameters")]
		public Brush TextNegColorBrush { get; set; }

		[Browsable(false)]
		public string TextNegColorBrushSerialize
		{
			get { return Serialize.BrushToString(TextNegColorBrush); }
			set { TextNegColorBrush = Serialize.StringToBrush(value); }
		}

		[XmlIgnore]
		[Display(ResourceType = typeof(Custom.Resource), Name = "POC Color", Order = 33, GroupName = "NinjaScriptParameters")]
		public Brush PocColorBrush { get; set; }

		[Browsable(false)]
		public string PocColorBrushSerialize
		{
			get { return Serialize.BrushToString(PocColorBrush); }
			set { PocColorBrush = Serialize.StringToBrush(value); }
		}

		[XmlIgnore]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Value Area Color", Order = 34, GroupName = "NinjaScriptParameters")]
		public Brush ValueAreaColorBrush { get; set; }

		[Browsable(false)]
		public string ValueAreaColorBrushSerialize
		{
			get { return Serialize.BrushToString(ValueAreaColorBrush); }
			set { ValueAreaColorBrush = Serialize.StringToBrush(value); }
		}
		#endregion
	}
}
public enum CamsVpR2GranularityType {Minute, Tick, Day, Volume}
public enum CamsVpR2OrientationType {Left, Right}
public enum CamsVpR2ScaleType {MaxOfLadder, Fixed}
public enum CamsVpR2RegionFromType {All, Window, Date, Bar, Daily, None}
public enum CamsVpR2RegionToType {Current, Window, Date, Bar, Daily, None}
public enum CamsVpR2ShowScaleType {Volume, AskBid, Delta, None}
public enum CamsVpR2BarType {Total, Ask, Bid, AskBid, Delta}

#region NinjaScript generated code. Neither change nor remove.

namespace NinjaTrader.NinjaScript.Indicators
{
	public partial class Indicator : NinjaTrader.Gui.NinjaScript.IndicatorRenderBase
	{
		private Cam.CamsVpR2[] cacheCamsVpR2;
		public Cam.CamsVpR2 CamsVpR2(CamsVpR2ScaleType scaleType, double fixedScalingNumber, int minSize, double deltaScale, int zOrder, CamsVpR2OrientationType orientation, CamsVpR2ShowScaleType showScale, CamsVpR2GranularityType granularity, CamsVpR2RegionFromType regionFrom, CamsVpR2RegionToType regionTo, DateTime dateFrom, DateTime dateTo)
		{
			return CamsVpR2(Input, scaleType, fixedScalingNumber, minSize, deltaScale, zOrder, orientation, showScale, granularity, regionFrom, regionTo, dateFrom, dateTo);
		}

		public Cam.CamsVpR2 CamsVpR2(ISeries<double> input, CamsVpR2ScaleType scaleType, double fixedScalingNumber, int minSize, double deltaScale, int zOrder, CamsVpR2OrientationType orientation, CamsVpR2ShowScaleType showScale, CamsVpR2GranularityType granularity, CamsVpR2RegionFromType regionFrom, CamsVpR2RegionToType regionTo, DateTime dateFrom, DateTime dateTo)
		{
			if (cacheCamsVpR2 != null)
				for (int idx = 0; idx < cacheCamsVpR2.Length; idx++)
					if (cacheCamsVpR2[idx] != null && cacheCamsVpR2[idx].scaleType == scaleType && cacheCamsVpR2[idx].fixedScalingNumber == fixedScalingNumber && cacheCamsVpR2[idx].minSize == minSize && cacheCamsVpR2[idx].deltaScale == deltaScale && cacheCamsVpR2[idx].zOrder == zOrder && cacheCamsVpR2[idx].orientation == orientation && cacheCamsVpR2[idx].showScale == showScale && cacheCamsVpR2[idx].granularity == granularity && cacheCamsVpR2[idx].regionFrom == regionFrom && cacheCamsVpR2[idx].regionTo == regionTo && cacheCamsVpR2[idx].dateFrom == dateFrom && cacheCamsVpR2[idx].dateTo == dateTo && cacheCamsVpR2[idx].EqualsInput(input))
						return cacheCamsVpR2[idx];
			return CacheIndicator<Cam.CamsVpR2>(new Cam.CamsVpR2(){ scaleType = scaleType, fixedScalingNumber = fixedScalingNumber, minSize = minSize, deltaScale = deltaScale, zOrder = zOrder, orientation = orientation, showScale = showScale, granularity = granularity, regionFrom = regionFrom, regionTo = regionTo, dateFrom = dateFrom, dateTo = dateTo }, input, ref cacheCamsVpR2);
		}
	}
}

namespace NinjaTrader.NinjaScript.MarketAnalyzerColumns
{
	public partial class MarketAnalyzerColumn : MarketAnalyzerColumnBase
	{
		public Indicators.Cam.CamsVpR2 CamsVpR2(CamsVpR2ScaleType scaleType, double fixedScalingNumber, int minSize, double deltaScale, int zOrder, CamsVpR2OrientationType orientation, CamsVpR2ShowScaleType showScale, CamsVpR2GranularityType granularity, CamsVpR2RegionFromType regionFrom, CamsVpR2RegionToType regionTo, DateTime dateFrom, DateTime dateTo)
		{
			return indicator.CamsVpR2(Input, scaleType, fixedScalingNumber, minSize, deltaScale, zOrder, orientation, showScale, granularity, regionFrom, regionTo, dateFrom, dateTo);
		}

		public Indicators.Cam.CamsVpR2 CamsVpR2(ISeries<double> input , CamsVpR2ScaleType scaleType, double fixedScalingNumber, int minSize, double deltaScale, int zOrder, CamsVpR2OrientationType orientation, CamsVpR2ShowScaleType showScale, CamsVpR2GranularityType granularity, CamsVpR2RegionFromType regionFrom, CamsVpR2RegionToType regionTo, DateTime dateFrom, DateTime dateTo)
		{
			return indicator.CamsVpR2(input, scaleType, fixedScalingNumber, minSize, deltaScale, zOrder, orientation, showScale, granularity, regionFrom, regionTo, dateFrom, dateTo);
		}
	}
}

namespace NinjaTrader.NinjaScript.Strategies
{
	public partial class Strategy : NinjaTrader.Gui.NinjaScript.StrategyRenderBase
	{
		public Indicators.Cam.CamsVpR2 CamsVpR2(CamsVpR2ScaleType scaleType, double fixedScalingNumber, int minSize, double deltaScale, int zOrder, CamsVpR2OrientationType orientation, CamsVpR2ShowScaleType showScale, CamsVpR2GranularityType granularity, CamsVpR2RegionFromType regionFrom, CamsVpR2RegionToType regionTo, DateTime dateFrom, DateTime dateTo)
		{
			return indicator.CamsVpR2(Input, scaleType, fixedScalingNumber, minSize, deltaScale, zOrder, orientation, showScale, granularity, regionFrom, regionTo, dateFrom, dateTo);
		}

		public Indicators.Cam.CamsVpR2 CamsVpR2(ISeries<double> input , CamsVpR2ScaleType scaleType, double fixedScalingNumber, int minSize, double deltaScale, int zOrder, CamsVpR2OrientationType orientation, CamsVpR2ShowScaleType showScale, CamsVpR2GranularityType granularity, CamsVpR2RegionFromType regionFrom, CamsVpR2RegionToType regionTo, DateTime dateFrom, DateTime dateTo)
		{
			return indicator.CamsVpR2(input, scaleType, fixedScalingNumber, minSize, deltaScale, zOrder, orientation, showScale, granularity, regionFrom, regionTo, dateFrom, dateTo);
		}
	}
}

#endregion
