#region Using declarations
using System;
using System.Collections.Generic;
using System.Collections.Concurrent;
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Forms;
using System.Windows.Input;
using System.Windows.Media;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Gui;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Gui.SuperDom;
using NinjaTrader.Gui.Tools;
using NinjaTrader.Data;
using NinjaTrader.NinjaScript;
using NinjaTrader.Core.FloatingPoint;
using NinjaTrader.NinjaScript.DrawingTools;
using SharpDX;
using SharpDX.DirectWrite;
#endregion

namespace NinjaTrader.NinjaScript.Indicators.Infinity
{
	public class SnapShot : Indicator
	{
		#region snapShot
		
		public class snapShot
		{
			public double vol = 0.0;
			public double poc = 0.0;
			public double dta = 0.0;
			
			public ConcurrentDictionary<double, RowData> lst = new ConcurrentDictionary<double, RowData>();
			
			public void calc(MasterInstrument mi, double ts)
			{
				if(!this.lst.IsEmpty)
				{
					// set poc
					
					this.poc = this.lst.Keys.Aggregate((i, j) => this.lst[i].tv > this.lst[j].tv ? i : j);
					
					// set imbalances & delta
					
					double askSum = 0.0;
					double bidSum = 0.0;
					
					foreach(KeyValuePair<double, RowData> rd in this.lst)
					{
						double volR = 0.0;
						double askP = 0.0;
						double bidP = 0.0;
						double askV = 0.0;
						double bidV = 0.0;
						
						// ask imbalance
						
						askP = mi.RoundToTickSize(rd.Key);
						bidP = mi.RoundToTickSize(rd.Key - ts);
						
						askV = this.getAskVolume(askP);
						bidV = this.getBidVolume(bidP);
						
						rd.Value.ab = (askV > bidV) ? 1 : 0;
						
						if(askV + bidV > 0)
						{
							if(askV > bidV)
							{
								volR = (askV - bidV) / (askV + bidV);
							}
							
						 	rd.Value.ai = volR;
						}
						
						// bid imbalance
						
						volR = 0.0;
						askP = mi.RoundToTickSize(rd.Key + ts);
						bidP = mi.RoundToTickSize(rd.Key);
						
						askV = this.getAskVolume(askP);
						bidV = this.getBidVolume(bidP);
						
						rd.Value.bb = (bidV > askV) ? 1 : 0;
						
						if(bidV + askV > 0)
						{
							if(bidV > askV)
							{
								volR = (bidV - askV) / (bidV + askV);
							}
							
						 	rd.Value.bi = volR;
						}
						
						// delta
						
						askSum += rd.Value.av;
						bidSum += rd.Value.bv;
					}
					
					// set delta
					
					this.dta = (askSum - bidSum);
				}
			}
			
			// getAskVolume
			//
			public double getAskVolume(double prc)
			{
				double v = 0.0;
				
				if(this.lst.ContainsKey(prc))
				{
					v = this.lst[prc].av;
				}
				
				return v;
			}
			
			// getBidVolume
			//
			public double getBidVolume(double prc)
			{
				double v = 0.0;
				
				if(this.lst.ContainsKey(prc))
				{
					v = this.lst[prc].bv;
				}
				
				return v;
			}
		}
		
		#endregion
		
		#region RowData
		
		public class RowData
		{
			public double tv = 0.0; // total volume
			public double av = 0.0;	// ask volume
			public double bv = 0.0; // bid volume
			public double ai = 0.0; // ask imbalance
			public double bi = 0.0; // bid imbalance
			public double ad = 0.0; // ask depth
			public double bd = 0.0; // bid depth
			public int    ab = 0;   // ask is bigger
			public int    bb = 0;   // bid is bigger
		}
		
		#endregion
		
		#region Variables
		
		private double bid = 0.0;
		private double ask = 0.0;
		private double cls = 0.0;
		private double vol = 0.0;
		
		private snapShot ss;
		
		private SimpleFont sf;
		
		GlyphTypeface gtf;
		System.Windows.Media.Typeface tfc;
		
		private int fSize = 0;
		
		// ---
		
		#endregion
		
		#region Menu
		
		private NinjaTrader.Gui.Chart.ChartTab		chartTab;
		private NinjaTrader.Gui.Chart.Chart			chartWindow;
		private bool								isToolBarButtonAdded;
		private System.Windows.DependencyObject		searchObject;
		private System.Windows.Controls.TabItem		tabItem;
		private System.Windows.Controls.Menu		theMenu;
		private string								theMenuAutomationID;
		private NinjaTrader.Gui.Tools.NTMenuItem	topMenuItem;
		private NinjaTrader.Gui.Tools.NTMenuItem	topMenuItemSubItem1;
		private NinjaTrader.Gui.Tools.NTMenuItem	topMenuItemSubItem2;
		private NinjaTrader.Gui.Tools.NTMenuItem	topMenuItemSubItem3;
		
		#endregion
		
		// OnStateChange
		//
		protected override void OnStateChange()
		{
			if(State == State.SetDefaults)
			{
				Description					= @"";
				Name						= "SnapShot";
				Calculate					= Calculate.OnEachTick;
				IsOverlay					= true;
				DisplayInDataBox			= true;
				DrawOnPricePanel			= true;
				DrawHorizontalGridLines		= false;
				DrawVerticalGridLines		= false;
				PaintPriceMarkers			= false;
				ScaleJustification			= NinjaTrader.Gui.Chart.ScaleJustification.Right;
				IsSuspendedWhileInactive	= true;
				
				showSnapShot  = false;
				rightMargin   = 0;
				snapShotWidth = 120;
				resetOnNewBar = false;
				
				minImbVolume = 100;
				minImbRatio  = 0.4;
				
				bckColor = Brushes.DimGray;
				txtColor = Brushes.Black;
				askColor = Brushes.Green;
				bidColor = Brushes.Firebrick;
			}
			else if(State == State.Configure)
			{
				sf = new NinjaTrader.Gui.Tools.SimpleFont("Consolas", 11);
				
				gtf = new GlyphTypeface();
				tfc = new System.Windows.Media.Typeface(new System.Windows.Media.FontFamily(sf.Family.ToString()), FontStyles.Normal, FontWeights.Normal, FontStretches.Normal);
            	tfc.TryGetGlyphTypeface(out gtf);
				
				if(ChartBars != null)
				{
					ZOrder = ChartBars.ZOrder + 1;
				}
			}
			else if(State == State.Historical)
			{
				if(ChartControl != null && !isToolBarButtonAdded)
				{
					ChartControl.Dispatcher.InvokeAsync((Action)(() =>
					{
						InsertWPFControls();
					}));
				}
				
				try
    			{
					ChartControl.Dispatcher.InvokeAsync(() =>
                	{
						ChartPanel.MouseDoubleClick += new MouseButtonEventHandler(dblClickEvent);
					});
				}
				catch(Exception e) { Console.WriteLine("Error: '{0}'", e); }
			}
			else if(State == State.Terminated)
			{
				if(ChartControl != null)
				{
					ChartControl.Dispatcher.InvokeAsync((Action)(() =>
					{
						RemoveWPFControls();
					}));
					
					if(ChartPanel != null)
					{
						try
	        			{
							ChartControl.Dispatcher.InvokeAsync(() =>
		                	{
								ChartPanel.MouseDoubleClick -= dblClickEvent;
							});
						}
						catch(Exception e) { Console.WriteLine("Error: '{0}'", e); }
					}
				}
			}
		}
		
		#region Mouse Events

		// midasMouseEvents
		//
		private void dblClickEvent(object sender, MouseButtonEventArgs e)
		{
			if(Control.ModifierKeys == Keys.Control)
			{
				if(ss != null)
				{
					ss.vol = 0.0;
					ss.poc = 0.0;
					ss.dta = 0.0;
					
					ss.lst.Clear();
				}
			}
			else
			{
				showSnapShot = !showSnapShot;
				
				if(showSnapShot)
				{
					ChartControl.Properties.BarMarginRight += (snapShotWidth + 10);	
				}
				else
				{
					ChartControl.Properties.BarMarginRight -= (snapShotWidth + 10);
				}
				
				topMenuItemSubItem1.Foreground = (showSnapShot) ? Brushes.Silver : Brushes.DimGray;
			}
			
			ForceRefresh();
		}
		
		#endregion
		
		#region Menu
		
		// InsertWPFControls
		//
		protected void InsertWPFControls()
		{
			chartWindow = System.Windows.Window.GetWindow(ChartControl.Parent) as Chart;
			
			theMenuAutomationID = string.Format("ChartToolbarBarSS{0}", DateTime.Now.ToString("yyMMddhhmmss"));
			
			foreach(System.Windows.DependencyObject item in chartWindow.MainMenu)
			{
				if(System.Windows.Automation.AutomationProperties.GetAutomationId(item) == theMenuAutomationID)
				{
					return;
				}
			}
			
			theMenu = new System.Windows.Controls.Menu
			{
				VerticalAlignment			= VerticalAlignment.Center,
				VerticalContentAlignment	= VerticalAlignment.Center,
				Style						= System.Windows.Application.Current.TryFindResource("SystemMenuStyle") as Style
			};
			
			System.Windows.Automation.AutomationProperties.SetAutomationId(theMenu, theMenuAutomationID);
			
			System.Windows.Media.Geometry topMenuItem1Icon = System.Windows.Media.Geometry.Parse("M2,5H10V2H12V22H10V18H6V15H10V13H4V10H10V8H2V5M14,5H17V8H14V5M14,10H19V13H14V10M14,15H22V18H14V15Z");
			
			topMenuItem = new Gui.Tools.NTMenuItem()
			{
				Header				= "SnapShot",
				Foreground			= Brushes.Silver,
				Icon				= topMenuItem1Icon,
				Margin				= new System.Windows.Thickness(0),
				Padding				= new System.Windows.Thickness(1),
				VerticalAlignment	= VerticalAlignment.Center,
				FontSize            = 12,
				Style				= System.Windows.Application.Current.TryFindResource("MainMenuItem") as Style
			};
			
			theMenu.Items.Add(topMenuItem);
			
			topMenuItemSubItem1 = new Gui.Tools.NTMenuItem()
			{
				BorderThickness		= new System.Windows.Thickness(0),
				Header				= "Show (Double Click)",
			    Foreground			= (showSnapShot) ? Brushes.Silver : Brushes.DimGray,
				Style				= System.Windows.Application.Current.TryFindResource("InstrumentMenuItem") as Style
			};
			
			topMenuItemSubItem1.Click += TopMenuItem1SubItem1_Click;
			topMenuItem.Items.Add(topMenuItemSubItem1);
			
			topMenuItemSubItem2 = new Gui.Tools.NTMenuItem()
			{
				Header				= "Reset (Ctrl + Double Click)",
				Foreground			= Brushes.Silver,
				Style				= System.Windows.Application.Current.TryFindResource("InstrumentMenuItem") as Style
			};
			
			topMenuItemSubItem2.Click += TopMenuItem1SubItem2_Click;
			topMenuItem.Items.Add(topMenuItemSubItem2);
			
			topMenuItemSubItem3 = new Gui.Tools.NTMenuItem()
			{
				Header				= "Reset on new Bar",
				Foreground			= (resetOnNewBar) ? Brushes.Silver : Brushes.DimGray,
				Style				= System.Windows.Application.Current.TryFindResource("InstrumentMenuItem") as Style
			};
			
			topMenuItemSubItem3.Click += TopMenuItem1SubItem3_Click;
			topMenuItem.Items.Add(topMenuItemSubItem3);
			
			chartWindow.MainMenu.Add(theMenu);
			
			foreach(System.Windows.Controls.TabItem tab in chartWindow.MainTabControl.Items)
			{
				if((tab.Content as ChartTab).ChartControl == ChartControl && tab == chartWindow.MainTabControl.SelectedItem)
				{
					topMenuItem.Visibility = Visibility.Visible;
				}
			}
			
			chartWindow.MainTabControl.SelectionChanged += MySelectionChangedHandler;
		}
		
		// MySelectionChangedHandler
		//
		private void MySelectionChangedHandler(object sender, System.Windows.Controls.SelectionChangedEventArgs e)
		{
			if(e.AddedItems.Count <= 0)
			{
				return;
			}
			
			tabItem = e.AddedItems[0] as System.Windows.Controls.TabItem;
			
			if(tabItem == null)
			{
				return;
			}
			
			chartTab = tabItem.Content as NinjaTrader.Gui.Chart.ChartTab;
			
			if(chartTab != null)
			{
				if(theMenu != null)
				{
					theMenu.Visibility = (chartTab.ChartControl == ChartControl) ? Visibility.Visible : Visibility.Collapsed;
				}
			}
		}
		
		// RemoveWPFControls
		//
		protected void RemoveWPFControls()
		{
			if(topMenuItemSubItem1 != null)
			{
				topMenuItemSubItem1.Click -= TopMenuItem1SubItem1_Click;
			}
			
			if(topMenuItemSubItem2 != null)
			{
				topMenuItemSubItem2.Click -= TopMenuItem1SubItem2_Click;
			}
			
			if(topMenuItemSubItem3 != null)
			{
				topMenuItemSubItem3.Click -= TopMenuItem1SubItem3_Click;
			}
			
			if(theMenu != null)
			{
				chartWindow.MainMenu.Remove(theMenu);
			}
			
			chartWindow.MainTabControl.SelectionChanged -= MySelectionChangedHandler;
		}
		
		// TopMenuItem1SubItem1_Click
		//
		protected void TopMenuItem1SubItem1_Click(object sender, System.Windows.RoutedEventArgs e)
		{
			showSnapShot = !showSnapShot;
			
			if(showSnapShot)
			{
				ChartControl.Properties.BarMarginRight += (snapShotWidth + 10);	
			}
			else
			{
				ChartControl.Properties.BarMarginRight -= (snapShotWidth + 10);
			}
			
			topMenuItemSubItem1.Foreground = (showSnapShot) ? Brushes.Silver : Brushes.DimGray;
			
		  	ForceRefresh();
		}
		
		// TopMenuItem1SubItem2_Click
		//
		protected void TopMenuItem1SubItem2_Click(object sender, System.Windows.RoutedEventArgs e)
		{
			if(ss != null)
			{
				ss.vol = 0.0;
				ss.poc = 0.0;
				ss.dta = 0.0;
				
				ss.lst.Clear();
			}
			
		  	ForceRefresh();
		}
		
		// TopMenuItem1SubItem3_Click
		//
		protected void TopMenuItem1SubItem3_Click(object sender, System.Windows.RoutedEventArgs e)
		{
			resetOnNewBar = !resetOnNewBar;
			
		  	ForceRefresh();
		}
		
		#endregion
		
		#region OnMarketData
		
		// OnMarketData
		//
		protected override void OnMarketData(MarketDataEventArgs e)
		{
			if(!showSnapShot)	 		{ return; }
			if(State != State.Realtime) { return; }
			
			if(ss == null)
			{
				ss = new snapShot();
			}
			
			if(e.MarketDataType == MarketDataType.Last)
			{
				ask = Instrument.MasterInstrument.RoundToTickSize(e.Ask);
				bid = Instrument.MasterInstrument.RoundToTickSize(e.Bid);
				cls = Instrument.MasterInstrument.RoundToTickSize(e.Price);
				vol = e.Volume;
				
				if(!ss.lst.ContainsKey(cls))
				{
					ss.lst.TryAdd(cls, new RowData());
				}
				
				if(cls >= ask)
				{
					ss.vol += vol;
					ss.lst[cls].tv += vol;
					ss.lst[cls].av += vol;
					
					ss.calc(Instrument.MasterInstrument, TickSize);
				}
				
				if(cls <= bid)
				{
					ss.vol += vol;
					ss.lst[cls].tv += vol;
					ss.lst[cls].bv += vol;
					
					ss.calc(Instrument.MasterInstrument, TickSize);
				}
			}
		}
		
		#endregion
		
		#region OnMarketDepth
		
		// OnMarketDepth
		//
		protected override void OnMarketDepth(MarketDepthEventArgs e)
		{
			if(State != State.Realtime) { return; }
			if(ss == null) { return; }
			if(ss.lst.Count == 0) { return; }
			
			if(ss.lst.ContainsKey(e.Price))
			{
				if(e.MarketDataType == MarketDataType.Ask)
				{
					ss.lst[e.Price].ad = e.Volume;
				}
				if(e.MarketDataType == MarketDataType.Bid)
				{
					ss.lst[e.Price].bd = e.Volume;
				}
			}
		}
		
		#endregion
		
		// OnBarUpdate
		//
		protected override void OnBarUpdate()
		{
			if(State != State.Realtime) { return; }
			
			if(resetOnNewBar)
			{
				if(IsFirstTickOfBar)
				{
					if(ss != null)
					{
						ss.vol = 0.0;
						ss.poc = 0.0;
						ss.dta = 0.0;
						
						ss.lst.Clear();
					}
				}
			}
		}
		
		// isActiveTab
		//
		private bool isActiveTab()
		{
			bool isActive = false;
			
			NinjaTrader.Gui.Chart.Chart	cWindow = System.Windows.Window.GetWindow(ChartControl.Parent) as Chart;
			
			foreach(System.Windows.Controls.TabItem tab in cWindow.MainTabControl.Items)
			{
				if((tab.Content as ChartTab).ChartControl == ChartControl && tab == cWindow.MainTabControl.SelectedItem)
				{
					isActive = true;
					break;
				}
			}
			
			return isActive;
		}
		
		// OnRender
		//
		protected override void OnRender(ChartControl chartControl, ChartScale chartScale)
		{
			if(Bars == null || Bars.Instrument == null || IsInHitTest || !isActiveTab()) { return; }
			
			base.OnRender(chartControl, chartScale);
			
			setFontSize(chartControl, chartScale);
			drawSnapShot(chartControl, chartScale);
		}
		
		// setFontSize
		//
		private void setFontSize(ChartControl chartControl, ChartScale chartScale)
		{
			if(ss == null) { return; }
			if(ss.lst.IsEmpty) { return; }
			
			float y1 = 0f;
			float y2 = 0f;
			float ht = 0f;
			float mh = float.MaxValue;
			
			foreach(KeyValuePair<double, RowData> rd in ss.lst)
			{
				y1 = ((chartScale.GetYByValue(rd.Key) + chartScale.GetYByValue(rd.Key + TickSize)) / 2);
				y2 = ((chartScale.GetYByValue(rd.Key) + chartScale.GetYByValue(rd.Key - TickSize)) / 2);
				
				ht = Math.Abs(y2 - y1);
				mh = (ht < mh) ? ht : mh;
				
				if(mh < 5)
				{
					break;
				}
			}
			
			ht = ht - 7f;
			
			fSize = (int)Math.Round(ht / gtf.CapsHeight, 0);
			fSize = Math.Max(1, fSize);
			
			sf.Size = fSize;
		}
		
		#region drawSnapShot
		
		// drawSnapShot
		//
		private void drawSnapShot(ChartControl chartControl, ChartScale chartScale)
		{
			if(!showSnapShot) 	  { return; }
			if(ss == null) 		  { return; }
			if(ss.lst.Count == 0) { return; }
			
			SharpDX.Direct2D1.AntialiasMode oldAntialiasMode = RenderTarget.AntialiasMode;
			RenderTarget.AntialiasMode = SharpDX.Direct2D1.AntialiasMode.Aliased;
			
			SharpDX.Direct2D1.Brush bckBrush = bckColor.ToDxBrush(RenderTarget);
			SharpDX.Direct2D1.Brush txtBrush = txtColor.ToDxBrush(RenderTarget);
			SharpDX.Direct2D1.Brush askBrush = askColor.ToDxBrush(RenderTarget);
			SharpDX.Direct2D1.Brush bidBrush = bidColor.ToDxBrush(RenderTarget);
			
			SharpDX.RectangleF rect = new SharpDX.RectangleF();
			SharpDX.Vector2    vec1 = new SharpDX.Vector2();
			SharpDX.Vector2    vec2 = new SharpDX.Vector2();
			
			TextLayout tl;
			TextFormat tf;
			SharpDX.DirectWrite.FontWeight wg;
			
			GlyphTypeface gtf = new GlyphTypeface();
			System.Windows.Media.Typeface tFace = new System.Windows.Media.Typeface(new System.Windows.Media.FontFamily(sf.Family.ToString()), FontStyles.Normal, FontWeights.Normal, FontStretches.Normal);
            tFace.TryGetGlyphTypeface(out gtf);
			
			float rx = chartControl.CanvasRight - rightMargin;
			float x1,x2,y1,y2 = 0;
			float tx,ty = 0;
			float wd,ht = 0f;
			float cw = getCellWidth();
				  cw = (float)Math.Max(cw, Math.Round(snapShotWidth / 8.0, 0));
			float pw = snapShotWidth - (2 * cw);
			float fw = (2 * cw);
			
			double dt = 0.0;
			
			double pocPrc = ss.poc;
			double pocVol = ss.lst[pocPrc].tv;
			
			// ---
			
			foreach(KeyValuePair<double, RowData> rd in ss.lst)
			{
				y1 = ((chartScale.GetYByValue(rd.Key) + chartScale.GetYByValue(rd.Key + TickSize)) / 2);
				y2 = ((chartScale.GetYByValue(rd.Key) + chartScale.GetYByValue(rd.Key - TickSize)) / 2);
				
				ht = Math.Abs(y2 - y1);
				
				// footprint ask -----------------------------------------------------------------------
				
				// rect
				
				rect.X      = (float)(rx - cw);
				rect.Y      = (float)y1;
				rect.Width  = (float)cw;
				rect.Height = (float)ht;
				
				bckBrush.Opacity = 1.0f;
				
				RenderTarget.DrawRectangle(rect, bckBrush);
				
				// imbalance
				
				if(rd.Value.av >= minImbVolume && rd.Value.ai >= minImbRatio)
				{
					rect.Width  -= 1f;
					rect.Height -= 1f;
					
					askBrush.Opacity = 0.3f;
					
					RenderTarget.FillRectangle(rect, askBrush);
				}
				else
				{
					rect.Width  -= 1f;
					rect.Height -= 1f;
					
					bckBrush.Opacity = 0.2f;
					
					RenderTarget.FillRectangle(rect, bckBrush);
				}
				
				rect.Width  += 1f;
				rect.Height += 1f;
				
				// text
				
				if(sf.Size >= 5)
				{
					wg = (rd.Value.ab == 1) ? SharpDX.DirectWrite.FontWeight.UltraBlack : SharpDX.DirectWrite.FontWeight.Normal;
					
					tf = new TextFormat(new SharpDX.DirectWrite.Factory(), sf.Family.ToString(), wg, SharpDX.DirectWrite.FontStyle.Normal, (float)sf.Size);
					
					tf.TextAlignment = SharpDX.DirectWrite.TextAlignment.Leading;
					
					tl = new TextLayout(Core.Globals.DirectWriteFactory, rd.Value.av.ToString(), tf, ChartPanel.W, ChartPanel.H);
					
					tx = rect.X;
					ty = (float)(chartScale.GetYByValue(rd.Key) - (sf.Size * gtf.Baseline) + ((sf.Size * gtf.CapsHeight) / 2));
					
					vec1.X = tx + 3f;
					vec1.Y = ty;
					
					txtBrush.Opacity = (rd.Value.ab == 1) ? 1.0f : 0.8f;
					
					RenderTarget.DrawTextLayout(vec1, tl, txtBrush);
					
					tl.Dispose();
					tf.Dispose();
				}
				
				// footprint bid -----------------------------------------------------------------------
				
				// rect
				
				rect.X      = (float)(rx - cw * 2);
				rect.Y      = (float)y1;
				rect.Width  = (float)cw;
				rect.Height = (float)ht;
				
				bckBrush.Opacity = 1.0f;
				
				RenderTarget.DrawRectangle(rect, bckBrush);
				
				// imbalance
				
				if(rd.Value.bv >= minImbVolume && rd.Value.bi >= minImbRatio)
				{
					rect.Width  -= 1f;
					rect.Height -= 1f;
					
					bidBrush.Opacity = 0.3f;
					
					RenderTarget.FillRectangle(rect, bidBrush);
				}
				else
				{
					rect.Width  -= 1f;
					rect.Height -= 1f;
					
					bckBrush.Opacity = 0.2f;
					
					RenderTarget.FillRectangle(rect, bckBrush);
				}
				
				rect.Width  += 1f;
				rect.Height += 1f;
				
				// text
				
				if(sf.Size >= 5)
				{
					wg = (rd.Value.bb == 1) ? SharpDX.DirectWrite.FontWeight.UltraBlack : SharpDX.DirectWrite.FontWeight.Normal;
					
					tf = new TextFormat(new SharpDX.DirectWrite.Factory(), sf.Family.ToString(), wg, SharpDX.DirectWrite.FontStyle.Normal, (float)sf.Size);
					
					tf.TextAlignment = SharpDX.DirectWrite.TextAlignment.Trailing;
					
					tl = new TextLayout(Core.Globals.DirectWriteFactory, rd.Value.bv.ToString(), tf, rect.Width, rect.Height);
					
					tx = rect.X;
					ty = (float)(chartScale.GetYByValue(rd.Key) - (sf.Size * gtf.Baseline) + ((sf.Size * gtf.CapsHeight) / 2));
					
					vec1.X = tx - 3f;
					vec1.Y = ty;
					
					txtBrush.Opacity = (rd.Value.bb == 1) ? 1.0f : 0.8f;
					
					RenderTarget.DrawTextLayout(vec1, tl, txtBrush);
					
					tl.Dispose();
					tf.Dispose();
				}
				
				// profile -----------------------------------------------------------------------------
				
				wd = (float)((pw / pocVol) * rd.Value.tv);
				
				rect.X      = (float)(rx - fw - wd);
				rect.Y      = (float)y1;
				rect.Width  = (float)wd;
				rect.Height = (float)ht;
				
				bckBrush.Opacity = 1.0f;
				
				if(wd >= 1f)
				{
					RenderTarget.DrawRectangle(rect, bckBrush);
				}
				
				rect.Width  -= 1f;
				rect.Height -= 1f;
				
				bckBrush.Opacity = 0.2f;
				
				if(wd >= 1f)
				{
					RenderTarget.FillRectangle(rect, bckBrush);
				}
				
				// profile - delta ---------------------------------------------------------------------
				
				dt = rd.Value.av - rd.Value.bv;
				
				wd = (float)((pw / pocVol) * Math.Abs(dt));
				
				if(wd >= 1f)
				{
					rect.X = (float)(rx - fw - wd);
					rect.Width = (float)wd;
					
					if(dt > 0.0)
					{
						askBrush.Opacity = 0.3f;
						
						RenderTarget.FillRectangle(rect, askBrush);
					}
					else
					{
						bidBrush.Opacity = 0.3f;
						
						RenderTarget.FillRectangle(rect, bidBrush);
					}
				}
			}
			
			// delta -------------------------------------------------------------------------------
			
			y1 = ((chartScale.GetYByValue(ss.lst.Keys.Max()) + chartScale.GetYByValue(ss.lst.Keys.Max() + TickSize)) / 2) - 3f;
			y2 = ((chartScale.GetYByValue(ss.lst.Keys.Min()) + chartScale.GetYByValue(ss.lst.Keys.Min() - TickSize)) / 2) + 2f;
			
			if(ss.dta > 0.0)
			{
				vec1.X = rx;
				vec1.Y = y1;
				
				vec2.X = rx - cw * 2 - 1;
				vec2.Y = y1;
				
				askBrush.Opacity = 1.0f;
				
				RenderTarget.DrawLine(vec1, vec2, askBrush, 2);
				
				// ---
				
				vec1.X = rx;
				vec1.Y = y2;
				
				vec2.X = rx - cw * 2 - 1;
				vec2.Y = y2;
				
				askBrush.Opacity = 1.0f;
				
				RenderTarget.DrawLine(vec1, vec2, askBrush, 2);
			}
			
			if(ss.dta < 0.0)
			{
				vec1.X = rx;
				vec1.Y = y1;
				
				vec2.X = rx - cw * 2 - 1;
				vec2.Y = y1;
				
				bidBrush.Opacity = 1.0f;
				
				RenderTarget.DrawLine(vec1, vec2, bidBrush, 2);
				
				// ---
				
				vec1.X = rx;
				vec1.Y = y2;
				
				vec2.X = rx - cw * 2 - 1;
				vec2.Y = y2;
				
				bidBrush.Opacity = 1.0f;
				
				RenderTarget.DrawLine(vec1, vec2, bidBrush, 2);
			}
			
			// imbalances - inner rectangle
			
			foreach(KeyValuePair<double, RowData> rd in ss.lst)
			{
				y1 = ((chartScale.GetYByValue(rd.Key) + chartScale.GetYByValue(rd.Key + TickSize)) / 2);
				y2 = ((chartScale.GetYByValue(rd.Key) + chartScale.GetYByValue(rd.Key - TickSize)) / 2);
				
				ht = Math.Abs(y2 - y1);
				
				// ask imbalance
				
				if(rd.Value.av >= minImbVolume && rd.Value.ai >= minImbRatio)
				{	
					rect.X      = (float)(rx - cw) + 1f;
					rect.Y      = (float)y1 + 1f;
					rect.Width  = (float)cw - 2f;
					rect.Height = (float)ht - 2f;
					
					askBrush.Opacity = 0.8f;
					
					RenderTarget.DrawRectangle(rect, askBrush);
				}
				
				// bid imbalance
				
				if(rd.Value.bv >= minImbVolume && rd.Value.bi >= minImbRatio)
				{
					rect.X      = (float)(rx - cw * 2) + 1f;
					rect.Y      = (float)y1 + 1f;
					rect.Width  = (float)cw - 2f;
					rect.Height = (float)ht - 2f;
					
					bidBrush.Opacity = 0.8f;
					
					RenderTarget.DrawRectangle(rect, bidBrush);
				}
			}
			
			// ---
			
			gtf	  = null;
			tFace = null;
			
			bckBrush.Dispose();
			txtBrush.Dispose();
			askBrush.Dispose();
			bidBrush.Dispose();
			
			RenderTarget.AntialiasMode = oldAntialiasMode;
			
			// ---
			
			if(chartScale.Properties.YAxisRangeType == YAxisRangeType.Fixed)
			{
				double prc = Bars.GetClose(ChartBars.ToIndex);
				double rng = chartScale.MaxMinusMin;
				double off = TickSize * 3;
				double dif = TickSize * 4;
				
				if(prc >= chartScale.MaxValue - off)
				{
					chartScale.Properties.FixedScaleMax = prc + dif;
					chartScale.Properties.FixedScaleMin = (prc + dif) - rng;
				}
				
				if(prc <= chartScale.MinValue + off)
				{
					chartScale.Properties.FixedScaleMin = (prc - dif);
					chartScale.Properties.FixedScaleMax = (prc - dif) + rng;
				}
			}
		}
		
		#endregion
		
		#region Utilities
		
		// getTextWidth
		//
		private float getTextWidth(string text)
		{
			float textWidth = 0f;
			
			if(text.Length > 0)
			{
				TextFormat tf = new TextFormat(new SharpDX.DirectWrite.Factory(), sf.Family.ToString(), SharpDX.DirectWrite.FontWeight.UltraBlack, SharpDX.DirectWrite.FontStyle.Normal, (float)sf.Size);
				TextLayout tl = new TextLayout(Core.Globals.DirectWriteFactory, text, tf, ChartPanel.W, ChartPanel.H);
				
				textWidth = tl.Metrics.Width;
				
				tf.Dispose();
				tl.Dispose();
			}
			
			return textWidth;
		}
		
		// getCellWidth
		//
		private float getCellWidth()
		{
			float curWidth = 0f;
			float maxWidth = 0f;
			
			if(ss != null && ss.lst.Count > 0)
			{
				foreach(KeyValuePair<double, RowData> rd in ss.lst)
				{
					curWidth = getTextWidth(rd.Value.av.ToString());
					maxWidth = (curWidth > maxWidth) ? curWidth : maxWidth;
					
					curWidth = getTextWidth(rd.Value.bv.ToString());
					maxWidth = (curWidth > maxWidth) ? curWidth : maxWidth;
				}
			}
			
			return maxWidth + 6f;
		}
		
		#endregion
		
		#region Parameters
		
		[NinjaScriptProperty]
		[Display(Name = "Show SnapShot", GroupName = "1.0 General Parameters", Order = 0)]
		public bool showSnapShot
		{ get; set; }
		
		// ---
		
		[Range(0, int.MaxValue), NinjaScriptProperty]
		[Display(Name = "Right Margin", GroupName = "1.0 General Parameters", Order = 1)]
		public int rightMargin
		{ get; set; }
		
		// ---
		
		[Range(30, int.MaxValue), NinjaScriptProperty]
		[Display(Name = "Width", GroupName = "1.0 General Parameters", Order = 2)]
		public int snapShotWidth
		{ get; set; }
		
		// ---
		
		[NinjaScriptProperty]
		[Display(Name = "Reset on new Bar", GroupName = "1.0 General Parameters", Order = 3)]
		public bool resetOnNewBar
		{ get; set; }
		
		// ---
		
		[Range(0.0, double.MaxValue), NinjaScriptProperty]
		[Display(Name = "Min. Volume", GroupName = "2.0 Imbalance Parameters", Order = 0)]
		public double minImbVolume
		{ get; set; }
		
		[Range(0.0, double.MaxValue), NinjaScriptProperty]
		[Display(Name = "Min. Ratio", GroupName = "2.0 Imbalance Parameters", Order = 1)]
		public double minImbRatio
		{ get; set; }
		
		// ---
		
		[NinjaScriptProperty]
		[XmlIgnore]
		[Display(Name = "Border Color", GroupName = "3. Color Parameters", Order = 0)]
		public Brush bckColor
		{ get; set; }
		
		[Browsable(false)]
		public string bckColorSerializable
		{
			get { return Serialize.BrushToString(bckColor); }
			set { bckColor = Serialize.StringToBrush(value); }
		}
		
		// ---
		
		[NinjaScriptProperty]
		[XmlIgnore]
		[Display(Name = "Text Color", GroupName = "3. Color Parameters", Order = 1)]
		public Brush txtColor
		{ get; set; }
		
		[Browsable(false)]
		public string txtColorSerializable
		{
			get { return Serialize.BrushToString(txtColor); }
			set { txtColor = Serialize.StringToBrush(value); }
		}
		
		// ---
		
		[NinjaScriptProperty]
		[XmlIgnore]
		[Display(Name = "Ask Color", GroupName = "3. Color Parameters", Order = 2)]
		public Brush askColor
		{ get; set; }
		
		[Browsable(false)]
		public string askColorSerializable
		{
			get { return Serialize.BrushToString(askColor); }
			set { askColor = Serialize.StringToBrush(value); }
		}
		
		// ---
		
		[NinjaScriptProperty]
		[XmlIgnore]
		[Display(Name = "Bid Color", GroupName = "3. Color Parameters", Order = 3)]
		public Brush bidColor
		{ get; set; }
		
		[Browsable(false)]
		public string bidColorSerializable
		{
			get { return Serialize.BrushToString(bidColor); }
			set { bidColor = Serialize.StringToBrush(value); }
		}
		
		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.

namespace NinjaTrader.NinjaScript.Indicators
{
	public partial class Indicator : NinjaTrader.Gui.NinjaScript.IndicatorRenderBase
	{
		private Infinity.SnapShot[] cacheSnapShot;
		public Infinity.SnapShot SnapShot(bool showSnapShot, int rightMargin, int snapShotWidth, bool resetOnNewBar, double minImbVolume, double minImbRatio, Brush bckColor, Brush txtColor, Brush askColor, Brush bidColor)
		{
			return SnapShot(Input, showSnapShot, rightMargin, snapShotWidth, resetOnNewBar, minImbVolume, minImbRatio, bckColor, txtColor, askColor, bidColor);
		}

		public Infinity.SnapShot SnapShot(ISeries<double> input, bool showSnapShot, int rightMargin, int snapShotWidth, bool resetOnNewBar, double minImbVolume, double minImbRatio, Brush bckColor, Brush txtColor, Brush askColor, Brush bidColor)
		{
			if (cacheSnapShot != null)
				for (int idx = 0; idx < cacheSnapShot.Length; idx++)
					if (cacheSnapShot[idx] != null && cacheSnapShot[idx].showSnapShot == showSnapShot && cacheSnapShot[idx].rightMargin == rightMargin && cacheSnapShot[idx].snapShotWidth == snapShotWidth && cacheSnapShot[idx].resetOnNewBar == resetOnNewBar && cacheSnapShot[idx].minImbVolume == minImbVolume && cacheSnapShot[idx].minImbRatio == minImbRatio && cacheSnapShot[idx].bckColor == bckColor && cacheSnapShot[idx].txtColor == txtColor && cacheSnapShot[idx].askColor == askColor && cacheSnapShot[idx].bidColor == bidColor && cacheSnapShot[idx].EqualsInput(input))
						return cacheSnapShot[idx];
			return CacheIndicator<Infinity.SnapShot>(new Infinity.SnapShot(){ showSnapShot = showSnapShot, rightMargin = rightMargin, snapShotWidth = snapShotWidth, resetOnNewBar = resetOnNewBar, minImbVolume = minImbVolume, minImbRatio = minImbRatio, bckColor = bckColor, txtColor = txtColor, askColor = askColor, bidColor = bidColor }, input, ref cacheSnapShot);
		}
	}
}

namespace NinjaTrader.NinjaScript.MarketAnalyzerColumns
{
	public partial class MarketAnalyzerColumn : MarketAnalyzerColumnBase
	{
		public Indicators.Infinity.SnapShot SnapShot(bool showSnapShot, int rightMargin, int snapShotWidth, bool resetOnNewBar, double minImbVolume, double minImbRatio, Brush bckColor, Brush txtColor, Brush askColor, Brush bidColor)
		{
			return indicator.SnapShot(Input, showSnapShot, rightMargin, snapShotWidth, resetOnNewBar, minImbVolume, minImbRatio, bckColor, txtColor, askColor, bidColor);
		}

		public Indicators.Infinity.SnapShot SnapShot(ISeries<double> input , bool showSnapShot, int rightMargin, int snapShotWidth, bool resetOnNewBar, double minImbVolume, double minImbRatio, Brush bckColor, Brush txtColor, Brush askColor, Brush bidColor)
		{
			return indicator.SnapShot(input, showSnapShot, rightMargin, snapShotWidth, resetOnNewBar, minImbVolume, minImbRatio, bckColor, txtColor, askColor, bidColor);
		}
	}
}

namespace NinjaTrader.NinjaScript.Strategies
{
	public partial class Strategy : NinjaTrader.Gui.NinjaScript.StrategyRenderBase
	{
		public Indicators.Infinity.SnapShot SnapShot(bool showSnapShot, int rightMargin, int snapShotWidth, bool resetOnNewBar, double minImbVolume, double minImbRatio, Brush bckColor, Brush txtColor, Brush askColor, Brush bidColor)
		{
			return indicator.SnapShot(Input, showSnapShot, rightMargin, snapShotWidth, resetOnNewBar, minImbVolume, minImbRatio, bckColor, txtColor, askColor, bidColor);
		}

		public Indicators.Infinity.SnapShot SnapShot(ISeries<double> input , bool showSnapShot, int rightMargin, int snapShotWidth, bool resetOnNewBar, double minImbVolume, double minImbRatio, Brush bckColor, Brush txtColor, Brush askColor, Brush bidColor)
		{
			return indicator.SnapShot(input, showSnapShot, rightMargin, snapShotWidth, resetOnNewBar, minImbVolume, minImbRatio, bckColor, txtColor, askColor, bidColor);
		}
	}
}

#endregion
