#region Using declarations
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Input;
using System.Windows.Media;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Gui;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Gui.SuperDom;
using NinjaTrader.Data;
using NinjaTrader.NinjaScript;
using NinjaTrader.Core.FloatingPoint;
using NinjaTrader.NinjaScript.DrawingTools;

#endregion

//This namespace holds Indicators in this folder and is required. Do not change it. 
namespace NinjaTrader.NinjaScript.Indicators.Sim22
{
	/// <summary>
	/// Based on Gomi's GOM delta.
	/// Zondor 'unGom' Bid-Ask conversion to NT8 with Tick Replay.
	/// Sim22 addition of various other GOM formulae, code tidying and extra features. NT8b9 Feb 2016.
	/// </summary>
	public class Sim22_DeltaV2 : Indicator
	{   
		Sim22_DeltaV2_Type				type	= Sim22_DeltaV2_Type.BidAsk;
		Sim22_DeltaV2_FilterModeType	filter	= Sim22_DeltaV2_FilterModeType.None;
		
		private Series<double>	O;
		private Series<double>	C;
		private Series<double>	H;
		private Series<double>	L;
		
		private double 			tv, deltaM 				= 0.0;
		private double 			CD;
		private double 			B,A,Lm, L0, L1 			= 0.0;

		private	int				lastDirection 			= 0;
		private	int				lastPrice				= 0;
		private	bool			startlookingforreversal = false;
		private string			filterString			= null;
		
		protected override void OnStateChange()
		{   
			if (State == State.SetDefaults)
			{
				Description							= @"Delta. Gomi, Zondor, Sim22";
				Name								= "Sim22_DeltaV2";
				Calculate							= Calculate.OnEachTick;
				MaximumBarsLookBack 				= MaximumBarsLookBack.Infinite;
				IsOverlay							= false;
				DisplayInDataBox					= true;
				DrawOnPricePanel					= true;
				DrawHorizontalGridLines				= false;
				DrawVerticalGridLines				= false;
				PaintPriceMarkers					= true;
				ScaleJustification					= NinjaTrader.Gui.Chart.ScaleJustification.Right;
				IsSuspendedWhileInactive			= false;
				
				AddPlot(Brushes.RoyalBlue, "Delta");
				AddLine(new Stroke(Brushes.Gray, DashStyleHelper.Dot ,1), 0, "ZeroLine");
				
				BarUpBrush							= Brushes.LimeGreen;
				BarDnBrush							= Brushes.Red;
				OutlineUpBrush                      = Brushes.Green;
				OutlineDnBrush                      = Brushes.DarkRed;
				
				cumDelta							= true;
				resetDelta							= false;
				filterSize							= 1;
				invertDelta							= false;
			}
			else if (State == State.Configure)
			{    
				BarUpBrush.Freeze();							
				BarDnBrush.Freeze();
				OutlineUpBrush.Freeze();
				OutlineDnBrush.Freeze();
				
				
				O = new Series<double>(this, MaximumBarsLookBack.Infinite);
				C = new Series<double>(this, MaximumBarsLookBack.Infinite);
				H = new Series<double>(this, MaximumBarsLookBack.Infinite);
				L = new Series<double>(this, MaximumBarsLookBack.Infinite);
				
				if (filter== Sim22_DeltaV2_FilterModeType.GreaterOrEqualTo)
					filterString	= "Filter >= " + filterSize;
				else if (filter== Sim22_DeltaV2_FilterModeType.LessThanOrEqualTo)
					filterString	= "Filter <= " + filterSize;	
			}
		}
		
		public override string FormatPriceMarker(double price)
		{
		     return price.ToString("N0");
		} 
		
		public override string DisplayName
		{
		    get { return type.ToString() + (invertDelta ? " (Inverted)" : "") + (resetDelta ? "\rEOD reset " : "")  + (filterString != null ? "\r" + filterString : "")  ; }
		}
		
		#region Calculate Delta
		
        protected double 	CurrentDelta(double _Ask, double _Bid, double _Last, double _Volume)
		{   
			L0 = _Last; A = _Ask; B = _Bid; tv = _Volume; deltaM = 0;
			
			int direction 	= lastDirection;
			
			if (type	== Sim22_DeltaV2_Type.BidAsk) 
			{
                if(A > B)
				{
					if (L0 >= A)       
					{ 
						deltaM =   tv;	
					} 
					
				   	if (L0 <= B)       
					{ 
						deltaM =  -tv;	
					}
					
				   	if(L0 < A && L0 > B)
					{   
						if(A - L0 < L0 - B)    
						{ 
							deltaM =   tv;	
						}
						if(A - L0 > L0 - B)    
						{ 
							deltaM =  -tv;	
						}
					}
				}
				else if (A < B)
				{
				    if (L0 >= B)        
					{ 
						deltaM =  tv;	
						
					}	
				    if (L0 <= A)        
					{ 
						deltaM =  -tv;	
						
					}
					if(L0 < B && L0 > A)
					{   
						if(B - L0 < L0 - A)      
						{ 
							deltaM =   tv;	
							
						}
						if(B - L0 > L0 - A)      
						{ 
							deltaM =  -tv;	
							
						}
					}
				}
					
				else if(A == B)
				{    
					if (L0 > A)          
					{ 
						deltaM =   tv;		
					}
					if (L0 < B)          
					{ 
						deltaM =  -tv;	
					}	
				}
				else L0 = L0;
				
				
			}
			else if (type	== Sim22_DeltaV2_Type.UpDownTick)
			{
				if (L1 == 0d)
					deltaM	= 0.0;
				
				if (L0 > L1)
				{
					deltaM 	= tv;
				}	
				else if (L0 < L1)
				{
					deltaM 	= -tv;	
				}	
				else
				{
					deltaM = 0.0;
				}	
			}
			else if (type == Sim22_DeltaV2_Type.UpDownTickWithContinuation || type	== Sim22_DeltaV2_Type.UpDownOneTickWithContinuation)
			{
				if (L0 > L1)
				{
					direction	= 1;
				}	
				else if (L0 < L1)
				{
					direction	= -1;
				}
				
				if (type	== Sim22_DeltaV2_Type.UpDownOneTickWithContinuation)
					deltaM	= direction;
				else
					deltaM	= direction * tv;	
			}
			else if (type	== Sim22_DeltaV2_Type.Hybrid)
			{

				if (L0 > L1)
				{
					direction = 1;
					startlookingforreversal = false;
				}
				else if (L0 < L1)
				{
					direction = -1;
					startlookingforreversal = false;
				}

				if (!startlookingforreversal)
					if (direction == 1)
						startlookingforreversal = (L0 <= B) ? true: false;
					else
						startlookingforreversal = (L0 >= A) ? true: false;

				if (L0 == L1)
				{
					if ((direction == 1) && startlookingforreversal && ((L0 == A) || (L0 < A && L0 > B)))
						direction = -1;

					else if ((direction == -1) && startlookingforreversal && ((L0 == B) || (L0 < A && L0 > B)))
						direction = 1;
				}
				
				deltaM = direction * tv;
			}
			L1	= L0;
			lastDirection	= direction;
			
			if ((filter == Sim22_DeltaV2_FilterModeType.GreaterOrEqualTo) && (tv < filterSize))
				deltaM = 0.0;

			if ((filter == Sim22_DeltaV2_FilterModeType.LessThanOrEqualTo) && (tv > filterSize))
				deltaM = 0.0;
			
			if (invertDelta)
				return -deltaM;
			else
				return deltaM;
		}
		
		#endregion
		
		protected override void OnMarketData(MarketDataEventArgs e)
		{   
			if (CurrentBar == -1) 
				return;
            if(e.MarketDataType != MarketDataType.Last)
				return;
			if(CurrentBars[0] == -1)
				return;

			if(BarsInProgress == 0)
			{   
				if (IsFirstTickOfBar)
				{
					if (CurrentBar == 0)
						CD = 0.0;
					
					if (CurrentBar > 0)
						CD = C[1];
					
					if (!cumDelta || (Bars.IsResetOnNewTradingDay && Bars.IsFirstBarOfSession && resetDelta))
					{
						CD = 0.0;
					}
					
					O[0] = CD;		
					H[0] = CD;  
					L[0] = CD;    
					C[0] = CD;
				}
				
				deltaM = CurrentDelta(e.Ask, e.Bid, e.Price, e.Volume);
				
			   	CD += deltaM;
				
			   	Values[0][0] = CD;
				
				H[0] = Math.Max(H[0],CD);   
				L[0] = Math.Min(L[0],CD);    
				C[0] = CD; 	
			}
		}				
		protected override void OnBarUpdate()
		{ 
			//For price marker
			if (CD > O[0])
				PlotBrushes[0][0] = BarUpBrush;
			else if (CD < O[0])
				PlotBrushes[0][0] = BarDnBrush;
			else if (CurrentBar != 0)
				PlotBrushes[0][0] = PlotBrushes[0][1];
			else
				PlotBrushes[0][0] = Brushes.Black;	
		}
		
		public override void OnCalculateMinMax()
		{  
			if (Bars == null) 
				return;
			
			int lastBar = ChartBars.ToIndex;
			int firstBar = ChartBars.FromIndex;

			double min = Double.MaxValue;
			double max = Double.MinValue;

			for (int index = firstBar; index <= lastBar; index++)
			{
				{
					min = Math.Min(min, L.GetValueAt(index));
					max = Math.Max(max, H.GetValueAt(index));
				}
			}

			if ((max - min) < 1)
			{
				min -= 1;
				max += 1;
			}
			
		   MinValue = min;
		   MaxValue = max;
		} 
				
		protected override void OnRender(ChartControl chartControl, ChartScale chartScale)
		{   
			if( Bars == null || ChartControl == null || Bars.Instrument == null || !IsVisible) 
			{
				return;
			}
			
			NinjaTrader.Data.Bars	bars	= ChartBars.Bars;
				
			try
			{
				int cbti = ChartBars.ToIndex;
				int cbfi = ChartBars.FromIndex;			
				float yLo, yHi, yOpen, yClose, xCenter = 0f;
				int middle, height = 0;
				float top, bottom;
				
				SharpDX.Direct2D1.Brush brushBarDX 		= Brushes.Gray.ToDxBrush(RenderTarget);
				SharpDX.Direct2D1.Brush brushOutlineDX 	= Brushes.Black.ToDxBrush(RenderTarget);
				
				SharpDX.Direct2D1.Brush brushZeroDX 	= Lines[0].BrushDX;
				SharpDX.Direct2D1.StrokeStyle	lzSS	= Lines[0].StrokeStyle;
				float	zeroLineY						= chartScale.GetYByValue(0);
				RenderTarget.DrawLine(new SharpDX.Vector2(ChartPanel.X,zeroLineY), new SharpDX.Vector2(ChartPanel.W, zeroLineY   ), brushZeroDX, 1f, lzSS);
				
				float width = (float)Math.Max(2,ChartBars.Properties.ChartStyle.BarWidth*2);
				
				for (int idx = cbfi ; idx <= cbti; idx++)
				{
					double	OpValue			= O.GetValueAt(idx);
					double	HiValue			= H.GetValueAt(idx);
					double	LoValue			= L.GetValueAt(idx);
					double	ClValue			= C.GetValueAt(idx);
					
					double	ClValue1		= idx > 0 ? C.GetValueAt(idx - 1): ClValue;
					double	OpValue1		= idx > 0 ? O.GetValueAt(idx - 1): OpValue;
					
					yOpen 					= chartScale.GetYByValue(OpValue);
					yHi   					= chartScale.GetYByValue(HiValue);
					yLo   					= chartScale.GetYByValue(LoValue);
					yClose					= chartScale.GetYByValue(ClValue);
					
					height 					= (int)Math.Abs(yOpen - yClose);
					top 					= (OpValue > ClValue ? yOpen : yClose);
					bottom					= (OpValue < ClValue ? yOpen : yClose);
					xCenter					= chartControl.GetXByBarIndex(ChartBars, idx);
					float xPosition 		= xCenter - (float)(width / 2);
					
	               
					SharpDX.RectangleF rect = new SharpDX.RectangleF(xPosition, top, width, height);
					
					if (ClValue > OpValue)	    
					{ 
						brushBarDX			= BarUpBrush.ToDxBrush(RenderTarget);
						brushOutlineDX		= OutlineUpBrush.ToDxBrush(RenderTarget);
					}
					else if (ClValue < OpValue)	
					{ 
						brushBarDX			= BarDnBrush.ToDxBrush(RenderTarget);
						brushOutlineDX		= OutlineDnBrush.ToDxBrush(RenderTarget);
					}
					else					
					{ 	//use last bar color
						if (ClValue1 > OpValue1)	    
						{ 
							brushBarDX			= BarUpBrush.ToDxBrush(RenderTarget);
							brushOutlineDX		= OutlineUpBrush.ToDxBrush(RenderTarget);
						} 
						else if (ClValue1 < OpValue1)	
						{ 
							brushBarDX			= BarDnBrush.ToDxBrush(RenderTarget);
							brushOutlineDX		= OutlineDnBrush.ToDxBrush(RenderTarget);
						}
					}
					// else gray/black by default
					
						
					RenderTarget.FillRectangle(rect, brushBarDX);
					RenderTarget.DrawRectangle(rect, brushOutlineDX);
					RenderTarget.DrawLine(new SharpDX.Vector2(xCenter,yHi), new SharpDX.Vector2(xCenter, top   ), brushOutlineDX, 1f);
					RenderTarget.DrawLine(new SharpDX.Vector2(xCenter,yLo), new SharpDX.Vector2(xCenter, bottom), brushOutlineDX, 1f); 
					
				}
				
				brushBarDX.Dispose();
				brushOutlineDX.Dispose();
				brushZeroDX.Dispose();
				lzSS.Dispose();
			
			} catch{}
		}
		
		
		#region Properties
		
		[Browsable(false)]
		[XmlIgnore()]
		public Series<double> Delta
		{
			get { return Values[0]; }
		}
		
		[XmlIgnore]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Up color", Order = 6, GroupName = "3. Colors")]
		public Brush BarUpBrush		
		{ get; set; }
		
		[Browsable(false)]
		public string BarUpBrushSerialize
		{
			get { return Serialize.BrushToString(BarUpBrush); }
   			set { BarUpBrush = Serialize.StringToBrush(value); }
		}
		
		[XmlIgnore]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Down color", Order = 8, GroupName = "3. Colors")]
		public Brush BarDnBrush
		{ get; set; }
		
		[Browsable(false)]
		public string BarDnBrushSerialize
		{
			get { return Serialize.BrushToString(BarDnBrush); }
   			set { BarDnBrush = Serialize.StringToBrush(value); }
		}
		
		[XmlIgnore]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Up color outline", Order = 7, GroupName = "3. Colors")]
		public Brush OutlineUpBrush
		{ get; set; }
		
		[Browsable(false)]
		public string OutlineUpBrushSerialize
		{
			get { return Serialize.BrushToString(OutlineUpBrush); }
   			set { OutlineUpBrush = Serialize.StringToBrush(value); }
		}
		
		[XmlIgnore]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Dn color outline", Order = 9, GroupName = "3. Colors")]
		public Brush OutlineDnBrush
		{ get; set; }
		
		[Browsable(false)]
		public string OutlineDnBrushSerialize
		{
			get { return Serialize.BrushToString(OutlineDnBrush); }
   			set { OutlineDnBrush = Serialize.StringToBrush(value); }
		}
		
		[NinjaScriptProperty]
		[Display(Name="Delta type", Description="", Order=0, GroupName = "1. Delta")]
		public Sim22_DeltaV2_Type Type
		{
			get { return type; }
			set { type = value; }
		}
		
		[NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Cumulative delta?", GroupName = "1. Delta", Order = 1)]
		public bool cumDelta
		{ get; set; }
		
		[NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "EOD reset?", GroupName = "1. Delta", Order = 2)]
		public bool resetDelta
		{ get; set; }
		
		[NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Invert delta?", GroupName = "1. Delta", Order = 3)]
		public bool invertDelta
		{ get; set; }
		
		[NinjaScriptProperty]
		[Display(Name="Filter type", Description="", Order=0, GroupName = "2. Filter")]
		public Sim22_DeltaV2_FilterModeType Filter
		{
			get { return filter; }
			set { filter = value; }
		}
		
		[Range(1, int.MaxValue), NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), Name = "Filter size", GroupName = "2. Filter", Order = 1)]
		public int filterSize 
		{ get; set; }
		
		#endregion

	}
	
}
#region Enum

	public enum Sim22_DeltaV2_Type
	{
		BidAsk,
		UpDownTick,
		UpDownTickWithContinuation,
		UpDownOneTickWithContinuation,
		Hybrid,
	}
	
	public enum Sim22_DeltaV2_FilterModeType
	{
		GreaterOrEqualTo,
		LessThanOrEqualTo,
		None
	}
	
#endregion

#region NinjaScript generated code. Neither change nor remove.

namespace NinjaTrader.NinjaScript.Indicators
{
	public partial class Indicator : NinjaTrader.Gui.NinjaScript.IndicatorRenderBase
	{
		private Sim22.Sim22_DeltaV2[] cacheSim22_DeltaV2;
		public Sim22.Sim22_DeltaV2 Sim22_DeltaV2(Sim22_DeltaV2_Type type, bool cumDelta, bool resetDelta, bool invertDelta, Sim22_DeltaV2_FilterModeType filter, int filterSize)
		{
			return Sim22_DeltaV2(Input, type, cumDelta, resetDelta, invertDelta, filter, filterSize);
		}

		public Sim22.Sim22_DeltaV2 Sim22_DeltaV2(ISeries<double> input, Sim22_DeltaV2_Type type, bool cumDelta, bool resetDelta, bool invertDelta, Sim22_DeltaV2_FilterModeType filter, int filterSize)
		{
			if (cacheSim22_DeltaV2 != null)
				for (int idx = 0; idx < cacheSim22_DeltaV2.Length; idx++)
					if (cacheSim22_DeltaV2[idx] != null && cacheSim22_DeltaV2[idx].Type == type && cacheSim22_DeltaV2[idx].cumDelta == cumDelta && cacheSim22_DeltaV2[idx].resetDelta == resetDelta && cacheSim22_DeltaV2[idx].invertDelta == invertDelta && cacheSim22_DeltaV2[idx].Filter == filter && cacheSim22_DeltaV2[idx].filterSize == filterSize && cacheSim22_DeltaV2[idx].EqualsInput(input))
						return cacheSim22_DeltaV2[idx];
			return CacheIndicator<Sim22.Sim22_DeltaV2>(new Sim22.Sim22_DeltaV2(){ Type = type, cumDelta = cumDelta, resetDelta = resetDelta, invertDelta = invertDelta, Filter = filter, filterSize = filterSize }, input, ref cacheSim22_DeltaV2);
		}
	}
}

namespace NinjaTrader.NinjaScript.MarketAnalyzerColumns
{
	public partial class MarketAnalyzerColumn : MarketAnalyzerColumnBase
	{
		public Indicators.Sim22.Sim22_DeltaV2 Sim22_DeltaV2(Sim22_DeltaV2_Type type, bool cumDelta, bool resetDelta, bool invertDelta, Sim22_DeltaV2_FilterModeType filter, int filterSize)
		{
			return indicator.Sim22_DeltaV2(Input, type, cumDelta, resetDelta, invertDelta, filter, filterSize);
		}

		public Indicators.Sim22.Sim22_DeltaV2 Sim22_DeltaV2(ISeries<double> input , Sim22_DeltaV2_Type type, bool cumDelta, bool resetDelta, bool invertDelta, Sim22_DeltaV2_FilterModeType filter, int filterSize)
		{
			return indicator.Sim22_DeltaV2(input, type, cumDelta, resetDelta, invertDelta, filter, filterSize);
		}
	}
}

namespace NinjaTrader.NinjaScript.Strategies
{
	public partial class Strategy : NinjaTrader.Gui.NinjaScript.StrategyRenderBase
	{
		public Indicators.Sim22.Sim22_DeltaV2 Sim22_DeltaV2(Sim22_DeltaV2_Type type, bool cumDelta, bool resetDelta, bool invertDelta, Sim22_DeltaV2_FilterModeType filter, int filterSize)
		{
			return indicator.Sim22_DeltaV2(Input, type, cumDelta, resetDelta, invertDelta, filter, filterSize);
		}

		public Indicators.Sim22.Sim22_DeltaV2 Sim22_DeltaV2(ISeries<double> input , Sim22_DeltaV2_Type type, bool cumDelta, bool resetDelta, bool invertDelta, Sim22_DeltaV2_FilterModeType filter, int filterSize)
		{
			return indicator.Sim22_DeltaV2(input, type, cumDelta, resetDelta, invertDelta, filter, filterSize);
		}
	}
}

#endregion
